/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2016 The Kuali Foundation
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.gl.batch.service.impl;

import org.kuali.kfs.coa.businessobject.OrganizationReversion;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.gl.GeneralLedgerConstants;
import org.kuali.kfs.gl.batch.service.OrganizationReversionProcess;
import org.kuali.kfs.gl.batch.service.OrganizationReversionProcessService;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSKeyConstants;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.sys.service.ReportWriterService;
import org.kuali.kfs.sys.service.impl.KfsParameterConstants;
import org.kuali.rice.core.api.config.property.ConfigurationService;
import org.springframework.transaction.annotation.Transactional;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.HashMap;
import java.util.Map;

/**
 * The base implementation of OrganizationReversionProcessService
 */
@Transactional
public class OrganizationReversionProcessServiceImpl implements OrganizationReversionProcessService {
    private static org.apache.log4j.Logger LOG = org.apache.log4j.Logger.getLogger(OrganizationReversionProcessServiceImpl.class);

    private static final String ORGANIZATION_REVERSION_PRIOR_YEAR_ACCOUNT_PROCESS_BEAN_NAME = "glOrganizationReversionPriorYearAccountProcess";
    private static final String ORGANIZATION_REVERSION_CURRENT_YEAR_ACCOUNT_PROCESS_BEAN_NAME = "glOrganizationReversionCurrentYearAccountProcess";

    private static final String DATE_FORMAT = "yyyy-MM-dd";

    private ReportWriterService organizationReversionReportWriterService;
    private ParameterService parameterService;
    private ConfigurationService configurationService;

    /**
     * Gets the organizationReversionReportWriterService attribute.
     *
     * @return Returns the organizationReversionReportWriterService.
     */
    public ReportWriterService getOrganizationReversionReportWriterService() {
        return organizationReversionReportWriterService;
    }

    /**
     * Sets the organizationReversionReportWriterService attribute value.
     *
     * @param organizationReversionReportWriterService The organizationReversionReportWriterService to set.
     */
    public void setOrganizationReversionReportWriterService(ReportWriterService organizationReversionReportWriterService) {
        this.organizationReversionReportWriterService = organizationReversionReportWriterService;
    }

    /**
     * Runs the Organization Reversion Year End Process for the end of a fiscal year (ie, a process that
     * runs before the fiscal year end, and thus uses current account, etc.)
     *
     * @param outputGroup                 the origin entry group that this process should save entries to
     * @param jobParameters               the parameters used in the process
     * @param organizationReversionCounts a Map of named statistics generated by running the process
     * @see org.kuali.kfs.gl.batch.service.OrganizationReversionProcessService#organizationReversionProcessEndOfYear(org.kuali.kfs.gl.businessobject.OriginEntryGroup, java.util.Map, java.util.Map)
     */
    public void organizationReversionPriorYearAccountProcess(Map jobParameters, Map<String, Integer> organizationReversionCounts) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("organizationReversionProcessEndOfYear() started");
        }
        OrganizationReversionProcess orp = SpringContext.getBean(OrganizationReversionProcess.class, OrganizationReversionProcessServiceImpl.ORGANIZATION_REVERSION_PRIOR_YEAR_ACCOUNT_PROCESS_BEAN_NAME);

        orp.organizationReversionProcess(jobParameters, organizationReversionCounts);

        writeReports(orp, jobParameters, organizationReversionCounts);
    }

    /**
     * Organization Reversion Year End Process for the beginning of a fiscal year (ie, the process as it runs
     * after the fiscal year end, thus using prior year account, etc.)
     *
     * @param jobParameters               the parameters used in the process
     * @param organizationReversionCounts a Map of named statistics generated by running the process
     * @see org.kuali.kfs.gl.batch.service.OrganizationReversionProcessService#organizationReversionProcessBeginningOfYear(org.kuali.kfs.gl.businessobject.OriginEntryGroup, java.util.Map, java.util.Map)
     */
    public void organizationReversionCurrentYearAccountProcess(Map jobParameters, Map<String, Integer> organizationReversionCounts) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("organizationReversionProcessEndOfYear() started");
        }
        OrganizationReversionProcess orp = SpringContext.getBean(OrganizationReversionProcess.class, OrganizationReversionProcessServiceImpl.ORGANIZATION_REVERSION_CURRENT_YEAR_ACCOUNT_PROCESS_BEAN_NAME);

        LOG.info("processing organization reversions for current year accounts");
        orp.organizationReversionProcess(jobParameters, organizationReversionCounts);

        writeReports(orp, jobParameters, organizationReversionCounts);
    }

    /**
     * Returns a Map with the properly initialized parameters for an organization reversion job that is about to run
     *
     * @return a Map holding parameters for the job
     * @see org.kuali.kfs.gl.batch.service.OrganizationReversionProcessService#getJobParameters()
     */
    public Map getJobParameters() {
        // Get job parameters
        Map jobParameters = new HashMap();
        String strTransactionDate = getParameterService().getParameterValueAsString(KfsParameterConstants.GENERAL_LEDGER_BATCH.class, GeneralLedgerConstants.ANNUAL_CLOSING_TRANSACTION_DATE_PARM);
        jobParameters.put(KFSConstants.UNALLOC_OBJECT_CD, getParameterService().getParameterValueAsString(OrganizationReversion.class, GeneralLedgerConstants.OrganizationReversionProcess.UNALLOC_OBJECT_CODE_PARM));
        jobParameters.put(KFSConstants.BEG_BUD_CASH_OBJECT_CD, getParameterService().getParameterValueAsString(OrganizationReversion.class, GeneralLedgerConstants.OrganizationReversionProcess.CARRY_FORWARD_OBJECT_CODE));
        jobParameters.put(KFSConstants.FUND_BAL_OBJECT_CD, getParameterService().getParameterValueAsString(KfsParameterConstants.GENERAL_LEDGER_BATCH.class, GeneralLedgerConstants.ANNUAL_CLOSING_FUND_BALANCE_OBJECT_CODE_PARM));
        String strUniversityFiscalYear = getParameterService().getParameterValueAsString(KfsParameterConstants.GENERAL_LEDGER_BATCH.class, GeneralLedgerConstants.ANNUAL_CLOSING_FISCAL_YEAR_PARM);

        try {
            SimpleDateFormat sdf = new SimpleDateFormat(DATE_FORMAT);
            java.util.Date jud = sdf.parse(strTransactionDate);
            jobParameters.put(KFSConstants.TRANSACTION_DT, new java.sql.Date(jud.getTime()));
        } catch (ParseException e) {
            throw new IllegalArgumentException("TRANSACTION_DT is an invalid date");
        }
        try {
            jobParameters.put(KFSConstants.UNIV_FISCAL_YR, new Integer(strUniversityFiscalYear));
        } catch (NumberFormatException nfe) {
            throw new IllegalArgumentException("UNIV_FISCAL_YR is an invalid year");
        }
        return jobParameters;
    }

    /**
     * @param organizationReversionProcess
     * @param jobParameters
     * @param counts
     */
    public void writeReports(OrganizationReversionProcess organizationReversionProcess, Map jobParameters, Map<String, Integer> counts) {
        // write job parameters
        for (Object jobParameterKeyAsObject : jobParameters.keySet()) {
            if (jobParameterKeyAsObject != null) {
                final String jobParameterKey = jobParameterKeyAsObject.toString();
                getOrganizationReversionReportWriterService().writeParameterLine("%32s %10s", jobParameterKey, jobParameters.get(jobParameterKey));
            }
        }

        // write statistics
        getOrganizationReversionReportWriterService().writeStatisticLine("NUMBER OF GLBL RECORDS READ....: %10d", counts.get("balancesRead"));
        getOrganizationReversionReportWriterService().writeStatisticLine("NUMBER OF GLBL RECORDS SELECTED: %10d", counts.get("balancesSelected"));
        getOrganizationReversionReportWriterService().writeStatisticLine("NUMBER OF SEQ RECORDS WRITTEN..: %10d", counts.get("recordsWritten"));
        getOrganizationReversionReportWriterService().pageBreak();

        // write ledger report
        getOrganizationReversionReportWriterService().writeSubTitle(getConfigurationService().getPropertyValueAsString(KFSKeyConstants.MESSAGE_REPORT_YEAR_END_ORGANIZATION_REVERSION_LEDGER_TITLE_LINE));
        organizationReversionProcess.writeLedgerSummaryReport(getOrganizationReversionReportWriterService());
    }

    /**
     * Sets the implementation of ParameterService to use
     *
     * @param parameterService an implementation of ParameterService
     */
    public void setParameterService(ParameterService parameterService) {
        this.parameterService = parameterService;
    }

    /**
     * Sets the implementation of the ConfigurationService to use
     *
     * @param configurationService an implementation of the ConfigurationService
     */
    public void setConfigurationService(ConfigurationService configurationService) {
        this.configurationService = configurationService;
    }

    /**
     * Gets the parameterService attribute.
     *
     * @return Returns the parameterService.
     */
    public ParameterService getParameterService() {
        return parameterService;
    }

    /**
     * Gets the configurationService attribute.
     *
     * @return Returns the configurationService.
     */
    public ConfigurationService getConfigurationService() {
        return configurationService;
    }
}
