/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2016 The Kuali Foundation
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.tem.document.service.impl;

import org.apache.commons.lang.StringUtils;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.integration.ar.AccountsReceivableCustomerInvoice;
import org.kuali.kfs.integration.ar.AccountsReceivableModuleService;
import org.kuali.kfs.krad.bo.AdHocRouteRecipient;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.service.DocumentService;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.tem.TemConstants;
import org.kuali.kfs.module.tem.TemConstants.TravelAuthorizationParameters;
import org.kuali.kfs.module.tem.TemPropertyConstants;
import org.kuali.kfs.module.tem.batch.TaxableRamificationNotificationStep;
import org.kuali.kfs.module.tem.businessobject.TravelAdvance;
import org.kuali.kfs.module.tem.businessobject.TravelerDetail;
import org.kuali.kfs.module.tem.document.TaxableRamificationDocument;
import org.kuali.kfs.module.tem.document.TravelAuthorizationDocument;
import org.kuali.kfs.module.tem.document.service.TaxableRamificationDocumentService;
import org.kuali.kfs.module.tem.document.service.TravelDocumentService;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.rice.core.api.datetime.DateTimeService;
import org.kuali.rice.core.api.util.type.KualiDecimal;
import org.kuali.rice.kew.api.exception.WorkflowException;
import org.kuali.rice.kim.api.identity.Person;
import org.kuali.rice.kim.api.identity.PersonService;
import org.springframework.transaction.annotation.Transactional;

import java.sql.Date;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * implement the service calls and operations on tax ramification document
 */
public class TaxableRamificationDocumentServiceImpl implements TaxableRamificationDocumentService {
    public static org.apache.log4j.Logger LOG = org.apache.log4j.Logger.getLogger(TaxableRamificationDocumentServiceImpl.class);

    private DocumentService documentService;
    private ParameterService parameterService;
    private BusinessObjectService businessObjectService;
    private TravelDocumentService travelDocumentService;
    private AccountsReceivableModuleService accountsReceivableModuleService;

    /**
     * @see org.kuali.kfs.module.tem.service.TaxableRamificationDocumentService#getAllQualifiedOutstandingTravelAdvance()
     */
    @Override
    public List<TravelAdvance> getAllQualifiedOutstandingTravelAdvance() {
        List<TravelAdvance> qualifiedOutstandingTravelAdvance = new ArrayList<TravelAdvance>();

        List<String> customerTypeCodes = this.getTravelerCustomerTypes();
        Integer customerInvoiceAge = this.getNotificationOnDays();

        Date lastTaxableRamificationNotificationDate = this.getLastTaxableRamificationNotificationDate();
        if (ObjectUtils.isNull(lastTaxableRamificationNotificationDate)) {
            lastTaxableRamificationNotificationDate = SpringContext.getBean(DateTimeService.class).getCurrentSqlDate();
        }
        Map<String, KualiDecimal> invoiceOpenAmountMap = this.getAccountsReceivableModuleService().getCustomerInvoiceOpenAmount(customerTypeCodes, customerInvoiceAge, lastTaxableRamificationNotificationDate);
        Set<String> arInvoiceDocNumbers = invoiceOpenAmountMap.keySet();

        if (ObjectUtils.isNotNull(arInvoiceDocNumbers) && !arInvoiceDocNumbers.isEmpty()) {
            qualifiedOutstandingTravelAdvance = this.getTravelDocumentService().getOutstandingTravelAdvanceByInvoice(arInvoiceDocNumbers);
        }

        return qualifiedOutstandingTravelAdvance;
    }

    /**
     * @see org.kuali.kfs.module.tem.service.TaxRamificationDocumentService#blanketApproveTaxRamificationDocument(org.kuali.kfs.module.tem.document.TaxRamificationDocument)
     */
    @Override
    @Transactional
    public void blanketApproveRamificationDocument(TaxableRamificationDocument taxRamificationDocument) {
        try {
            this.addAdHocRoutePersons(taxRamificationDocument);

            this.getDocumentService().blanketApproveDocument(taxRamificationDocument, KFSConstants.EMPTY_STRING, new ArrayList<AdHocRouteRecipient>(taxRamificationDocument.getAdHocRoutePersons()));
        } catch (WorkflowException we) {
            LOG.error("Failed to blanket approve the given tax ramification document. ", we);
            throw new RuntimeException(we);
        }
    }

    /**
     * @see org.kuali.kfs.module.tem.service.TaxRamificationDocumentService#createAndBlanketApproveTaxRamificationDocument(org.kuali.kfs.module.tem.businessobject.TravelAdvance)
     */
    @Override
    @Transactional
    public TaxableRamificationDocument createAndBlanketApproveRamificationDocument(TravelAdvance travelAdvance) {
        if (this.hasTaxableRamification(travelAdvance)) {
            throw new RuntimeException("There exists a tax ramification document created from the given travel advance. " + travelAdvance);
        }

        TaxableRamificationDocument taxRamificationDocument = this.createRamificationDocument(travelAdvance);

        if (ObjectUtils.isNotNull(taxRamificationDocument)) {
            this.blanketApproveRamificationDocument(taxRamificationDocument);
        }

        return taxRamificationDocument;
    }

    /**
     * @see org.kuali.kfs.module.tem.service.TaxRamificationDocumentService#hasTaxRamification(org.kuali.kfs.module.tem.businessobject.TravelAdvance)
     */
    @Override
    public boolean hasTaxableRamification(TravelAdvance travelAdvance) {
        Map<String, Object> fieldValues = new HashMap<String, Object>();
        fieldValues.put(TemPropertyConstants.TRAVEL_ADVANCE_DOCUMENT_NUMBER, travelAdvance.getDocumentNumber());

        int count = this.getBusinessObjectService().countMatching(TaxableRamificationDocument.class, fieldValues);

        return count > 0;
    }

    /**
     * @see org.kuali.kfs.module.tem.service.TaxRamificationDocumentService#createTaxRamificationDocument(org.kuali.kfs.module.tem.businessobject.TravelAdvance)
     */
    @Override
    public TaxableRamificationDocument createRamificationDocument(TravelAdvance travelAdvance) {
        try {
            TaxableRamificationDocument taxRamificationDocument = (TaxableRamificationDocument) this.getDocumentService().getNewDocument(TaxableRamificationDocument.class);

            this.populateTaxRamificationDocument(taxRamificationDocument, travelAdvance);

            return taxRamificationDocument;
        } catch (WorkflowException we) {
            LOG.error(we);
            throw new RuntimeException(we);
        }
    }

    /**
     * add adhoc recipients to the given taxable ramification document
     */
    protected void addAdHocRoutePersons(TaxableRamificationDocument taxableRamificationDocument) {
        final Set<String> adHocRecipientPrincipalIds = getFYIRecipientsPrincipalIds(taxableRamificationDocument);
        for (String principalId : adHocRecipientPrincipalIds) {
            getTravelDocumentService().addAdHocFYIRecipient(taxableRamificationDocument, principalId);
        }
    }

    /**
     * collect all adhoc recipients of the given tax ramification document.  This is meant for overriding, in case other recipients should
     * be notified
     */
    protected Set<String> getFYIRecipientsPrincipalIds(TaxableRamificationDocument taxRamificationDocument) {
        Set<String> adHocRecipients = new HashSet<String>();

        return adHocRecipients;
    }

    /**
     * populate the given tax ramification document with the information provided by the given travel advance
     */
    protected void populateTaxRamificationDocument(TaxableRamificationDocument taxRamificationDocument, TravelAdvance travelAdvance) {
        taxRamificationDocument.setArInvoiceDocNumber(travelAdvance.getArInvoiceDocNumber());

        taxRamificationDocument.setTravelAdvanceDocumentNumber(travelAdvance.getDocumentNumber());
        taxRamificationDocument.setTravelAdvance(travelAdvance);

        TravelAuthorizationDocument travelAuthorizationDocument = this.getTravelAuthorizationDocument(travelAdvance);
        String travelDocumentIdentifier = travelAuthorizationDocument.getTravelDocumentIdentifier();
        taxRamificationDocument.setTravelDocumentIdentifier(travelDocumentIdentifier);

        TravelerDetail travelerDetail = travelAuthorizationDocument.getTraveler();
        this.refreshTraverler(travelerDetail);
        taxRamificationDocument.setTravelerDetailId(travelerDetail.getId());
        taxRamificationDocument.setTravelerDetail(travelerDetail);

        AccountsReceivableCustomerInvoice customerInvoice = this.getOpenCustomerInvoice(travelAdvance);
        taxRamificationDocument.setOpenAmount(customerInvoice.getOpenAmount());
        taxRamificationDocument.setInvoiceAmount(customerInvoice.getTotalDollarAmount());
        taxRamificationDocument.setDueDate(customerInvoice.getInvoiceDueDate());

        String taxRamificationNotice = this.getNotificationText();
        taxRamificationDocument.setTaxableRamificationNotice(taxRamificationNotice);

        taxRamificationDocument.getDocumentHeader().setOrganizationDocumentNumber(String.valueOf(travelDocumentIdentifier));

        String travelerPrincipalName = StringUtils.upperCase(travelerDetail.getPrincipalName());
        String description = this.getNotificationSubject() + travelerPrincipalName;
        taxRamificationDocument.getDocumentHeader().setDocumentDescription(StringUtils.left(description, KFSConstants.getMaxLengthOfDocumentDescription()));
    }

    /**
     * refresh the given traveler's information
     */
    protected void refreshTraverler(TravelerDetail travelerDetail) {
        if (ObjectUtils.isNull(travelerDetail)) {
            return;
        }

        String principalName = travelerDetail.getPrincipalName();
        if (StringUtils.isNotBlank(principalName)) {
            return;
        }

        String principalId = travelerDetail.getPrincipalId();
        Person person = SpringContext.getBean(PersonService.class).getPerson(principalId);

        if (ObjectUtils.isNotNull(person)) {
            travelerDetail.setPrincipalName(person.getPrincipalName());
        }
    }

    /**
     * get the last taxable ramification notification date
     */
    protected Date getLastTaxableRamificationNotificationDate() {
        return this.getTravelDocumentService().findLatestTaxableRamificationNotificationDate();
    }

    /**
     * get the travel authorization document associated with the given travel advance
     */
    protected TravelAuthorizationDocument getTravelAuthorizationDocument(TravelAdvance travelAdvance) {
        String travelAuthorizationDocumentNumber = travelAdvance.getDocumentNumber();

        return this.getBusinessObjectService().findBySinglePrimaryKey(TravelAuthorizationDocument.class, travelAuthorizationDocumentNumber);
    }

    /**
     * get the open customer invoice amount associated with the given travel advance
     */
    protected AccountsReceivableCustomerInvoice getOpenCustomerInvoice(TravelAdvance travelAdvance) {
        String customerInvoiceDocumentNumber = travelAdvance.getArInvoiceDocNumber();

        return this.getAccountsReceivableModuleService().getOpenCustomerInvoice(customerInvoiceDocumentNumber);
    }

    /**
     * get the notification text from an application parameter
     */
    protected String getNotificationText() {
        return this.getParameterService().getParameterValueAsString(TaxableRamificationNotificationStep.class, TemConstants.TaxRamificationParameter.NOTIFICATION_TEXT_PARAM_NAME);
    }


    /**
     * get the notification subject
     */
    protected String getNotificationSubject() {
        return "Notice for ";
    }

    /**
     * get the notification text from an application parameter
     */
    protected List<String> getTravelerCustomerTypes() {
        return new ArrayList<String>(getParameterService().getParameterValuesAsString(TravelAuthorizationDocument.class, TravelAuthorizationParameters.CUSTOMER_TYPE_CODE));
    }

    /**
     * get the timing of when the aging email notification should be sent
     */
    protected Integer getNotificationOnDays() {
        String daysAsString = this.getParameterService().getParameterValueAsString(TaxableRamificationNotificationStep.class, TemConstants.TaxRamificationParameter.NOTIFICATION_DAYS_PARAM_NAME);
        if (!StringUtils.isNumeric(daysAsString)) {
            return TemConstants.DEFAULT_NOTIFICATION_DAYS;
        }

        return Integer.parseInt(daysAsString);
    }

    /**
     * Gets the documentService attribute.
     *
     * @return Returns the documentService.
     */
    public DocumentService getDocumentService() {
        return documentService;
    }

    /**
     * Sets the documentService attribute value.
     *
     * @param documentService The documentService to set.
     */
    public void setDocumentService(DocumentService documentService) {
        this.documentService = documentService;
    }

    /**
     * Gets the parameterService attribute.
     *
     * @return Returns the parameterService.
     */
    public ParameterService getParameterService() {
        return parameterService;
    }

    /**
     * Sets the parameterService attribute value.
     *
     * @param parameterService The parameterService to set.
     */
    public void setParameterService(ParameterService parameterService) {
        this.parameterService = parameterService;
    }

    /**
     * Gets the businessObjectService attribute.
     *
     * @return Returns the businessObjectService.
     */
    public BusinessObjectService getBusinessObjectService() {
        return businessObjectService;
    }

    /**
     * Sets the businessObjectService attribute value.
     *
     * @param businessObjectService The businessObjectService to set.
     */
    public void setBusinessObjectService(BusinessObjectService businessObjectService) {
        this.businessObjectService = businessObjectService;
    }

    /**
     * Gets the accountsReceivableModuleService attribute.
     *
     * @return Returns the accountsReceivableModuleService.
     */
    public AccountsReceivableModuleService getAccountsReceivableModuleService() {
        return accountsReceivableModuleService;
    }

    /**
     * Sets the accountsReceivableModuleService attribute value.
     *
     * @param accountsReceivableModuleService The accountsReceivableModuleService to set.
     */
    public void setAccountsReceivableModuleService(AccountsReceivableModuleService accountsReceivableModuleService) {
        this.accountsReceivableModuleService = accountsReceivableModuleService;
    }

    /**
     * Gets the travelDocumentService attribute.
     *
     * @return Returns the travelDocumentService.
     */
    public TravelDocumentService getTravelDocumentService() {
        return travelDocumentService;
    }

    /**
     * Sets the travelDocumentService attribute value.
     *
     * @param travelDocumentService The travelDocumentService to set.
     */
    public void setTravelDocumentService(TravelDocumentService travelDocumentService) {
        this.travelDocumentService = travelDocumentService;
    }
}
