/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2020 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.businessobject.admin;

import org.kuali.kfs.krad.bo.BusinessObjectBase;
import org.kuali.kfs.krad.util.KRADConstants;
import org.kuali.kfs.krad.util.KRADUtils;
import org.kuali.kfs.module.ar.businessobject.InvoiceTemplate;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.businessobject.admin.DefaultBoAdminService;
import org.kuali.kfs.sys.identity.KfsKimAttributes;
import org.kuali.kfs.kim.api.KimConstants;
import org.kuali.rice.kim.api.identity.Person;

import java.util.HashMap;
import java.util.Map;

public class InvoiceTemplateAdminService extends DefaultBoAdminService {

    @Override
    protected final Map<String, String> buildRoleQualifications(Object businessObject, String principalId) {
        Map<String, String> roleQualifications = super.buildRoleQualifications(businessObject, principalId);
        roleQualifications.putAll(KRADUtils.getNamespaceAndComponentSimpleName(businessObject.getClass()));
        InvoiceTemplate invoiceTemplate = (InvoiceTemplate) businessObject;
        roleQualifications.put(KfsKimAttributes.CHART_OF_ACCOUNTS_CODE, invoiceTemplate.getBillByChartOfAccountCode());
        roleQualifications.put(KfsKimAttributes.ORGANIZATION_CODE, invoiceTemplate.getBilledByOrganizationCode());
        return roleQualifications;
    }

    /**
     * Overrides the default because in the case of Invoice Templates, we want to base the copy authorization on
     * the users chart/org (passed as role qualifications), whereas typically if the user can create a doc, they can
     * copy one, regardless of qualifications. If we always passed the qualifications, some lookups would be missing
     * copy links.
     *
     * @param businessObject the Invoice Template to check copy permissions for
     * @param person the person to check copy permissions for
     * @return if the person is allowed to copy this Invoice Template
     */
    @Override
    public boolean allowsCopy(BusinessObjectBase businessObject, Person person) {
        if (businessObject == null || !maintDocSupportsCopy(businessObject)) {
            return false;
        }

        String principalId = person.getPrincipalId();
        Map<String, String> roleQualifiers = new HashMap<>(buildRoleQualifications(businessObject, principalId));
        Map<String, String> permissionDetails = new HashMap<>(buildPermissionDetails(businessObject));
        permissionDetails.put(KimConstants.AttributeConstants.DOCUMENT_TYPE_NAME,
                documentDictionaryService.getMaintenanceDocumentTypeName(businessObject.getClass()));
        permissionDetails.put(KRADConstants.MAINTENANCE_ACTN, KRADConstants.MAINTENANCE_NEW_ACTION);

        return permissionService.isAuthorizedByTemplate(person.getPrincipalId(), KFSConstants.CoreModuleNamespaces.KFS,
                KimConstants.PermissionTemplateNames.CREATE_MAINTAIN_RECORDS, permissionDetails, roleQualifiers);
    }
}
