/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2020 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.businessobject.lookup;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.integration.ar.AccountsReceivableBillingFrequency;
import org.kuali.kfs.integration.cg.ContractsAndGrantsBillingAward;
import org.kuali.kfs.integration.cg.ContractsAndGrantsModuleBillingService;
import org.kuali.kfs.kns.document.authorization.BusinessObjectRestrictions;
import org.kuali.kfs.kns.lookup.CollectionIncomplete;
import org.kuali.kfs.kns.lookup.HtmlData;
import org.kuali.kfs.kns.lookup.LookupUtils;
import org.kuali.kfs.kns.web.comparator.CellComparatorHelper;
import org.kuali.kfs.kns.web.struts.form.LookupForm;
import org.kuali.kfs.kns.web.ui.Column;
import org.kuali.kfs.kns.web.ui.ResultRow;
import org.kuali.kfs.krad.util.BeanPropertyComparator;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.KRADConstants;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.krad.util.UrlFactory;
import org.kuali.kfs.module.ar.ArConstants;
import org.kuali.kfs.module.ar.ArPropertyConstants;
import org.kuali.kfs.module.ar.businessobject.ContractsGrantsInvoiceLookupResult;
import org.kuali.kfs.module.ar.businessobject.ContractsGrantsInvoiceLookupResultAward;
import org.kuali.kfs.module.ar.report.service.ContractsGrantsInvoiceReportService;
import org.kuali.kfs.module.ar.report.service.ContractsGrantsReportHelperService;
import org.kuali.kfs.module.ar.web.ui.ContractsGrantsLookupResultRow;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.rice.core.web.format.Formatter;
import org.kuali.kfs.kim.api.KimConstants;
import org.kuali.rice.kim.api.identity.Person;
import org.kuali.rice.kim.api.services.IdentityManagementService;
import org.kuali.rice.krad.bo.BusinessObject;

import java.beans.PropertyDescriptor;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * Defines a lookupable helper service class for Contracts & Grants Invoices.
 */
public class ContractsGrantsInvoiceLookupableHelperServiceImpl extends
        AccountsReceivableLookupableHelperServiceImplBase {

    private static final Logger LOG = LogManager.getLogger(ContractsGrantsInvoiceLookupableHelperServiceImpl.class);
    protected ContractsAndGrantsModuleBillingService contractsAndGrantsModuleBillingService;
    protected ContractsGrantsReportHelperService contractsGrantsReportHelperService;
    protected ContractsGrantsInvoiceReportService contractsGrantsInvoiceReportService;
    private IdentityManagementService identityManagementService;

    /**
     * This method performs the lookup and returns a collection of lookup items
     *
     * @param lookupForm  struts form
     * @param resultTable table to add the result rows to
     * @param bounded     boolean whether the results should be bounded or not
     * @return collection of search results
     */
    @Override
    public Collection<? extends BusinessObject> performLookup(LookupForm lookupForm, Collection<ResultRow> resultTable,
            boolean bounded) {
        List<? extends BusinessObject> displayList;

        // Call search method to get results - always use unbounded to get the entire set of results.
        displayList = getSearchResultsUnbounded(lookupForm.getFieldsForLookup());

        List<String> pkNames = getBusinessObjectMetaDataService().listPrimaryKeyFieldNames(getBusinessObjectClass());
        Person user = getPerson();

        for (BusinessObject element : displayList) {
            LOG.debug("Doing lookup for " + element.getClass());

            ContractsGrantsInvoiceLookupResult result = (ContractsGrantsInvoiceLookupResult) element;

            BusinessObjectRestrictions businessObjectRestrictions = getBusinessObjectAuthorizationService()
                    .getLookupResultRestrictions(result, user);
            List<ResultRow> subResultRows = buildSubResultRows(result,
                    businessObjectRestrictions);

            Collection<Column> columns = getColumns(element, businessObjectRestrictions);
            ContractsGrantsLookupResultRow row = new ContractsGrantsLookupResultRow((List<Column>) columns,
                    subResultRows, "", getActionUrls(element, pkNames, businessObjectRestrictions));
            resultTable.add(row);
        }

        return displayList;
    }

    Person getPerson() {
        return GlobalVariables.getUserSession().getPerson();
    }

    protected List<ResultRow> buildSubResultRows(ContractsGrantsInvoiceLookupResult result,
            BusinessObjectRestrictions businessObjectRestrictions) {

        List<ResultRow> subResultRows = new ArrayList<>();
        List<String> subResultRowProposalNumbers = new ArrayList<>();

        for (ContractsGrantsInvoiceLookupResultAward lookupResultAward : result.getLookupResultAwards()) {
            List<Column> subResultColumns = new ArrayList<>();

            for (String propertyName : result.getAwardAttributesForDisplay()) {
                subResultColumns.add(setupResultsColumn(lookupResultAward, propertyName, businessObjectRestrictions));
            }

            ResultRow subResultRow = new ResultRow(subResultColumns, "", "");
            if (needsObjectIdForCheckbox(subResultRowProposalNumbers, lookupResultAward)) {
                subResultRow.setObjectId(lookupResultAward.getObjectId());
                subResultRowProposalNumbers.add(lookupResultAward.getProposalNumber());
            }
            subResultRows.add(subResultRow);
        }

        return subResultRows;
    }

    /**
     * We want to populate the object id and display a select checkbox for the row if it is MS or PDBS billing
     * or the first award account for the award.
     *
     * @param subResultRowProposalNumbers list of proposal numbers for this row
     * @param lookupResultAward the lookup result award for checking billing schedule and getting the proposal number
     * @return true if the objectId should be set, false otherwise
     */
    private boolean needsObjectIdForCheckbox(List<String> subResultRowProposalNumbers,
            ContractsGrantsInvoiceLookupResultAward lookupResultAward) {
        return isScheduledBilling(lookupResultAward)
                || !subResultRowProposalNumbers.contains(lookupResultAward.getProposalNumber());
    }

    private boolean isScheduledBilling(ContractsGrantsInvoiceLookupResultAward result) {
        return ArConstants.BillingFrequencyValues.isMilestone(result)
                || ArConstants.BillingFrequencyValues.isPredeterminedBilling(result);
    }

    /**
     * overriding this method to convert the list of awards to a list of ContractsGrantsInvoiceLookupResult
     */
    @Override
    public List<? extends BusinessObject> getSearchResultsUnbounded(Map<String, String> fieldValues) {
        Collection searchResultsCollection;
        // Get the list of awards
        searchResultsCollection = getSearchResultsHelper(LookupUtils.forceUppercase(
                getBusinessObjectClass(), fieldValues), true);
        // Convert to suitable list
        searchResultsCollection = getContractsGrantsInvoiceReportService()
                .getPopulatedContractsGrantsInvoiceLookupResults(searchResultsCollection);
        filterSearchResults(searchResultsCollection);
        return this.buildSearchResultList(searchResultsCollection, (long) searchResultsCollection.size());
    }

    /**
     * build the search result list from the given collection and the number of all qualified search results
     *
     * @param searchResultsCollection the given search results, which may be a subset of the qualified search results
     * @param actualSize              the number of all qualified search results
     * @return the search result list with the given results and actual size
     */
    protected List buildSearchResultList(Collection searchResultsCollection, Long actualSize) {
        CollectionIncomplete results = new CollectionIncomplete(searchResultsCollection, actualSize);

        // Sort list if default sort column given
        List searchResults = results;
        List defaultSortColumns = getDefaultSortColumns();
        if (defaultSortColumns.size() > 0) {
            results.sort(new BeanPropertyComparator(defaultSortColumns, true));
        }
        return searchResults;
    }

    @Override
    protected List<? extends BusinessObject> getSearchResultsHelper(Map<String, String> fieldValues, boolean unbounded) {
        return contractsAndGrantsModuleBillingService.lookupAwards(fieldValues, unbounded);
    }

    /**
     * Filter out awards the user is not authorized to initiate CINV docs for (not a fund manager for the award) from
     * the search results.
     *
     * @param searchResultsCollection search results to filter
     */
    void filterSearchResults(Collection<ContractsGrantsInvoiceLookupResult> searchResultsCollection) {
        for (ContractsGrantsInvoiceLookupResult lookupResult: searchResultsCollection) {
            lookupResult.setLookupResultAwards(removeAwardsIfNotAuthorized(lookupResult));
        }

        searchResultsCollection.removeIf(lookupResult -> lookupResult.getLookupResultAwards().size() == 0);
    }

    private List<ContractsGrantsInvoiceLookupResultAward> removeAwardsIfNotAuthorized(
            ContractsGrantsInvoiceLookupResult lookupResult) {

        Map<String, String> permissionDetails = new HashMap<>();
        permissionDetails.put(KimConstants.AttributeConstants.DOCUMENT_TYPE_NAME,
                ArConstants.ArDocumentTypeCodes.CONTRACTS_GRANTS_INVOICE);
        Map<String, String> qualificationDetails = new HashMap<>();

        return lookupResult.getLookupResultAwards().stream()
                .filter(lookupResultAward -> {
                    qualificationDetails.put(KFSPropertyConstants.PROPOSAL_NUMBER,
                            lookupResultAward.getProposalNumber());
                    return identityManagementService.isAuthorizedByTemplateName(
                            getPrincipalId(),
                            KRADConstants.KUALI_RICE_SYSTEM_NAMESPACE,
                            KimConstants.PermissionTemplateNames.INITIATE_DOCUMENT, permissionDetails,
                            qualificationDetails);
                }).collect(Collectors.toList());
    }

    String getPrincipalId() {
        return GlobalVariables.getUserSession().getPrincipalId();
    }

    /**
     * @param element
     * @param attributeName
     * @return Column
     */
    protected Column setupResultsColumn(BusinessObject element, String attributeName, BusinessObjectRestrictions businessObjectRestrictions) {
        Column col = new Column();

        col.setPropertyName(attributeName);

        String columnTitle = getDataDictionaryService().getAttributeLabel(element.getClass(), attributeName);
        if (StringUtils.isBlank(columnTitle)) {
            columnTitle = getDataDictionaryService().getCollectionLabel(element.getClass(), attributeName);
        }
        col.setColumnTitle(columnTitle);
        col.setMaxLength(getDataDictionaryService().getAttributeMaxLength(element.getClass(), attributeName));

        try {
            Class formatterClass = getDataDictionaryService().getAttributeFormatter(element.getClass(), attributeName);
            Formatter formatter = null;
            if (formatterClass != null) {
                formatter = (Formatter) formatterClass.newInstance();
                col.setFormatter(formatter);
            }

            // Pick off result column from result list, do formatting
            String propValue = KFSConstants.EMPTY_STRING;
            Object prop = ObjectUtils.getPropertyValue(element, attributeName);

            // Set comparator and formatter based on property type
            Class propClass = null;

            PropertyDescriptor propDescriptor = PropertyUtils.getPropertyDescriptor(element, col.getPropertyName());
            if (propDescriptor != null) {
                propClass = propDescriptor.getPropertyType();
            }

            // Formatters
            if (prop != null) {
                propValue = getContractsGrantsReportHelperService().formatByType(prop, formatter);
            }

            // Comparator
            col.setComparator(CellComparatorHelper.getAppropriateComparatorForPropertyClass(propClass));
            col.setValueComparator(CellComparatorHelper.getAppropriateValueComparatorForPropertyClass(propClass));
            propValue = super.maskValueIfNecessary(element, col.getPropertyName(), propValue, businessObjectRestrictions);
            col.setPropertyValue(propValue);

            if (StringUtils.isNotBlank(propValue)) {
                col.setColumnAnchor(getInquiryUrl(element, col.getPropertyName()));
            }
        } catch (InstantiationException ie) {
            throw new RuntimeException("Unable to get new instance of formatter class for property " + col.getPropertyName(), ie);
        } catch (IllegalAccessException | InvocationTargetException | NoSuchMethodException ex) {
            throw new RuntimeException("Cannot access PropertyType for property " + "'" + col.getPropertyName() + "' " + " on an instance of '" + element.getClass().getName() + "'.", ex);
        }
        return col;
    }


    /**
     * Overridden to build Inquiry URL for Billing Frequency and Proposal Number since
     * ContractsGrantsInvoiceLookupResultAward is not persistable so the super call won't build the Inquiry URLs
     * correctly.
     */
    @Override
    public HtmlData getInquiryUrl(BusinessObject bo, String propertyName) {
        if (ArPropertyConstants.BILLING_FREQUENCY_CODE.equals(propertyName)) {
            return buildInquiryUrl(AccountsReceivableBillingFrequency.class.getName(), KFSPropertyConstants.FREQUENCY,
                    ((ContractsGrantsInvoiceLookupResultAward) bo).getBillingFrequencyCode()
            );
        } else if (KFSPropertyConstants.PROPOSAL_NUMBER.equals(propertyName)) {
            return buildInquiryUrl(ContractsAndGrantsBillingAward.class.getName(), KFSPropertyConstants.PROPOSAL_NUMBER,
                    ((ContractsGrantsInvoiceLookupResultAward) bo).getProposalNumber()
            );
        } else {
            return super.getInquiryUrl(bo, propertyName);
        }
    }

    private HtmlData buildInquiryUrl(String className, String parameterKey, String parameterValue) {
        HtmlData.AnchorHtmlData inquiryHref = new HtmlData.AnchorHtmlData(KRADConstants.EMPTY_STRING,
                KRADConstants.EMPTY_STRING);

        Map<String, String> parameters = new HashMap<>();
        parameters.put(KRADConstants.BUSINESS_OBJECT_CLASS_ATTRIBUTE, className);
        parameters.put(KRADConstants.DISPATCH_REQUEST_PARAMETER, KFSConstants.START_METHOD);
        parameters.put(parameterKey, parameterValue);
        inquiryHref.setHref(UrlFactory.parameterizeUrl(KFSConstants.INQUIRY_ACTION, parameters));

        return inquiryHref;
    }

    /**
     * Constructs the list of columns for the search results. All properties for the column objects come from the DataDictionary.
     *
     * @param bo
     * @return Collection<Column>
     */
    protected Collection<Column> getColumns(BusinessObject bo, BusinessObjectRestrictions businessObjectRestrictions) {
        Collection<Column> columns = new ArrayList<>();

        for (String attributeName : getBusinessObjectDictionaryService().getLookupResultFieldNames(bo.getClass())) {
            columns.add(setupResultsColumn(bo, attributeName, businessObjectRestrictions));
        }
        return columns;
    }

    public ContractsAndGrantsModuleBillingService getContractsAndGrantsModuleBillingService() {
        return contractsAndGrantsModuleBillingService;
    }

    public void setContractsAndGrantsModuleBillingService(ContractsAndGrantsModuleBillingService contractsAndGrantsModuleBillingService) {
        this.contractsAndGrantsModuleBillingService = contractsAndGrantsModuleBillingService;
    }

    public ContractsGrantsReportHelperService getContractsGrantsReportHelperService() {
        return contractsGrantsReportHelperService;
    }

    public void setContractsGrantsReportHelperService(ContractsGrantsReportHelperService contractsGrantsReportHelperService) {
        this.contractsGrantsReportHelperService = contractsGrantsReportHelperService;
    }

    public ContractsGrantsInvoiceReportService getContractsGrantsInvoiceReportService() {
        return contractsGrantsInvoiceReportService;
    }

    public void setContractsGrantsInvoiceReportService(ContractsGrantsInvoiceReportService contractsGrantsInvoiceReportService) {
        this.contractsGrantsInvoiceReportService = contractsGrantsInvoiceReportService;
    }

    public IdentityManagementService getIdentityManagementService() {
        return identityManagementService;
    }

    public void setIdentityManagementService(IdentityManagementService identityManagementService) {
        this.identityManagementService = identityManagementService;
    }
}
