/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2020 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.businessobject.lookup;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.kns.web.struts.form.LookupForm;
import org.kuali.kfs.kns.web.ui.ResultRow;
import org.kuali.kfs.krad.dao.LookupDao;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.KRADConstants;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ar.ArConstants;
import org.kuali.kfs.module.ar.ArPropertyConstants;
import org.kuali.kfs.module.ar.businessobject.ContractsGrantsInvoiceReport;
import org.kuali.kfs.module.ar.document.ContractsGrantsInvoiceDocument;
import org.kuali.kfs.module.ar.document.CustomerInvoiceDocument;
import org.kuali.kfs.module.ar.document.service.CustomerInvoiceDocumentService;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.businessobject.FinancialSystemDocumentHeader;
import org.kuali.kfs.sys.document.service.FinancialSystemDocumentService;
import org.kuali.rice.core.api.datetime.DateTimeService;
import org.kuali.rice.core.api.search.SearchOperator;
import org.kuali.rice.core.api.util.type.KualiDecimal;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class ContractsGrantsInvoiceReportLookupableHelperServiceImpl
        extends ContractsGrantsReportLookupableHelperServiceImplBase {

    protected CustomerInvoiceDocumentService customerInvoiceDocumentService;
    protected DateTimeService dateTimeService;
    protected FinancialSystemDocumentService financialSystemDocumentService;
    private LookupDao lookupDao;

    /**
     * Validate the pattern for the ageInDays and remainingValue
     */
    @Override
    public void validateSearchParameters(Map<String, String> fieldValues) {
        super.validateSearchParameters(fieldValues);
        validateSearchParametersForOperatorAndValue(fieldValues, ArPropertyConstants.AGE_IN_DAYS);
        validateSearchParametersForOperatorAndValue(fieldValues, ArPropertyConstants.REMAINING_AMOUNT);

        final String upperBoundInvoiceDueDate = fieldValues.get(
                ArPropertyConstants.CustomerInvoiceDocumentFields.INVOICE_DUE_DATE);
        validateDateField(upperBoundInvoiceDueDate, ArPropertyConstants.CustomerInvoiceDocumentFields.INVOICE_DUE_DATE,
                getDateTimeService());
        final String lowerBoundInvoiceDueDate = fieldValues.get(KRADConstants.LOOKUP_RANGE_LOWER_BOUND_PROPERTY_PREFIX
                + ArPropertyConstants.CustomerInvoiceDocumentFields.INVOICE_DUE_DATE);
        validateDateField(lowerBoundInvoiceDueDate, KRADConstants.LOOKUP_RANGE_LOWER_BOUND_PROPERTY_PREFIX
                + ArPropertyConstants.CustomerInvoiceDocumentFields.INVOICE_DUE_DATE, getDateTimeService());
        final String lowerBoundInvoiceDate = fieldValues.get(
                ArPropertyConstants.ContractsGrantsAgingReportFields.INVOICE_DATE_FROM);
        validateDateField(lowerBoundInvoiceDate, ArPropertyConstants.ContractsGrantsAgingReportFields.INVOICE_DATE_FROM,
                getDateTimeService());
        final String upperBoundInvoiceDate = fieldValues.get(
                ArPropertyConstants.ContractsGrantsAgingReportFields.INVOICE_DATE_TO);
        validateDateField(upperBoundInvoiceDate, ArPropertyConstants.ContractsGrantsAgingReportFields.INVOICE_DATE_TO,
                getDateTimeService());
    }

    @Override
    public Collection<ContractsGrantsInvoiceReport> performLookup(LookupForm lookupForm,
            Collection<ResultRow> resultTable, boolean bounded) {
        Map<String, String> lookupFormFields = lookupForm.getFieldsForLookup();

        setBackLocation(lookupForm.getFieldsForLookup().get(KRADConstants.BACK_LOCATION));
        setDocFormKey(lookupForm.getFieldsForLookup().get(KRADConstants.DOC_FORM_KEY));

        Collection<ContractsGrantsInvoiceReport> displayList = new ArrayList<>();

        Map<String, String> invoiceLookupFields = buildCriteriaForInvoiceLookup(lookupFormFields);
        invoiceLookupFields.put(ArPropertyConstants.OPEN_INVOICE_IND, KRADConstants.YES_INDICATOR_VALUE);
        invoiceLookupFields.put(KFSPropertyConstants.DOCUMENT_HEADER + "." +
                KFSPropertyConstants.WORKFLOW_DOCUMENT_STATUS_CODE,
                StringUtils.join(getFinancialSystemDocumentService().getSuccessfulDocumentStatuses(),
                        SearchOperator.OR.op()));
        List<CustomerInvoiceDocument> openInvoiceDocs = new ArrayList<>();

        if (GlobalVariables.getMessageMap().getErrorCount() == 0) {
            openInvoiceDocs.addAll(findInvoiceDocs(invoiceLookupFields));
        }

        final String invoiceReportOption = lookupForm.getFields().get(ArConstants.INVOICE_REPORT_OPTION);

        java.util.Date today = new java.util.Date();

        OperatorAndValue ageInDaysOperator = buildOperatorAndValueFromField(lookupFormFields,
                ArPropertyConstants.AGE_IN_DAYS);
        OperatorAndValue remainingAmountOperator = buildOperatorAndValueFromField(lookupFormFields,
                ArPropertyConstants.REMAINING_AMOUNT);

        for (CustomerInvoiceDocument openCGInvoiceDoc : openInvoiceDocs) {
            if (invoiceReportOption.equals(ArConstants.PAST_DUE_INVOICES)
                    && !openCGInvoiceDoc.getInvoiceDueDate().before(today)) {
                continue;
            }

            if (!ObjectUtils.isNull(ageInDaysOperator)
                    && !ageInDaysOperator.applyComparison(openCGInvoiceDoc.getAge())) {
                continue;
            }

            KualiDecimal paymentAmount = getCustomerInvoiceDocumentService()
                    .calculateAppliedPaymentAmount(openCGInvoiceDoc);
            KualiDecimal remainingAmount = openCGInvoiceDoc.getFinancialSystemDocumentHeader()
                    .getFinancialDocumentTotalAmount().subtract(paymentAmount);
            if (!ObjectUtils.isNull(remainingAmountOperator)
                    && !remainingAmountOperator.applyComparison(remainingAmount)) {
                continue;
            }

            ContractsGrantsInvoiceReport cgInvoiceReport = new ContractsGrantsInvoiceReport();

            cgInvoiceReport.setDocumentNumber(openCGInvoiceDoc.getDocumentNumber());
            if (openCGInvoiceDoc instanceof ContractsGrantsInvoiceDocument) {
                cgInvoiceReport.setProposalNumber(((ContractsGrantsInvoiceDocument) openCGInvoiceDoc)
                        .getInvoiceGeneralDetail().getProposalNumber());
            }

            FinancialSystemDocumentHeader documentHeader =
                    (FinancialSystemDocumentHeader) openCGInvoiceDoc.getDocumentHeader();
            cgInvoiceReport.setInvoiceType(documentHeader.getWorkflowDocumentTypeName());

            Date docCreateDate = documentHeader.getWorkflowCreateDate();
            cgInvoiceReport.setInvoiceDate(new java.sql.Date(docCreateDate.getTime()));
            cgInvoiceReport.setInvoiceDueDate(openCGInvoiceDoc.getInvoiceDueDate());
            if (openCGInvoiceDoc.isOpenInvoiceIndicator()) {
                cgInvoiceReport.setOpenInvoiceIndicator(ArConstants.ReportsConstants.INVOICE_INDICATOR_OPEN);
            } else {
                cgInvoiceReport.setOpenInvoiceIndicator(ArConstants.ReportsConstants.INVOICE_INDICATOR_CLOSE);
            }
            cgInvoiceReport.setCustomerNumber(openCGInvoiceDoc.getAccountsReceivableDocumentHeader()
                    .getCustomerNumber());
            cgInvoiceReport.setCustomerName(openCGInvoiceDoc.getAccountsReceivableDocumentHeader().getCustomer()
                    .getCustomerName());
            cgInvoiceReport.setInvoiceAmount(documentHeader.getFinancialDocumentTotalAmount());

            cgInvoiceReport.setPaymentAmount(paymentAmount);
            cgInvoiceReport.setRemainingAmount(remainingAmount);

            if (openCGInvoiceDoc.getAge() != null) {
                cgInvoiceReport.setAgeInDays(openCGInvoiceDoc.getAge().longValue());
            }
            displayList.add(cgInvoiceReport);
        }

        buildResultTable(lookupForm, displayList, resultTable);

        return displayList;
    }

    /**
     * This method looks for both Customer Invoice Documents (INV) and Contracts & Grants Invoice Documents (CINV)
     * that match the search criteria. It uses the LookupDao instead of the LookupService so that it can pass false
     * as the usePrimaryKeyValuesOnly value to findCollectionBySearchHelper and bypass the PK direct query if the
     * search criteria included a document number. This is because both INV docs and CINV docs are stored in the same
     * table and doing a direct PK query twice would return two documents, one INV and one CINV for the same doc #.
     * As you might expect, this results in issues that we would rather avoid.
     *
     * @param invoiceLookupFields search criteria
     * @return list of invoice documents matching the criteria
     */
    private List<CustomerInvoiceDocument> findInvoiceDocs(Map<String, String> invoiceLookupFields) {
        List<CustomerInvoiceDocument> openInvoiceDocs = new ArrayList<>();

        final boolean docTypeCriteriaSpecified = invoiceLookupFields.containsKey(KFSPropertyConstants.DOCUMENT_HEADER
                + "." + KFSPropertyConstants.WORKFLOW_DOCUMENT_TYPE_NAME);
        if (!docTypeCriteriaSpecified) {
            invoiceLookupFields.put(KFSPropertyConstants.DOCUMENT_HEADER + "."
                    + KFSPropertyConstants.WORKFLOW_DOCUMENT_TYPE_NAME, ArConstants.INV_DOCUMENT_TYPE);
        }
        if (StringUtils.equals(invoiceLookupFields.get(KFSPropertyConstants.DOCUMENT_HEADER + "."
                + KFSPropertyConstants.WORKFLOW_DOCUMENT_TYPE_NAME), ArConstants.INV_DOCUMENT_TYPE)) {
            if (!invoiceLookupFields.containsKey(ArPropertyConstants.ContractsGrantsInvoiceDocumentFields
                    .PROPOSAL_NUMBER)) {
                openInvoiceDocs.addAll(lookupDao.findCollectionBySearchHelper(CustomerInvoiceDocument.class,
                        invoiceLookupFields, true, false));
            }
        }
        if (!docTypeCriteriaSpecified) {
            invoiceLookupFields.put(KFSPropertyConstants.DOCUMENT_HEADER + "."
                    + KFSPropertyConstants.WORKFLOW_DOCUMENT_TYPE_NAME,
                    ArConstants.ArDocumentTypeCodes.CONTRACTS_GRANTS_INVOICE);
        }
        if (StringUtils.equals(invoiceLookupFields.get(KFSPropertyConstants.DOCUMENT_HEADER + "."
                + KFSPropertyConstants.WORKFLOW_DOCUMENT_TYPE_NAME),
                ArConstants.ArDocumentTypeCodes.CONTRACTS_GRANTS_INVOICE)) {
            openInvoiceDocs.addAll(lookupDao.findCollectionBySearchHelper(ContractsGrantsInvoiceDocument.class,
                    invoiceLookupFields, true, false));
        }

        return openInvoiceDocs;
    }

    /**
     * Pulls fields which can go directly to lookup from the given lookupFormFields and turns them into a Map of only
     * those fields which should be included in the lookup
     *
     * @param lookupFormFields the fields directly from the lookup form to tweak into the fields the lookup service is
     *                         going to want to use
     * @return the fields as the lookup service will positively react to them
     */
    private Map<String, String> buildCriteriaForInvoiceLookup(Map lookupFormFields) {
        Map<String, String> lookupFields = new HashMap<>();

        final String lowerBoundInvoiceDate = (String) lookupFormFields.get(
                ArPropertyConstants.ContractsGrantsAgingReportFields.INVOICE_DATE_FROM);
        final String upperBoundInvoiceDate = (String) lookupFormFields.get(
                ArPropertyConstants.ContractsGrantsAgingReportFields.INVOICE_DATE_TO);
        final String invoiceDateCriteria = getContractsGrantsReportHelperService().fixDateCriteria(lowerBoundInvoiceDate,
                upperBoundInvoiceDate, true);
        if (!StringUtils.isBlank(invoiceDateCriteria)) {
            lookupFields.put(KFSPropertyConstants.DOCUMENT_HEADER + "." + KFSPropertyConstants.WORKFLOW_CREATE_DATE,
                    invoiceDateCriteria);
        }

        final String invoiceAmount = (String) lookupFormFields.get(
                ArPropertyConstants.TransmitContractsAndGrantsInvoicesLookupFields.INVOICE_AMOUNT);
        if (!StringUtils.isBlank(invoiceAmount)) {
            lookupFields.put(KFSPropertyConstants.DOCUMENT_HEADER + "."
                    + KFSPropertyConstants.FINANCIAL_DOCUMENT_TOTAL_AMOUNT, invoiceAmount);
        }

        final String customerNumber = (String) lookupFormFields.get(ArPropertyConstants.CustomerFields.CUSTOMER_NUMBER);
        if (!StringUtils.isBlank(customerNumber)) {
            lookupFields.put(ArPropertyConstants.ACCOUNTS_RECEIVABLE_DOCUMENT_HEADER + "."
                    + ArPropertyConstants.CustomerFields.CUSTOMER_NUMBER, customerNumber);
        }

        final String invoiceType = (String) lookupFormFields.get(ArPropertyConstants.INVOICE_TYPE);
        if (!StringUtils.isBlank(invoiceType)) {
            lookupFields.put(KFSPropertyConstants.DOCUMENT_HEADER + "."
                    + KFSPropertyConstants.WORKFLOW_DOCUMENT_TYPE_NAME, invoiceType);
        }

        final String upperBoundInvoiceDueDate = (String) lookupFormFields.get(
                ArPropertyConstants.CustomerInvoiceDocumentFields.INVOICE_DUE_DATE);
        final String lowerBoundInvoiceDueDate = (String) lookupFormFields.get(
                KRADConstants.LOOKUP_RANGE_LOWER_BOUND_PROPERTY_PREFIX
                        + ArPropertyConstants.CustomerInvoiceDocumentFields.INVOICE_DUE_DATE);
        final String invoiceDueDateCriteria = getContractsGrantsReportHelperService()
                .fixDateCriteria(lowerBoundInvoiceDueDate, upperBoundInvoiceDueDate, false);
        if (!StringUtils.isBlank(invoiceDueDateCriteria)) {
            lookupFields.put(ArPropertyConstants.CustomerInvoiceDocumentFields.INVOICE_DUE_DATE, invoiceDueDateCriteria);
        }

        final String proposalNumber = (String) lookupFormFields.get(KFSPropertyConstants.PROPOSAL_NUMBER);
        if (!StringUtils.isBlank(proposalNumber)) {
            lookupFields.put(ArPropertyConstants.ContractsGrantsInvoiceDocumentFields.PROPOSAL_NUMBER, proposalNumber);
        }

        final String documentNumber = (String) lookupFormFields.get(KFSPropertyConstants.DOCUMENT_NUMBER);
        if (!StringUtils.isBlank(documentNumber)) {
            lookupFields.put(KFSPropertyConstants.DOCUMENT_NUMBER, documentNumber);
        }

        return lookupFields;
    }

    public CustomerInvoiceDocumentService getCustomerInvoiceDocumentService() {
        return customerInvoiceDocumentService;
    }

    public void setCustomerInvoiceDocumentService(CustomerInvoiceDocumentService customerInvoiceDocumentService) {
        this.customerInvoiceDocumentService = customerInvoiceDocumentService;
    }

    public DateTimeService getDateTimeService() {
        return dateTimeService;
    }

    public void setDateTimeService(DateTimeService dateTimeService) {
        this.dateTimeService = dateTimeService;
    }

    public FinancialSystemDocumentService getFinancialSystemDocumentService() {
        return financialSystemDocumentService;
    }

    public void setFinancialSystemDocumentService(FinancialSystemDocumentService financialSystemDocumentService) {
        this.financialSystemDocumentService = financialSystemDocumentService;
    }

    public void setLookupDao(LookupDao lookupDao) {
        this.lookupDao = lookupDao;
    }
}
