/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2020 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document.validation;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.integration.ar.AccountsReceivableModuleBillingService;
import org.kuali.kfs.integration.cg.ContractsAndGrantsBillingAwardAccount;
import org.kuali.kfs.kns.document.MaintenanceDocument;
import org.kuali.kfs.kns.maintenance.rules.MaintenanceDocumentRuleBase;
import org.kuali.kfs.krad.bo.PersistableBusinessObject;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ar.ArConstants;
import org.kuali.kfs.module.ar.ArKeyConstants;
import org.kuali.kfs.module.ar.ArPropertyConstants;
import org.kuali.kfs.module.ar.businessobject.Milestone;
import org.kuali.kfs.module.ar.businessobject.MilestoneSchedule;
import org.kuali.kfs.module.ar.document.service.MilestoneScheduleMaintenanceService;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.rice.core.api.util.type.AbstractKualiDecimal;
import org.kuali.rice.core.api.util.type.KualiDecimal;

import java.sql.Date;
import java.time.LocalDate;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

public class MilestoneScheduleRule extends MaintenanceDocumentRuleBase {

    private static final Logger LOG = LogManager.getLogger(MilestoneScheduleRule.class);

    MilestoneSchedule newMilestoneScheduleCopy;

    private static volatile AccountsReceivableModuleBillingService accountsReceivableModuleBillingService;
    private static volatile MilestoneScheduleMaintenanceService milestoneScheduleMaintenanceService;

    @Override
    public boolean processCustomAddCollectionLineBusinessRules(MaintenanceDocument document, String collectionName,
            PersistableBusinessObject line) {
        LOG.debug("Entering MilestoneScheduleRule.processCustomAddCollectionLineBusinessRules");

        boolean isValid = checkForDuplicateMilestoneNumber(collectionName, line);
        isValid &= checkMilestoneActualCompletionDate(collectionName, line);

        LOG.debug("Leaving MilestoneScheduleRule.processCustomAddCollectionLineBusinessRules");

        return isValid;
    }

    /**
     * Check to see if a Milestone with the milestone bill number already exists.
     *
     * @param collectionName name of the collection being added to
     * @param line           PersistableBusinessObject being added to the collection
     * @return true if there isn't already a milestone with the same milestone number, false otherwise
     */
    private boolean checkForDuplicateMilestoneNumber(String collectionName, PersistableBusinessObject line) {
        boolean isValid = true;

        if (StringUtils.equalsIgnoreCase(collectionName, ArConstants.MILESTONES_SECTION)) {
            Milestone milestone = (Milestone) line;
            String newMilestoneNumber = milestone.getMilestoneNumber();

            for (Milestone existingMilestone : newMilestoneScheduleCopy.getMilestones()) {
                if (existingMilestone.getMilestoneNumber().equals(newMilestoneNumber)) {
                    isValid = false;
                    putFieldError(collectionName, ArKeyConstants.ERROR_DUPLICATE_MILESTONE_NUMBER);
                    break;
                }
            }
        }

        return isValid;
    }

    private boolean checkMilestoneActualCompletionDate(String collectionName, PersistableBusinessObject line) {
        if (StringUtils.equalsIgnoreCase(collectionName, ArConstants.MILESTONES_SECTION)) {
            Milestone milestone = (Milestone) line;
            Date milestoneActualCompletionDate = milestone.getMilestoneActualCompletionDate();
            if (ObjectUtils.isNotNull(milestoneActualCompletionDate)
                    && milestoneActualCompletionDate.after(Date.valueOf(LocalDate.now()))) {
                putFieldError(collectionName, ArKeyConstants.ERROR_MILESTONE_ACTUAL_COMPLETION_DATE_IN_FUTURE);
                return false;
            }
        }
        return true;
    }

    @Override
    protected boolean processCustomSaveDocumentBusinessRules(MaintenanceDocument document) {
        LOG.debug("Entering MilestoneScheduleRule.processCustomSaveDocumentBusinessRules");

        processCustomRouteDocumentBusinessRules(document);

        LOG.debug("Leaving MilestoneScheduleRule.processCustomSaveDocumentBusinessRules");

        // save despite error messages
        return true;
    }

    @Override
    protected boolean processCustomRouteDocumentBusinessRules(MaintenanceDocument document) {
        LOG.debug("Entering MilestoneScheduleRule.processCustomRouteDocumentBusinessRules");

        boolean success = checkAward();
        success &= checkMilestoneScheduleAccount();
        success &= checkAwardBillingFrequency();
        success &= checkForDuplicateMilestoneNumbers();
        success &= checkMilestoneTotalAgainstAwardTotal();
        success &= checkMilestoneActualCompletionDates();

        LOG.debug("Leaving MilestoneScheduleRule.processCustomRouteDocumentBusinessRules");

        return success;
    }

    private boolean checkAward() {
        if (ObjectUtils.isNull(newMilestoneScheduleCopy.getAward())
                || StringUtils.isBlank(newMilestoneScheduleCopy.getAward().getProposalNumber())) {
            putFieldError(ArPropertyConstants.PROPOSAL_NUMBER_FOR_AWARD_ACCOUNT_LOOKUP,
                    ArKeyConstants.ERROR_AWARD_DOES_NOT_EXIST, newMilestoneScheduleCopy.getProposalNumber());
            return false;
        }

        return true;
    }

    private boolean checkMilestoneScheduleAccount() {
        if (ObjectUtils.isNull(newMilestoneScheduleCopy.getAward())) {
            return true;
        }

        List<ContractsAndGrantsBillingAwardAccount> activeAwardAccounts = newMilestoneScheduleCopy.getAward()
                .getActiveAwardAccounts();
        for (ContractsAndGrantsBillingAwardAccount account : activeAwardAccounts) {
            if (StringUtils.equals(account.getChartOfAccountsCode(), newMilestoneScheduleCopy.getChartOfAccountsCode())
                && StringUtils.equals(account.getAccountNumber(), newMilestoneScheduleCopy.getAccountNumber())) {
                return true;
            }
        }
        putFieldError(KFSPropertyConstants.ACCOUNT_NUMBER,
                ArKeyConstants.ERROR_MILESTONE_SCHEDULE_ACCOUNT_DOES_NOT_EXIST_ON_AWARD);
        return false;
    }

    /**
     * checks to see if the billing frequency on the award is Milestone
     *
     * @return true if the billing frequency is milestone, false otherwise
     */
    private boolean checkAwardBillingFrequency() {
        boolean success = false;

        if (ObjectUtils.isNull(newMilestoneScheduleCopy.getAward())
                || StringUtils.isBlank(newMilestoneScheduleCopy.getProposalNumber())) {
            return true;
        }

        if (ObjectUtils.isNotNull(newMilestoneScheduleCopy.getAward().getBillingFrequencyCode())) {
            if (ArConstants.BillingFrequencyValues.isMilestone(newMilestoneScheduleCopy.getAward())) {
                success = true;
            }
        }

        if (!success) {
            putFieldError(ArPropertyConstants.PROPOSAL_NUMBER_FOR_AWARD_ACCOUNT_LOOKUP,
                    ArKeyConstants.ERROR_AWARD_MILESTONE_SCHEDULE_INCORRECT_BILLING_FREQUENCY,
                    new String[]{newMilestoneScheduleCopy.getProposalNumber()});
        }

        return success;
    }

    /**
     * Check to see if there is more than one Milestone with the same milestone number.
     *
     * @return true if there is more than one Milestone with the same milestone number, false otherwise
     */
    private boolean checkForDuplicateMilestoneNumbers() {
        boolean isValid = true;

        Set<String> milestoneNumbers = new HashSet<>();
        Set<String> duplicateMilestoneNumbers = new HashSet<>();

        for (Milestone milestone : newMilestoneScheduleCopy.getMilestones()) {
            if (!milestoneNumbers.add(milestone.getMilestoneNumber())) {
                duplicateMilestoneNumbers.add(milestone.getMilestoneNumber());
            }
        }

        if (duplicateMilestoneNumbers.size() > 0) {
            isValid = false;
            int lineNum = 0;
            for (Milestone milestone : newMilestoneScheduleCopy.getMilestones()) {
                // If the Milestone has already been copied to the Invoice, it will be readonly, the user won't have
                // been able to change it and thus we don't need to highlight it as an error if it's a dupe. There will
                // be another dupe in the list that we will highlight.
                boolean copiedToInvoice = false;
                if (ObjectUtils.isNotNull(milestone.getMilestoneIdentifier())) {
                    if (getMilestoneScheduleMaintenanceService()
                            .hasMilestoneBeenCopiedToInvoice(milestone.getProposalNumber(),
                                    milestone.getMilestoneIdentifier().toString())) {
                        copiedToInvoice = true;
                    }
                }
                if (!copiedToInvoice) {
                    if (duplicateMilestoneNumbers.contains(milestone.getMilestoneNumber())) {
                        String errorPath = ArPropertyConstants.MilestoneScheduleFields.MILESTONES + "[" + lineNum + "]."
                                + ArPropertyConstants.MilestoneFields.MILESTONE_NUMBER;
                        putFieldError(errorPath, ArKeyConstants.ERROR_DUPLICATE_MILESTONE_NUMBER);
                    }
                }
                lineNum++;
            }
        }
        return isValid;
    }

    /**
     * Check to see if the milestone total for this schedule and other schedules associated with this award
     * is less than the award total amount.
     * @return true if the milestone total is less than the award total, false otherwise
     */
    private boolean checkMilestoneTotalAgainstAwardTotal() {
        if (ObjectUtils.isNull(newMilestoneScheduleCopy.getAward())
                || ObjectUtils.isNull(newMilestoneScheduleCopy.getAward().getAwardTotalAmount())) {
            return true;
        }

        final KualiDecimal awardTotalAmount = newMilestoneScheduleCopy.getAward().getAwardTotalAmount();
        KualiDecimal milestoneTotalAmount = newMilestoneScheduleCopy.getMilestones().stream()
                .filter(Milestone::isActive)
                .filter(milestone -> ObjectUtils.isNotNull(milestone.getMilestoneAmount()))
                .reduce(KualiDecimal.ZERO, (sum, milestone) ->
                        milestone.getMilestoneAmount().add(sum), AbstractKualiDecimal::add);

        KualiDecimal totalAmountFromOtherMilestoneSchedules = getAccountsReceivableModuleBillingService()
                .getMilestonesTotalAmountForOtherSchedules(newMilestoneScheduleCopy.getProposalNumber(),
                        newMilestoneScheduleCopy.getChartOfAccountsCode(), newMilestoneScheduleCopy.getAccountNumber());

        final KualiDecimal totalForAllMilestones = milestoneTotalAmount.add(totalAmountFromOtherMilestoneSchedules);
        if (totalForAllMilestones.isGreaterThan(awardTotalAmount)) {
            putFieldError(ArPropertyConstants.ScheduleFields.TOTAL_SCHEDULED_ACCOUNT,
                    ArKeyConstants.ERROR_MILESTONE_TOTAL_EXCEEDS_AWARD_TOTAL);
            return false;
        } else {
            return true;
        }
    }

    private boolean checkMilestoneActualCompletionDates() {
        boolean isValid = true;

        int lineNum = 0;
        for (Milestone milestone : newMilestoneScheduleCopy.getMilestones()) {
            Date milestoneActualCompletionDate = milestone.getMilestoneActualCompletionDate();
            if (ObjectUtils.isNotNull(milestoneActualCompletionDate)
                    && milestoneActualCompletionDate.after(Date.valueOf(LocalDate.now()))) {
                String errorPath = ArPropertyConstants.MilestoneScheduleFields.MILESTONES + "[" + lineNum + "]."
                        + ArPropertyConstants.MilestoneFields.MILESTONE_ACTUAL_COMPLETION_DATE;
                putFieldError(errorPath, ArKeyConstants.ERROR_MILESTONE_ACTUAL_COMPLETION_DATE_IN_FUTURE);
                isValid = false;
            }
            lineNum++;
        }

        return isValid;
    }

    @Override
    public void setupConvenienceObjects() {
        newMilestoneScheduleCopy = (MilestoneSchedule) super.getNewBo();
    }

    AccountsReceivableModuleBillingService getAccountsReceivableModuleBillingService() {
        if (accountsReceivableModuleBillingService == null) {
            accountsReceivableModuleBillingService = SpringContext.getBean(AccountsReceivableModuleBillingService.class);
        }
        return accountsReceivableModuleBillingService;
    }

    MilestoneScheduleMaintenanceService getMilestoneScheduleMaintenanceService() {
        if (milestoneScheduleMaintenanceService == null) {
            milestoneScheduleMaintenanceService = SpringContext.getBean(MilestoneScheduleMaintenanceService.class);
        }
        return milestoneScheduleMaintenanceService;
    }
}
