/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.batch;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.module.ar.batch.vo.CustomerAddressVO;
import org.kuali.kfs.module.ar.batch.vo.CustomerCsvHeaders;
import org.kuali.kfs.module.ar.batch.vo.CustomerVO;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * CSVBuilder convert the parsed data values into CustomerVO list, which is validated by CustomerLoad service and docs
 * creation
 */
public final class CustomerLoadCSVBuilder {

    /**
     * Private Constructor since this is a util class that should never be instantiated.
     */
    private CustomerLoadCSVBuilder() {
    }

    /**
     * Convert the parseData object into CustomerVO
     *
     * @param parseDataList
     * @return
     */
    public static List<CustomerVO> buildCustomerDigestVO(final List<Map<String, String>> parseDataList) {
        final List<CustomerVO> customerVOs = new ArrayList<>();

        CustomerVO customer = null;
        CustomerVO dataCustomer;
        CustomerAddressVO dataAddress;
        for (final Map<String, String> rowDataMap : parseDataList) {
            dataCustomer = buildCustomerFromDataMap(rowDataMap);
            dataAddress = buildAddressFromDataMap(rowDataMap);

            // if it is not a continuing customer with alternate address or it is the first iteration
            if (!isContinuingCustomer(customer, dataCustomer) || customer == null) {
                if (customer != null) {
                    customerVOs.add(customer);
                }
                // instantiate customer as new customer
                customer = dataCustomer;
            }
            // update the address
            customer.addCustomerAddress(dataAddress);
        }
        // add the last customer
        customerVOs.add(customer);

        return customerVOs;
    }

    /**
     * Continuing customer from CSV is defined by customer name field is empty
     *
     * @param customer
     * @param dataCustomer
     * @return
     */
    private static boolean isContinuingCustomer(final CustomerVO customer, final CustomerVO dataCustomer) {
        return StringUtils.isEmpty(dataCustomer.getCustomerName());
    }

    /**
     * build the CustomerAddressVO from data row
     *
     * @param rowDataMap
     * @return
     */
    private static CustomerAddressVO buildAddressFromDataMap(final Map<String, String> rowDataMap) {

        final CustomerAddressVO address = new CustomerAddressVO();
        address.setCustomerAddressName(rowDataMap.get(CustomerCsvHeaders.customerAddressName.name()));
        address.setCustomerLine1StreetAddress(rowDataMap.get(CustomerCsvHeaders.customerLine1StreetAddress.name()));
        address.setCustomerLine2StreetAddress(rowDataMap.get(CustomerCsvHeaders.customerLine2StreetAddress.name()));
        address.setCustomerCityName(rowDataMap.get(CustomerCsvHeaders.customerCityName.name()));
        address.setCustomerStateCode(rowDataMap.get(CustomerCsvHeaders.customerStateCode.name()));
        address.setCustomerZipCode(rowDataMap.get(CustomerCsvHeaders.customerZipCode.name()));
        address.setCustomerCountryCode(rowDataMap.get(CustomerCsvHeaders.customerCountryCode.name()));
        address.setCustomerAddressInternationalProvinceName(rowDataMap.get(CustomerCsvHeaders.customerAddressInternationalProvinceName.name()));
        address.setCustomerInternationalMailCode(rowDataMap.get(CustomerCsvHeaders.customerInternationalMailCode.name()));
        address.setCustomerEmailAddress(rowDataMap.get(CustomerCsvHeaders.customerAddressEmail.name()));
        address.setCustomerAddressTypeCode(rowDataMap.get(CustomerCsvHeaders.customerAddressTypeCode.name()));

        return address;
    }

    /**
     * build the CustomerVO from the data row
     *
     * @param rowDataMap
     * @return
     */
    private static CustomerVO buildCustomerFromDataMap(final Map<String, String> rowDataMap) {

        final CustomerVO customer = new CustomerVO();
        customer.setCustomerNumber(rowDataMap.get(CustomerCsvHeaders.customerNumber.name()));
        customer.setCustomerName(rowDataMap.get(CustomerCsvHeaders.customerName.name()));
        customer.setCustomerParentCompanyNumber(rowDataMap.get(CustomerCsvHeaders.customerParentCompanyNumber.name()));
        customer.setCustomerTypeCode(rowDataMap.get(CustomerCsvHeaders.customerTypeCode.name()));
        customer.setCustomerLastActivityDate(rowDataMap.get(CustomerCsvHeaders.customerLastActivityDate.name()));
        customer.setCustomerTaxTypeCode(rowDataMap.get(CustomerCsvHeaders.customerTaxTypeCode.name()));
        customer.setCustomerTaxNbr(rowDataMap.get(CustomerCsvHeaders.customerTaxNbr.name()));
        customer.setCustomerActiveIndicator(rowDataMap.get(CustomerCsvHeaders.customerActiveIndicator.name()));
        customer.setCustomerPhoneNumber(rowDataMap.get(CustomerCsvHeaders.customerPhoneNumber.name()));
        customer.setCustomer800PhoneNumber(rowDataMap.get(CustomerCsvHeaders.customer800PhoneNumber.name()));
        customer.setCustomerContactName(rowDataMap.get(CustomerCsvHeaders.customerContactName.name()));
        customer.setCustomerContactPhoneNumber(rowDataMap.get(CustomerCsvHeaders.customerContactPhoneNumber.name()));
        customer.setCustomerFaxNumber(rowDataMap.get(CustomerCsvHeaders.customerFaxNumber.name()));
        customer.setCustomerBirthDate(rowDataMap.get(CustomerCsvHeaders.customerBirthDate.name()));
        customer.setCustomerTaxExemptIndicator(rowDataMap.get(CustomerCsvHeaders.customerTaxExemptIndicator.name()));
        customer.setCustomerCreditLimitAmount(rowDataMap.get(CustomerCsvHeaders.customerCreditLimitAmount.name()));
        customer.setCustomerCreditApprovedByName(rowDataMap.get(CustomerCsvHeaders.customerCreditApprovedByName.name()));
        customer.setCustomerEmailAddress(rowDataMap.get(CustomerCsvHeaders.customerEmailAddress.name()));
        return customer;
    }

}
