/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document.authorization;

import org.kuali.kfs.kim.api.KimConstants;
import org.kuali.kfs.kim.impl.identity.Person;
import org.kuali.kfs.krad.document.Document;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ar.document.ContractsGrantsInvoiceDocument;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.document.authorization.AccountingDocumentAuthorizerBase;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

public class ContractsGrantsInvoiceDocumentAuthorizer extends AccountingDocumentAuthorizerBase {

    /**
     * Overridden to pass in the proposal number for the CINV doc to help determine whether the current user
     * is a fund manager for this doc/award.
     */
    @Override
    protected void addRoleQualification(final Object businessObject, final Map<String, String> attributes) {
        super.addRoleQualification(businessObject, attributes);

        final ContractsGrantsInvoiceDocument contractsGrantsInvoiceDocument = (ContractsGrantsInvoiceDocument) businessObject;
        if (ObjectUtils.isNotNull(contractsGrantsInvoiceDocument.getInvoiceGeneralDetail())
                && ObjectUtils.isNotNull(contractsGrantsInvoiceDocument.getInvoiceGeneralDetail().getProposalNumber())) {
            attributes.put(KFSPropertyConstants.PROPOSAL_NUMBER,
                    contractsGrantsInvoiceDocument.getInvoiceGeneralDetail().getProposalNumber());
        }
    }

    /**
     * Overridden so that not just initiators can edit the document overview.
     */
    @Override
    public boolean canEditDocumentOverview(final Document document, final Person user) {
        return isAuthorizedByTemplate(document,
                KFSConstants.CoreModuleNamespaces.KFS,
                KimConstants.PermissionTemplateNames.EDIT_DOCUMENT,
                user.getPrincipalId());
    }

    /**
     * Overridden to pass in the current user's principal id as a qualifier to help determine if they are a Fund
     * Manager.
     */
    @Override
    public boolean canInitiate(final String documentTypeName, final Person user) {
        final String nameSpaceCode = KFSConstants.CoreModuleNamespaces.KFS;
        final Map<String, String> permissionDetails = new HashMap<>();
        final Map<String, String> qualificationDetails = new HashMap<>();
        qualificationDetails.put(KimConstants.AttributeConstants.PRINCIPAL_ID, user.getPrincipalId());
        permissionDetails.put(KimConstants.AttributeConstants.DOCUMENT_TYPE_NAME, documentTypeName);
        return getPermissionService().isAuthorizedByTemplate(user.getPrincipalId(), nameSpaceCode,
            KimConstants.PermissionTemplateNames.INITIATE_DOCUMENT, permissionDetails, qualificationDetails);
    }

    /**
     * Overriding to make sure KFSConstants.KFS_ACTION_CAN_ERROR_CORRECT permission is not removed by super class
     */
    @Override
    public Set<String> getDocumentActions(
            final Document document, final Person user,
            final Set<String> documentActionsFromPresentationController) {
        final boolean canErrorCorrect = documentActionsFromPresentationController.contains(
                KFSConstants.KFS_ACTION_CAN_ERROR_CORRECT);
        final Set<String> documentActionsAfterProcessing = super.getDocumentActions(document, user,
                documentActionsFromPresentationController);

        if (canErrorCorrect) {
            documentActionsAfterProcessing.add(KFSConstants.KFS_ACTION_CAN_ERROR_CORRECT);
        }

        return documentActionsAfterProcessing;
    }

}
