/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document.authorization;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.integration.cg.ContractsAndGrantsBillingAward;
import org.kuali.kfs.kew.api.KewApiConstants;
import org.kuali.kfs.kew.doctype.bo.DocumentType;
import org.kuali.kfs.kew.routeheader.DocumentRouteHeaderValue;
import org.kuali.kfs.kim.api.KimConstants;
import org.kuali.kfs.kim.impl.identity.Person;
import org.kuali.kfs.krad.service.DocumentService;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.krad.workflow.authorizer.CfDocumentTypeAuthorizer;
import org.kuali.kfs.module.ar.document.ContractsGrantsInvoiceDocument;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.context.SpringContext;

import java.util.HashMap;
import java.util.Map;

public class ContractsGrantsInvoiceWorkflowDocumentAuthorizer extends CfDocumentTypeAuthorizer {

    /**
     * Overridden to pass in the proposal number for the CINV doc to help determine whether the current user
     * is a fund manager for this doc/award.
     *
     * @see org.kuali.kfs.kns.document.authorization.DocumentAuthorizerBase#canInitiate(java.lang.String, Person)
     */
    @Override
    protected Map<String, String> buildDocumentRoleQualifiers(final DocumentRouteHeaderValue document, final String routeNodeName) {
        final Map<String, String> qualifiers = super.buildDocumentRoleQualifiers(document, routeNodeName);

        final String documentNumber = document.getDocumentId();
        if (StringUtils.isNotBlank(documentNumber)) {
            final ContractsGrantsInvoiceDocument contractsGrantsInvoiceDocument =
                    (ContractsGrantsInvoiceDocument) SpringContext.getBean(DocumentService.class)
                            .getByDocumentHeaderId(documentNumber);
            if (ObjectUtils.isNotNull(contractsGrantsInvoiceDocument)) {
                final ContractsAndGrantsBillingAward award = contractsGrantsInvoiceDocument.getInvoiceGeneralDetail().getAward();
                qualifiers.put(KFSPropertyConstants.PROPOSAL_NUMBER, award.getProposalNumber());
            }
        }

        return qualifiers;
    }

    /**
     * Overridden to pass in the current user's principal id as a qualifier to help determine if they are a Fund Manager.
     */
    @Override
    public boolean canInitiate(final String principalId, final DocumentType documentType) {
        validatePrincipalId(principalId);
        validateDocumentType(documentType);

        final Map<String, String> permissionDetails = buildDocumentTypePermissionDetails(documentType, null, null, null);
        final Map<String, String> qualificationDetails = new HashMap<>();
        qualificationDetails.put(KimConstants.AttributeConstants.PRINCIPAL_ID, principalId);

        if (useKimPermission(KFSConstants.CoreModuleNamespaces.KFS, KewApiConstants.INITIATE_PERMISSION, permissionDetails, true)) {
            return getPermissionService().isAuthorizedByTemplate(principalId, KFSConstants.CoreModuleNamespaces.KFS,
                KewApiConstants.INITIATE_PERMISSION, permissionDetails, qualificationDetails);
        }
        return true;
    }

}
