/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document.authorization;

import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.kns.document.MaintenanceDocument;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.module.ar.ArConstants;
import org.kuali.kfs.module.ar.ArParameterConstants;
import org.kuali.kfs.module.ar.ArPropertyConstants;
import org.kuali.kfs.module.ar.businessobject.OrganizationOptions;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.sys.document.authorization.FinancialSystemMaintenanceDocumentPresentationControllerBase;
import org.kuali.kfs.sys.service.impl.KfsParameterConstants;
import org.kuali.kfs.kim.api.role.RoleService;
import org.kuali.kfs.kim.api.services.KimApiServiceLocator;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

public class OrganizationOptionsPresentationController extends FinancialSystemMaintenanceDocumentPresentationControllerBase {

    protected static final String ACCOUNTS_RECEIVABLE_MANAGER_ROLE_NAME = "Accounts Receivable Manager";

    @Override
    public Set<String> getConditionallyReadOnlySectionIds(final MaintenanceDocument document) {
        final Set<String> readOnlySectionIds = super.getConditionallyReadOnlySectionIds(document);
        setRemitToAddressSectionEditable(readOnlySectionIds);
        return readOnlySectionIds;
    }

    @Override
    public Set<String> getConditionallyReadOnlyPropertyNames(final MaintenanceDocument document) {
        final Set<String> readOnlyPropertyNames = super.getConditionallyReadOnlyPropertyNames(document);
        setRemitToNameEditable(readOnlyPropertyNames);
        setOrgPostalZipCodeEditable(readOnlyPropertyNames);
        setBillingOrgFieldsEditable(readOnlyPropertyNames, document);
        setProcessingOrgFieldsEditable(readOnlyPropertyNames, document);
        return readOnlyPropertyNames;
    }

    /**
     * Billing Chart/Org are always read-only on an edit. Always.
     * <p>
     * They are editable on an Add, but only if KIM lets you in on an Add, but that's handled elsewhere.
     *
     * @param readOnlyPropertyNames
     * @param document
     */
    protected void setBillingOrgFieldsEditable(final Set<String> readOnlyPropertyNames, final MaintenanceDocument document) {
        if (document.isEdit()) {
            readOnlyPropertyNames.add(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE);
            readOnlyPropertyNames.add(KFSPropertyConstants.ORGANIZATION_CODE);
        }
    }

    /**
     * Sets the processing Chart/Org code editable
     *
     * @param readOnlyPropertyNames
     * @param document
     */
    protected void setProcessingOrgFieldsEditable(final Set<String> readOnlyPropertyNames, final MaintenanceDocument document) {
        if (document.isEdit()) {

            final RoleService rms = KimApiServiceLocator.getRoleService();

            final String principalId = GlobalVariables.getUserSession().getPrincipalId();

            final List<String> roleIds = new ArrayList<>();
            roleIds.add(rms.getRoleIdByNamespaceCodeAndName(KFSConstants.CoreModuleNamespaces.KFS, ACCOUNTS_RECEIVABLE_MANAGER_ROLE_NAME));

            // editable only for the AR Manager role
            if (!rms.principalHasRole(principalId, roleIds, null)) {
                readOnlyPropertyNames.add(KFSPropertyConstants.PROCESSING_CHART_OF_ACCT_CD);
                readOnlyPropertyNames.add(KFSPropertyConstants.PROCESSING_ORGANIZATION_CODE);
            }
        }
    }

    /**
     * Sets the Remit-To Name (ie, OrgCheckPayableToName) to read only if that's how the system parameters are
     * configured, otherwise leave it read/write.
     *
     * @param readOnlyPropertyNames
     */
    protected void setRemitToNameEditable(final Set<String> readOnlyPropertyNames) {
        final ParameterService parameterService = SpringContext.getBean(ParameterService.class);
        // defaulting to true to preserve prior behavior
        if (parameterService.getParameterValueAsBoolean(OrganizationOptions.class,
                ArParameterConstants.REMIT_TO_IND, Boolean.TRUE)) {
            readOnlyPropertyNames.add(ArPropertyConstants.OrganizationOptionsFields.ORGANIZATION_CHECK_PAYABLE_TO_NAME);
        }
    }

    /**
     * Sets the OrgPostalZipCode to readonly if that's what the system parameters say, otherwise leave it
     * read/write.
     *
     * @param readOnlyPropertyNames
     */
    protected void setOrgPostalZipCodeEditable(final Set<String> readOnlyPropertyNames) {
        final ParameterService service = SpringContext.getBean(ParameterService.class);
        if (!service.getParameterValueAsBoolean(KfsParameterConstants.ACCOUNTS_RECEIVABLE_DOCUMENT.class, ArParameterConstants.SALES_TAX_IND, Boolean.FALSE)) {
            readOnlyPropertyNames.add(ArPropertyConstants.OrganizationOptionsFields.ORGANIZATION_POSTAL_ZIP_CODE);
        }
    }

    /**
     * Sets the whole Remit-To Address section to read-only if that's what the system parameter says, otherwise leave
     * it read/write.
     *
     * @param readOnlySectionIds
     */
    protected void setRemitToAddressSectionEditable(final Set<String> readOnlySectionIds) {
        final ParameterService service = SpringContext.getBean(ParameterService.class);
        final String addressEditable = service.getParameterValueAsString(OrganizationOptions.class,
                ArParameterConstants.REMIT_TO_ADDRESS_IND);
        if ("N".equalsIgnoreCase(addressEditable)) {
            readOnlySectionIds.add(ArConstants.OrganizationOptionsSections.EDIT_ORGANIZATION_REMIT_TO_ADDRESS);
        }
    }

}
