/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document.service.impl;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.integration.cg.ContractsAndGrantsBillingAward;
import org.kuali.kfs.integration.cg.ContractsAndGrantsBillingAwardAccount;
import org.kuali.kfs.integration.cg.ContractsAndGrantsModuleBillingService;
import org.kuali.kfs.krad.service.KualiModuleService;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ar.ArConstants;
import org.kuali.kfs.module.ar.businessobject.AwardAccountObjectCodeTotalBilled;
import org.kuali.kfs.module.ar.businessobject.ContractsGrantsLetterOfCreditReviewDetail;
import org.kuali.kfs.module.ar.dataaccess.AwardAccountObjectCodeTotalBilledDao;
import org.kuali.kfs.module.ar.document.ContractsGrantsLetterOfCreditReviewDocument;
import org.kuali.kfs.module.ar.document.service.ContractsGrantsInvoiceDocumentService;
import org.kuali.kfs.module.ar.document.service.ContractsGrantsLetterOfCreditReviewDocumentService;
import org.kuali.kfs.module.ar.service.ContractsGrantsInvoiceCreateDocumentService;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.businessobject.SystemOptions;
import org.kuali.kfs.sys.service.OptionsService;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

@Transactional
public class ContractsGrantsLetterOfCreditReviewDocumentServiceImpl implements
        ContractsGrantsLetterOfCreditReviewDocumentService {

    protected AwardAccountObjectCodeTotalBilledDao awardAccountObjectCodeTotalBilledDao;
    protected ContractsGrantsInvoiceCreateDocumentService contractsGrantsInvoiceCreateDocumentService;
    protected ContractsGrantsInvoiceDocumentService contractsGrantsInvoiceDocumentService;
    protected ContractsAndGrantsModuleBillingService contractsAndGrantsModuleBillingService;
    protected KualiModuleService kualiModuleService;
    protected OptionsService optionsService;

    /**
     * Calculates the amount to draw for each award account and puts it in a map, keyed by chart-account
     */
    @Override
    public Map<String, KualiDecimal> calculateAwardAccountAmountsToDraw(
            final ContractsAndGrantsBillingAward award,
            final List<ContractsAndGrantsBillingAwardAccount> awardAccounts) {
        final Map<String, KualiDecimal> amounts = new HashMap<>();

        final List<AwardAccountObjectCodeTotalBilled> awardAccountTotalBilledAmounts =
                getAwardAccountObjectCodeTotalBilledDao()
                        .getAwardAccountObjectCodeTotalBuildByProposalNumberAndAccount(awardAccounts);
        final SystemOptions systemOptions = optionsService.getCurrentYearOptions();

        for (final ContractsAndGrantsBillingAwardAccount awardAccount : awardAccounts) {
            // 2. Get the Cumulative amount from GL Balances.
            final KualiDecimal cumAmt = getContractsGrantsInvoiceDocumentService().getBudgetAndActualsForAwardAccount(
                    awardAccount, systemOptions.getActualFinancialBalanceTypeCd());
            KualiDecimal billedAmount = KualiDecimal.ZERO;

            // 3. Amount to Draw = Cumulative amount - Billed to Date.(This would be ultimately the current
            // expenditures in the invoice document.
            for (final AwardAccountObjectCodeTotalBilled awardAccountTotalBilledAmount : awardAccountTotalBilledAmounts) {
                if (StringUtils.equals(awardAccountTotalBilledAmount.getAccountNumber(), awardAccount.getAccountNumber())
                        && StringUtils.equals(awardAccountTotalBilledAmount.getChartOfAccountsCode(),
                            awardAccount.getChartOfAccountsCode())
                        && awardAccountTotalBilledAmount.getProposalNumber().equals(awardAccount.getProposalNumber())) {
                    billedAmount = billedAmount.add(awardAccountTotalBilledAmount.getTotalBilled());
                }
            }
            final KualiDecimal amountToDraw = cumAmt.subtract(billedAmount);
            amounts.put(getAwardAccountKey(awardAccount), amountToDraw);
        }

        return amounts;
    }

    /**
     * Generates the key as chart of accounts code - account number
     */
    @Override
    public String getAwardAccountKey(final ContractsAndGrantsBillingAwardAccount awardAccount) {
        return awardAccount.getChartOfAccountsCode() + '-' + awardAccount.getAccountNumber();
    }

    /**
     * This method retrieves the amount available to draw for the award accounts
     *
     * @param awardTotalAmount
     * @param awardAccounts
     */
    @Override
    public KualiDecimal getAmountAvailableToDraw(
            final KualiDecimal awardTotalAmount,
            final List<ContractsAndGrantsBillingAwardAccount> awardAccounts) {
        // Get the billed to date amount for every award account based on the criteria passed.
        final List<AwardAccountObjectCodeTotalBilled> awardAccountTotalBilledAmounts =
                getAwardAccountObjectCodeTotalBilledDao()
                        .getAwardAccountObjectCodeTotalBuildByProposalNumberAndAccount(awardAccounts);
        KualiDecimal billedAmount = KualiDecimal.ZERO;
        for (final AwardAccountObjectCodeTotalBilled awardAccountTotalBilledAmount : awardAccountTotalBilledAmounts) {
            billedAmount = billedAmount.add(awardAccountTotalBilledAmount.getTotalBilled());
        }

        return awardTotalAmount.subtract(billedAmount);
    }

    @Override
    public void generateContractsGrantsInvoiceDocuments(final ContractsGrantsLetterOfCreditReviewDocument locReviewDoc) {
        final List<ContractsAndGrantsBillingAward> awards = new ArrayList<>();
        // 1. compare the hiddenamountodraw and amount to draw field.
        final Set<String> proposalNumberSet = new HashSet<>();
        for (final ContractsGrantsLetterOfCreditReviewDetail detail : locReviewDoc.getHeaderReviewDetails()) {
            if (!proposalNumberSet.contains(detail.getProposalNumber())) {
                awards.add(retrieveAward(detail.getProposalNumber()));
                proposalNumberSet.add(detail.getProposalNumber());
            }
        }

        // To set the loc Creation Type to award based on the LOC Review document being retrieved.
        final String locCreationType = ObjectUtils.isNotNull(locReviewDoc.getLetterOfCreditFundCode()) ?
                ArConstants.LOC_BY_LOC_FUND : ArConstants.LOC_BY_LOC_FUND_GRP;
        getContractsGrantsInvoiceCreateDocumentService().createCGInvoiceDocumentsByAwards(awards,
                locReviewDoc.getAccountReviewDetails(), locCreationType);

        //To route the invoices automatically as the initiator would be system user after a wait time.
        try {
            Thread.sleep(100);
        } catch (final InterruptedException e) {
            throw new RuntimeException(e);
        }

        getContractsGrantsInvoiceCreateDocumentService().routeContractsGrantsInvoiceDocuments();
    }

    /**
     * Retrieves an award based on proposal number
     *
     * @param proposalNumber the proposal number to look up the award for
     * @return the award
     */
    protected ContractsAndGrantsBillingAward retrieveAward(final String proposalNumber) {
        final Map<String, Object> map = new HashMap<>();
        map.put(KFSPropertyConstants.PROPOSAL_NUMBER, proposalNumber);
        return kualiModuleService.getResponsibleModuleService(ContractsAndGrantsBillingAward.class)
                .getExternalizableBusinessObject(ContractsAndGrantsBillingAward.class, map);
    }

    @Override
    public List<ContractsAndGrantsBillingAward> getActiveAwardsByCriteria(final Map<String, Object> criteria) {
        return getKualiModuleService().getResponsibleModuleService(ContractsAndGrantsBillingAward.class)
                .getExternalizableBusinessObjectsList(ContractsAndGrantsBillingAward.class, criteria);
    }

    public AwardAccountObjectCodeTotalBilledDao getAwardAccountObjectCodeTotalBilledDao() {
        return awardAccountObjectCodeTotalBilledDao;
    }

    public void setAwardAccountObjectCodeTotalBilledDao(
            final AwardAccountObjectCodeTotalBilledDao awardAccountObjectCodeTotalBilledDao) {
        this.awardAccountObjectCodeTotalBilledDao = awardAccountObjectCodeTotalBilledDao;
    }

    public ContractsGrantsInvoiceDocumentService getContractsGrantsInvoiceDocumentService() {
        return contractsGrantsInvoiceDocumentService;
    }

    public void setContractsGrantsInvoiceDocumentService(
            final ContractsGrantsInvoiceDocumentService contractsGrantsInvoiceDocumentService) {
        this.contractsGrantsInvoiceDocumentService = contractsGrantsInvoiceDocumentService;
    }

    public ContractsAndGrantsModuleBillingService getContractsAndGrantsModuleBillingService() {
        return contractsAndGrantsModuleBillingService;
    }

    public void setContractsAndGrantsModuleBillingService(
            final ContractsAndGrantsModuleBillingService contractsAndGrantsModuleBillingService) {
        this.contractsAndGrantsModuleBillingService = contractsAndGrantsModuleBillingService;
    }

    public KualiModuleService getKualiModuleService() {
        return kualiModuleService;
    }

    public void setKualiModuleService(final KualiModuleService kualiModuleService) {
        this.kualiModuleService = kualiModuleService;
    }

    public OptionsService getOptionsService() {
        return optionsService;
    }

    public void setOptionsService(final OptionsService optionsService) {
        this.optionsService = optionsService;
    }

    public ContractsGrantsInvoiceCreateDocumentService getContractsGrantsInvoiceCreateDocumentService() {
        return contractsGrantsInvoiceCreateDocumentService;
    }

    public void setContractsGrantsInvoiceCreateDocumentService(
            final ContractsGrantsInvoiceCreateDocumentService contractsGrantsInvoiceCreateDocumentService) {
        this.contractsGrantsInvoiceCreateDocumentService = contractsGrantsInvoiceCreateDocumentService;
    }

}
