/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document.service.impl;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.coa.businessobject.ObjectCode;
import org.kuali.kfs.coa.service.ObjectCodeService;
import org.kuali.kfs.core.api.datetime.DateTimeService;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.kew.api.WorkflowDocument;
import org.kuali.kfs.kew.api.WorkflowDocumentFactory;
import org.kuali.kfs.kim.impl.identity.Person;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.service.DocumentService;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ar.ArConstants;
import org.kuali.kfs.module.ar.ArParameterConstants;
import org.kuali.kfs.module.ar.ArPropertyConstants;
import org.kuali.kfs.module.ar.batch.service.CustomerInvoiceWriteoffBatchService;
import org.kuali.kfs.module.ar.batch.vo.CustomerInvoiceWriteoffBatchVO;
import org.kuali.kfs.module.ar.businessobject.AccountsReceivableDocumentHeader;
import org.kuali.kfs.module.ar.businessobject.CustomerInvoiceDetail;
import org.kuali.kfs.module.ar.businessobject.CustomerInvoiceWriteoffLookupResult;
import org.kuali.kfs.module.ar.businessobject.InvoicePaidApplied;
import org.kuali.kfs.module.ar.businessobject.OrganizationAccountingDefault;
import org.kuali.kfs.module.ar.businessobject.lookup.CustomerInvoiceWriteoffLookupUtil;
import org.kuali.kfs.module.ar.document.CustomerCreditMemoDocument;
import org.kuali.kfs.module.ar.document.CustomerInvoiceDocument;
import org.kuali.kfs.module.ar.document.CustomerInvoiceWriteoffDocument;
import org.kuali.kfs.module.ar.document.service.AccountsReceivableDocumentHeaderService;
import org.kuali.kfs.module.ar.document.service.CustomerInvoiceDocumentService;
import org.kuali.kfs.module.ar.document.service.CustomerInvoiceWriteoffDocumentService;
import org.kuali.kfs.module.ar.document.service.CustomerService;
import org.kuali.kfs.module.ar.document.service.InvoicePaidAppliedService;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.service.FinancialSystemUserService;
import org.kuali.kfs.sys.service.UniversityDateService;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Transactional
public class CustomerInvoiceWriteoffDocumentServiceImpl implements CustomerInvoiceWriteoffDocumentService {
    protected ParameterService parameterService;
    protected UniversityDateService universityDateService;
    protected BusinessObjectService businessObjectService;
    protected AccountsReceivableDocumentHeaderService accountsReceivableDocumentHeaderService;
    protected CustomerInvoiceDocumentService customerInvoiceDocumentService;
    protected CustomerService customerService;
    protected DocumentService documentService;
    protected CustomerInvoiceWriteoffBatchService invoiceWriteoffBatchService;
    protected DateTimeService dateTimeService;
    protected InvoicePaidAppliedService<CustomerInvoiceDetail> paidAppliedService;
    protected FinancialSystemUserService financialSystemUserService;
    protected ObjectCodeService objectCodeService;

    @Override
    public void completeWriteoffProcess(final CustomerInvoiceWriteoffDocument writeoff) {
        //  retrieve the document and make sure its not already closed, crash if so
        final String invoiceNumber = writeoff.getFinancialDocumentReferenceInvoiceNumber();
        final CustomerInvoiceDocument invoice = (CustomerInvoiceDocument) documentService.getByDocumentHeaderId(
                invoiceNumber);
        if (!invoice.isOpenInvoiceIndicator()) {
            throw new UnsupportedOperationException("The Invoice Writeoff Document #" + writeoff.getDocumentNumber() + " attempted to writeoff " +
                "an Invoice [#" + invoiceNumber + "] that was already closed.  This is not supported.");
        }

        Integer paidAppliedItemNumber = 0;
        KualiDecimal totalApplied = KualiDecimal.ZERO;

        //  retrieve the customer invoice details, and generate paid applieds for each
        final List<CustomerInvoiceDetail> invoiceDetails = invoice.getCustomerInvoiceDetailsWithoutDiscounts();
        for (final CustomerInvoiceDetail invoiceDetail : invoiceDetails) {

            //   if no open amount on the detail, then do nothing
            if (invoiceDetail.getAmountOpen().isZero()) {
                continue;
            }

            //  retrieve the number of current paid applieds, so we dont have item number overlap
            if (paidAppliedItemNumber == 0) {
                paidAppliedItemNumber = paidAppliedService.getNumberOfInvoicePaidAppliedsForInvoiceDetail(invoiceNumber,
                    invoiceDetail.getInvoiceItemNumber());
            }

            //  create and save the paidApplied
            final InvoicePaidApplied invoicePaidApplied = new InvoicePaidApplied();
            invoicePaidApplied.setDocumentNumber(writeoff.getDocumentNumber());
            invoicePaidApplied.setPaidAppliedItemNumber(paidAppliedItemNumber++);
            invoicePaidApplied.setFinancialDocumentReferenceInvoiceNumber(invoiceNumber);
            invoicePaidApplied.setInvoiceItemNumber(invoiceDetail.getInvoiceItemNumber());
            invoicePaidApplied.setUniversityFiscalYear(universityDateService.getCurrentFiscalYear());
            invoicePaidApplied.setUniversityFiscalPeriodCode(universityDateService.getCurrentUniversityDate().getUniversityFiscalAccountingPeriod());
            invoicePaidApplied.setInvoiceItemAppliedAmount(invoiceDetail.getAmountOpen());
            businessObjectService.save(invoicePaidApplied);

            //  record how much we're applying
            totalApplied = totalApplied.add(invoicePaidApplied.getInvoiceItemAppliedAmount());
        }

        //  close the document
        customerInvoiceDocumentService.addCloseNote(invoice, writeoff.getDocumentHeader().getWorkflowDocument());
        invoice.setOpenInvoiceIndicator(false);
        invoice.setClosedDate(dateTimeService.getCurrentSqlDate());
        documentService.updateDocument(invoice);

        //  set the final document total for the invoice
        writeoff.setInvoiceWriteoffAmount(totalApplied);
        writeoff.getDocumentHeader().setFinancialDocumentTotalAmount(totalApplied);
        documentService.updateDocument(writeoff);
    }

    @Override
    public void setupDefaultValuesForNewCustomerInvoiceWriteoffDocument(final CustomerInvoiceWriteoffDocument customerInvoiceWriteoffDocument) {
        // update status
        customerInvoiceWriteoffDocument.setStatusCode(ArConstants.CustomerInvoiceWriteoffStatuses.IN_PROCESS);

        // set accounts receivable document header
        final AccountsReceivableDocumentHeader accountsReceivableDocumentHeader = accountsReceivableDocumentHeaderService.getNewAccountsReceivableDocumentHeaderForCurrentUser();
        accountsReceivableDocumentHeader.setDocumentNumber(customerInvoiceWriteoffDocument.getDocumentNumber());
        accountsReceivableDocumentHeader.setCustomerNumber(customerInvoiceWriteoffDocument.getCustomerInvoiceDocument().getAccountsReceivableDocumentHeader().getCustomerNumber());
        customerInvoiceWriteoffDocument.setAccountsReceivableDocumentHeader(accountsReceivableDocumentHeader);

        // if writeoffs are generated based on organization accounting default, populate those fields now
        final String writeoffGenerationOption = parameterService.getParameterValueAsString(
                CustomerInvoiceWriteoffDocument.class, ArParameterConstants.WRITEOFF_METHOD);
        final boolean isUsingOrgAcctDefaultWriteoffFAU = ArConstants.WRITEOFF_METHOD_ORG_ACCT_DEFAULT
                .equals(writeoffGenerationOption);

        final String writeoffTaxGenerationOption = parameterService.getParameterValueAsString(
                CustomerInvoiceWriteoffDocument.class, ArParameterConstants.SALES_TAX_ADJUSTMENT_IND);
        final boolean isUsingTaxGenerationMethodDisallow = KFSConstants.ParameterValues.NO
                .equals(writeoffTaxGenerationOption);
        if (isUsingOrgAcctDefaultWriteoffFAU || isUsingTaxGenerationMethodDisallow) {
            final Integer currentUniversityFiscalYear = universityDateService.getCurrentFiscalYear();

            final Map<String, Object> criteria = new HashMap<>(3);
            criteria.put(KFSPropertyConstants.UNIVERSITY_FISCAL_YEAR, currentUniversityFiscalYear);
            criteria.put(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, customerInvoiceWriteoffDocument.getCustomerInvoiceDocument().getBillByChartOfAccountCode());
            criteria.put(KFSPropertyConstants.ORGANIZATION_CODE, customerInvoiceWriteoffDocument.getCustomerInvoiceDocument().getBilledByOrganizationCode());

            final OrganizationAccountingDefault organizationAccountingDefault = businessObjectService.findByPrimaryKey(OrganizationAccountingDefault.class, criteria);
            if (ObjectUtils.isNotNull(organizationAccountingDefault)) {
                customerInvoiceWriteoffDocument.setChartOfAccountsCode(organizationAccountingDefault.getWriteoffChartOfAccountsCode());
                customerInvoiceWriteoffDocument.setAccountNumber(organizationAccountingDefault.getWriteoffAccountNumber());
                customerInvoiceWriteoffDocument.setSubAccountNumber(organizationAccountingDefault.getWriteoffSubAccountNumber());
                customerInvoiceWriteoffDocument.setFinancialObjectCode(organizationAccountingDefault.getWriteoffFinancialObjectCode());
                customerInvoiceWriteoffDocument.setFinancialSubObjectCode(organizationAccountingDefault.getWriteoffFinancialSubObjectCode());
                customerInvoiceWriteoffDocument.setProjectCode(organizationAccountingDefault.getWriteoffProjectCode());
                customerInvoiceWriteoffDocument.setOrganizationReferenceIdentifier(organizationAccountingDefault.getWriteoffOrganizationReferenceIdentifier());
            }
        }
    }

    @Override
    public boolean isCustomerInvoiceWriteoffDocumentApproved(final String customerInvoiceWriteoffDocumentNumber) {
        final Map<String, Object> criteria = new HashMap<>();
        criteria.put(KFSPropertyConstants.DOCUMENT_NUMBER, customerInvoiceWriteoffDocumentNumber);
        criteria.put(ArPropertyConstants.DOCUMENT_STATUS_CODE, KFSConstants.DocumentStatusCodes.APPROVED);
        return businessObjectService.countMatching(CustomerInvoiceWriteoffDocument.class, criteria) == 1;
    }

    @Override
    public Collection<CustomerInvoiceWriteoffLookupResult> getCustomerInvoiceDocumentsForInvoiceWriteoffLookup(final Map<String, String> fieldValues) {
        //  only one of these four will be used, based on priority
        final String customerNumber = fieldValues.get(ArPropertyConstants.CustomerFields.CUSTOMER_NUMBER);
        final String customerName = fieldValues.get(ArPropertyConstants.CustomerInvoiceWriteoffLookupResultFields.CUSTOMER_NAME);
        final String customerTypeCode = fieldValues.get(ArPropertyConstants.CustomerInvoiceWriteoffLookupResultFields.CUSTOMER_TYPE_CODE);
        final String customerInvoiceNumber = fieldValues.get(ArPropertyConstants.CustomerInvoiceWriteoffLookupResultFields.CUSTOMER_INVOICE_NUMBER);

        //  this may be combined with any of the four above
        final String age = fieldValues.get(ArPropertyConstants.CustomerInvoiceWriteoffLookupResultFields.AGE);

        //  this is the priority order for searching if multiples are entered
        Collection<CustomerInvoiceDocument> customerInvoiceDocuments;
        if (StringUtils.isNotEmpty(customerInvoiceNumber)) {
            customerInvoiceDocuments = new ArrayList<>();
            final CustomerInvoiceDocument customerInvoiceDocument = customerInvoiceDocumentService.getInvoiceByInvoiceDocumentNumber(customerInvoiceNumber);
            if (ObjectUtils.isNotNull(customerInvoiceDocument) && customerInvoiceDocument.isOpenInvoiceIndicator()) {
                customerInvoiceDocuments.add(customerInvoiceDocument);
            }
        } else if (StringUtils.isNotEmpty(customerNumber)) {
            customerInvoiceDocuments = customerInvoiceDocumentService.getOpenInvoiceDocumentsByCustomerNumber(customerNumber);
        } else if (StringUtils.isNotEmpty(customerName) && StringUtils.isNotEmpty(customerTypeCode)) {
            customerInvoiceDocuments = customerInvoiceDocumentService.getOpenInvoiceDocumentsByCustomerNameByCustomerType(customerName, customerTypeCode);
        } else if (StringUtils.isNotEmpty(customerName)) {
            customerInvoiceDocuments = customerInvoiceDocumentService.getOpenInvoiceDocumentsByCustomerName(customerName);
        } else if (StringUtils.isNotEmpty(customerTypeCode)) {
            customerInvoiceDocuments = customerInvoiceDocumentService.getOpenInvoiceDocumentsByCustomerType(customerTypeCode);
        } else {
            customerInvoiceDocuments = customerInvoiceDocumentService.getAllOpenCustomerInvoiceDocumentsWithoutWorkflow();
        }

        // attach headers
        customerInvoiceDocuments = customerInvoiceDocumentService.attachWorkflowHeadersToTheInvoices(customerInvoiceDocuments);
        // filter invoices which have related CRMs and writeoffs in route.
        final Collection<CustomerInvoiceDocument> filteredCustomerInvoiceDocuments = filterInvoices(customerInvoiceDocuments);

        //  if no age value was specified, then we're done!
        if (StringUtils.isEmpty(age)) {
            return CustomerInvoiceWriteoffLookupUtil.getPopulatedCustomerInvoiceWriteoffLookupResults(filteredCustomerInvoiceDocuments);
        }

        // walk through what we have, and do any extra filtering based on age, if necessary
        boolean eligibleInvoiceFlag;
        final Collection<CustomerInvoiceDocument> eligibleInvoices = new ArrayList<>();
        for (final CustomerInvoiceDocument invoice : filteredCustomerInvoiceDocuments) {
            eligibleInvoiceFlag = true;

            if (ObjectUtils.isNotNull(invoice.getAge())) {
                eligibleInvoiceFlag &= new Integer(age).compareTo(invoice.getAge()) <= 0;
            } else {
                eligibleInvoiceFlag = false;
            }

            if (eligibleInvoiceFlag) {
                eligibleInvoices.add(invoice);
            }
        }

        return CustomerInvoiceWriteoffLookupUtil.getPopulatedCustomerInvoiceWriteoffLookupResults(eligibleInvoices);
    }

    @Override
    public Collection<CustomerInvoiceDocument> filterInvoices(final Collection<CustomerInvoiceDocument> customerInvoiceDocuments) {
        final Collection<CustomerInvoiceDocument> filteredInvoices = new ArrayList<>();
        boolean hasNoDocumentsInRouteFlag;

        for (final CustomerInvoiceDocument invoice : customerInvoiceDocuments) {
            if (invoice.getDocumentHeader().getWorkflowDocument().isFinal()) {
                hasNoDocumentsInRouteFlag = checkIfThereIsNoAnotherCRMInRouteForTheInvoice(invoice.getDocumentNumber());
                if (hasNoDocumentsInRouteFlag) {
                    hasNoDocumentsInRouteFlag = checkIfThereIsNoAnotherWriteoffInRouteForTheInvoice(invoice.getDocumentNumber());
                }
                if (hasNoDocumentsInRouteFlag) {
                    filteredInvoices.add(invoice);
                }
            }
        }
        return filteredInvoices;
    }

    /**
     * This method checks if there is another CRM in route for the invoice
     *
     * @param invoiceDocumentNumber document number to check
     * @return true if no other CRMs are in route for the invoice, false otherwise
     */
    @Override
    public boolean checkIfThereIsNoAnotherCRMInRouteForTheInvoice(final String invoiceDocumentNumber) {
        WorkflowDocument workflowDocument;
        boolean isSuccess = true;

        final Map<String, String> fieldValues = new HashMap<>();
        fieldValues.put(ArPropertyConstants.CustomerInvoiceDocumentFields.FINANCIAL_DOCUMENT_REF_INVOICE_NUMBER, invoiceDocumentNumber);

        final Collection<CustomerCreditMemoDocument> customerCreditMemoDocuments = businessObjectService.findMatching(CustomerCreditMemoDocument.class, fieldValues);

        // no CRMs associated with the invoice are found
        if (customerCreditMemoDocuments.isEmpty()) {
            return true;
        }

        final String userId = GlobalVariables.getUserSession().getPrincipalId();
        for (final CustomerCreditMemoDocument customerCreditMemoDocument : customerCreditMemoDocuments) {
            workflowDocument = WorkflowDocumentFactory.loadDocument(userId, customerCreditMemoDocument.getDocumentNumber());
            if (!(workflowDocument.isApproved() || workflowDocument.isCanceled() || workflowDocument.isDisapproved())) {
                isSuccess = false;
                break;
            }
        }
        return isSuccess;
    }

    @Override
    public boolean checkIfThereIsNoAnotherWriteoffInRouteForTheInvoice(final String invoiceDocumentNumber) {
        WorkflowDocument workflowDocument;
        boolean isSuccess = true;

        final Map<String, String> fieldValues = new HashMap<>();
        fieldValues.put(ArPropertyConstants.CustomerInvoiceDocumentFields.FINANCIAL_DOCUMENT_REF_INVOICE_NUMBER,
                invoiceDocumentNumber);

        final Collection<CustomerInvoiceWriteoffDocument> customerInvoiceWriteoffDocuments =
                businessObjectService.findMatching(CustomerInvoiceWriteoffDocument.class, fieldValues);

        // no writeoffs associated with the invoice are found
        if (customerInvoiceWriteoffDocuments.isEmpty()) {
            return true;
        }

        final String userId = GlobalVariables.getUserSession().getPrincipalId();
        for (final CustomerInvoiceWriteoffDocument customerInvoiceWriteoffDocument : customerInvoiceWriteoffDocuments) {
            workflowDocument = WorkflowDocumentFactory.loadDocument(userId,
                    customerInvoiceWriteoffDocument.getDocumentNumber());
            if (!(workflowDocument.isApproved() || workflowDocument.isCanceled() || workflowDocument.isDisapproved())) {
                isSuccess = false;
                break;
            }
        }
        return isSuccess;
    }

    @Override
    public String sendCustomerInvoiceWriteoffDocumentsToBatch(
            final Person person,
            final Collection<CustomerInvoiceWriteoffLookupResult> customerInvoiceWriteoffLookupResults) {
        final CustomerInvoiceWriteoffBatchVO batch = new CustomerInvoiceWriteoffBatchVO(person.getPrincipalName());

        //  add the date
        batch.setSubmittedOn(dateTimeService.getCurrentTimestamp().toString());

        //  add the customer note, if one was added
        String note = null;
        for (final CustomerInvoiceWriteoffLookupResult customerInvoiceWriteoffLookupResult :
                customerInvoiceWriteoffLookupResults) {
            note = customerInvoiceWriteoffLookupResult.getCustomerNote();
        }
        if (StringUtils.isNotBlank(note)) {
            batch.setNote(note);
        }

        //  add the document numbers
        for (final CustomerInvoiceWriteoffLookupResult customerInvoiceWriteoffLookupResult :
                customerInvoiceWriteoffLookupResults) {
            for (final CustomerInvoiceDocument customerInvoiceDocument :
                    customerInvoiceWriteoffLookupResult.getCustomerInvoiceDocuments()) {
                batch.addInvoiceNumber(customerInvoiceDocument.getDocumentNumber());
            }
        }

        // use the batch service to create the XML and drop it in the directory
        return invoiceWriteoffBatchService.createBatchDrop(person, batch);
    }

    @Override
    public String createCustomerInvoiceWriteoffDocument(final String invoiceNumber, String note) {
        //  create the new writeoff document
        final CustomerInvoiceWriteoffDocument document = (CustomerInvoiceWriteoffDocument) documentService.getNewDocument(CustomerInvoiceWriteoffDocument.class);

        //  setup the defaults and tie it to the Invoice document
        document.setFinancialDocumentReferenceInvoiceNumber(invoiceNumber);
        setupDefaultValuesForNewCustomerInvoiceWriteoffDocument(document);
        document.getDocumentHeader().setDocumentDescription(ArConstants.CUSTOMER_INVOICE_WRITEOFF_DOCUMENT_DESCRIPTION + " " + invoiceNumber + ".");

        document.setCustomerNote(note);

        //  satisfy silly > 10 chars explanation rule
        if (StringUtils.isBlank(note)) {
            note = "Document created by batch process.";
        } else if (note.length() <= 10) {
            note = "Document created by batch process.  " + note;
        }
        document.setCustomerNote(note);

        //  route the document
        documentService.routeDocument(document, "Routed by Customer Invoice Writeoff Document Batch Service", null);

        return document.getDocumentNumber();
    }

    @Override
    public Collection<CustomerInvoiceWriteoffDocument> getCustomerCreditMemoDocumentByInvoiceDocument(final String invoiceNumber) {
        final Map<String, String> fieldValues = new HashMap<>(1);
        fieldValues.put(ArPropertyConstants.CustomerInvoiceDocumentFields.FINANCIAL_DOCUMENT_REF_INVOICE_NUMBER, invoiceNumber);
        return businessObjectService.findMatching(CustomerInvoiceWriteoffDocument.class, fieldValues);
    }

    @Override
    public String getFinancialObjectCode(
            final CustomerInvoiceDetail postable, final CustomerInvoiceWriteoffDocument poster,
            final boolean isUsingOrgAcctDefaultWriteoffFAU, final boolean isUsingChartForWriteoff, final String chartOfAccountsCode) {
        if (isUsingOrgAcctDefaultWriteoffFAU) {
            return poster.getFinancialObjectCode();
        } else if (isUsingChartForWriteoff) {
            return parameterService.getSubParameterValueAsString(CustomerInvoiceWriteoffDocument.class,
                    ArParameterConstants.OBJECT_CODES, chartOfAccountsCode);
        } else {
            return postable.getAccountsReceivableObjectCode();
        }
    }

    @Override
    public ObjectCode getObjectCode(
            final CustomerInvoiceDetail postable, final CustomerInvoiceWriteoffDocument poster,
            final boolean isUsingOrgAcctDefaultWriteoffFAU, final boolean isUsingChartForWriteoff, final String chartOfAccountsCode) {
        return objectCodeService.getByPrimaryIdForCurrentYear(
            chartOfAccountsCode,
            getFinancialObjectCode(postable, poster, isUsingOrgAcctDefaultWriteoffFAU, isUsingChartForWriteoff,
                    chartOfAccountsCode));
    }

    public void setParameterService(final ParameterService parameterService) {
        this.parameterService = parameterService;
    }

    public void setUniversityDateService(final UniversityDateService universityDateService) {
        this.universityDateService = universityDateService;
    }

    public void setBusinessObjectService(final BusinessObjectService businessObjectService) {
        this.businessObjectService = businessObjectService;
    }

    public void setCustomerInvoiceDocumentService(final CustomerInvoiceDocumentService customerInvoiceDocumentService) {
        this.customerInvoiceDocumentService = customerInvoiceDocumentService;
    }

    public void setAccountsReceivableDocumentHeaderService(final AccountsReceivableDocumentHeaderService accountsReceivableDocumentHeaderService) {
        this.accountsReceivableDocumentHeaderService = accountsReceivableDocumentHeaderService;
    }

    public void setCustomerService(final CustomerService customerService) {
        this.customerService = customerService;
    }

    public void setDocumentService(final DocumentService documentService) {
        this.documentService = documentService;
    }

    public void setInvoiceWriteoffBatchService(final CustomerInvoiceWriteoffBatchService invoiceWriteoffBatchService) {
        this.invoiceWriteoffBatchService = invoiceWriteoffBatchService;
    }

    public void setDateTimeService(final DateTimeService dateTimeService) {
        this.dateTimeService = dateTimeService;
    }

    public void setPaidAppliedService(final InvoicePaidAppliedService<CustomerInvoiceDetail> paidAppliedService) {
        this.paidAppliedService = paidAppliedService;
    }

    public void setFinancialSystemUserService(final FinancialSystemUserService financialSystemUserService) {
        this.financialSystemUserService = financialSystemUserService;
    }

    public void setObjectCodeService(final ObjectCodeService objectCodeService) {
        this.objectCodeService = objectCodeService;
    }

}
