/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document.service.impl;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.core.api.datetime.DateTimeService;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.kew.api.WorkflowDocument;
import org.kuali.kfs.kew.api.WorkflowDocumentFactory;
import org.kuali.kfs.krad.document.Document;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.service.DocumentService;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ar.ArConstants;
import org.kuali.kfs.module.ar.businessobject.AccountsReceivableDocumentHeader;
import org.kuali.kfs.module.ar.businessobject.CustomerInvoiceDetail;
import org.kuali.kfs.module.ar.businessobject.CustomerOpenItemReportDetail;
import org.kuali.kfs.module.ar.businessobject.InvoicePaidApplied;
import org.kuali.kfs.module.ar.businessobject.NonAppliedHolding;
import org.kuali.kfs.module.ar.document.CustomerCreditMemoDocument;
import org.kuali.kfs.module.ar.document.CustomerInvoiceDocument;
import org.kuali.kfs.module.ar.document.CustomerInvoiceWriteoffDocument;
import org.kuali.kfs.module.ar.document.PaymentApplicationAdjustableDocument;
import org.kuali.kfs.module.ar.document.PaymentApplicationAdjustmentDocument;
import org.kuali.kfs.module.ar.document.PaymentApplicationDocument;
import org.kuali.kfs.module.ar.document.dataaccess.AccountsReceivableDocumentHeaderDao;
import org.kuali.kfs.module.ar.document.dataaccess.CustomerInvoiceDetailDao;
import org.kuali.kfs.module.ar.document.dataaccess.NonAppliedHoldingDao;
import org.kuali.kfs.module.ar.document.service.CustomerInvoiceDocumentService;
import org.kuali.kfs.module.ar.document.service.CustomerOpenItemReportService;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.businessobject.DocumentHeader;
import org.springframework.transaction.annotation.Transactional;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

@Transactional
public class CustomerOpenItemReportServiceImpl implements CustomerOpenItemReportService {

    private static final Logger LOG = LogManager.getLogger();

    protected AccountsReceivableDocumentHeaderDao accountsReceivableDocumentHeaderDao;
    protected CustomerInvoiceDocumentService customerInvoiceDocumentService;
    protected DocumentService documentService;
    protected DateTimeService dateTimeService;
    protected CustomerInvoiceDetailDao customerInvoiceDetailDao;
    protected NonAppliedHoldingDao nonAppliedHoldingDao;
    protected BusinessObjectService businessObjectService;

    @Override
    public List getPopulatedReportDetails(final String customerNumber) {
        final List results = new ArrayList();

        final Collection<String> documentNumbers = accountsReceivableDocumentHeaderDao
                .getARDocumentNumbersIncludingHiddenApplicationByCustomerNumber(customerNumber);
        if (documentNumbers.size() == 0) {
            return results;
        }

        final String userId = GlobalVariables.getUserSession().getPrincipalId();
        final List<String> docHeaderIds = new ArrayList<>();
        final List<String> invoiceIds = new ArrayList<>();
        final List<String> paymentApplicationIds = new ArrayList<>();
        final List<String> paymentApplicationAdjustmentIds = new ArrayList<>();
        final HashMap<String, CustomerOpenItemReportDetail> details = new HashMap<>();
        WorkflowDocument workflowDocument;

        for (final String docNumber : documentNumbers) {
            final CustomerOpenItemReportDetail detail = new CustomerOpenItemReportDetail();

            // populate workflow document
            workflowDocument = WorkflowDocumentFactory.loadDocument(userId, docNumber);

            // do not display not approved documents
            if (ObjectUtils.isNull(workflowDocument.getDateApproved())) {
                continue;
            }
            final Date approvedDate = workflowDocument.getDateApproved().withTimeAtStartOfDay().toDate();

            final String documentType = workflowDocument.getDocumentTypeName();
            detail.setDocumentType(documentType);

            detail.setDocumentNumber(docNumber);

            if (documentType.equals(ArConstants.ArDocumentTypeCodes.INV_DOCUMENT_TYPE)
                    || documentType.equals(ArConstants.ArDocumentTypeCodes.CONTRACTS_GRANTS_INVOICE)) {
                invoiceIds.add(docNumber);
            } else {
                // Approved Date -> for invoices Due Date, for all other documents Approved Date
                detail.setDueApprovedDate(approvedDate);

                if (documentType.equals(ArConstants.ArDocumentTypeCodes.PAYMENT_APPLICATION_DOCUMENT_TYPE_CODE)) {
                    paymentApplicationIds.add(docNumber);
                } else if (documentType.equals(ArConstants.ArDocumentTypeCodes.PAYMENT_APPLICATION_ADJUSTMENT_DOCUMENT_TYPE_CODE)) {
                    paymentApplicationAdjustmentIds.add(docNumber);
                } else {
                    docHeaderIds.add(docNumber);
                }
            }
            details.put(docNumber, detail);
        }

        // add Unapplied Payment Applications
        final Collection<NonAppliedHolding> arNonAppliedHoldings = nonAppliedHoldingDao.getNonAppliedHoldingsForCustomer(
                customerNumber);
        final List<String> payAppUnappliedHoldingIds = new ArrayList<>();
        final List<String> appaUnappliedHoldingIds = new ArrayList<>();
        for (final NonAppliedHolding nonAppliedHolding : arNonAppliedHoldings) {
            // populate workflow document
            workflowDocument = WorkflowDocumentFactory.loadDocument(userId,
                    nonAppliedHolding.getReferenceFinancialDocumentNumber());

            final String documentNumber = nonAppliedHolding.getReferenceFinancialDocumentNumber();
            final String documentType = nonAppliedHolding.getDocumentHeader().getWorkflowDocumentTypeName();

            final CustomerOpenItemReportDetail detail = new CustomerOpenItemReportDetail();
            detail.setDocumentType(documentType);
            detail.setDocumentNumber(documentNumber);

            if (ObjectUtils.isNull(workflowDocument.getDateApproved())) {
                continue;
            }

            final Date documentApprovedDate = workflowDocument.getDateApproved().withTimeAtStartOfDay().toDate();
            detail.setDueApprovedDate(documentApprovedDate);
            details.put(nonAppliedHolding.getReferenceFinancialDocumentNumber(), detail);

            // Don't add NonAppliedHoldings that we've already captured in the payapp or appa docs
            if (paymentApplicationIds.contains(documentNumber) ||
                    paymentApplicationAdjustmentIds.contains(documentNumber)) {
                continue;
            } else if (documentType.equals(ArConstants.ArDocumentTypeCodes.PAYMENT_APPLICATION_DOCUMENT_TYPE_CODE)) {
                payAppUnappliedHoldingIds.add(nonAppliedHolding.getReferenceFinancialDocumentNumber());
            } else if (documentType.equals(ArConstants.ArDocumentTypeCodes.PAYMENT_APPLICATION_ADJUSTMENT_DOCUMENT_TYPE_CODE)) {
                appaUnappliedHoldingIds.add(nonAppliedHolding.getReferenceFinancialDocumentNumber());
            }
        }

        // for invoices
        populateReportDetailsForInvoices(invoiceIds, results, details);

        // for payment applications
        populateReportDetailsForPaymentApplications(paymentApplicationIds, customerNumber, results, details);

        // for payment application adjustments
        populateReportDetailsForPaymentApplicationAdjustments(paymentApplicationAdjustmentIds, customerNumber, results,
                details);

        // for unapplied payment applications from PayApps
        populateReportDetailsForPaymentApplications(payAppUnappliedHoldingIds, customerNumber, results, details);

        // for unapplied payment applications from APPAs
        populateReportDetailsForPaymentApplicationAdjustments(appaUnappliedHoldingIds, customerNumber, results, details);

        // for all other documents
        populateReportDetails(docHeaderIds, results, details);

        return results;
    }

    /**
     * This method populates CustomerOpenItemReportDetails (Customer Open Item Report)
     *
     * @param urlParameters
     */
    @Override
    public List getPopulatedReportDetails(final Map urlParameters) {
        final List results = new ArrayList();

        // get arDocumentHeaders
        final Collection<AccountsReceivableDocumentHeader> arDocumentHeaders = getARDocumentHeaders(urlParameters);
        if (arDocumentHeaders.size() == 0) {
            return results;
        }

        // get ids of arDocumentHeaders
        final List<String> arDocumentHeaderIds = new ArrayList<>();
        for (final AccountsReceivableDocumentHeader arDocHeader : arDocumentHeaders) {
            arDocumentHeaderIds.add(arDocHeader.getDocumentNumber());
        }

        // get invoices
        final String reportOption = ((String[]) urlParameters.get(KFSConstants.CustomerOpenItemReport.REPORT_OPTION))[0];
        final Collection<CustomerInvoiceDocument> invoices;
        Collection<CustomerInvoiceDetail> details = null;
        if (StringUtils.equals(reportOption, ArConstants.CustomerAgingReportFields.ACCT)) {
            final String accountNumber = ((String[]) urlParameters.get(KFSConstants.CustomerOpenItemReport.ACCOUNT_NUMBER))[0];
            details = customerInvoiceDetailDao.getCustomerInvoiceDetailsByAccountNumberByInvoiceDocumentNumbers(
                    accountNumber, arDocumentHeaderIds);
            invoices = getInvoicesByAccountNumberByDocumentIds(accountNumber, arDocumentHeaderIds, details);
        } else {
            invoices = getDocuments(CustomerInvoiceDocument.class, arDocumentHeaderIds);
        }
        if (ObjectUtils.isNull(invoices) | invoices.size() == 0) {
            return results;
        }

        final List<CustomerInvoiceDocument> selectedInvoices = new ArrayList<>();

        final String columnTitle = ((String[]) urlParameters.get(KFSConstants.CustomerOpenItemReport.COLUMN_TITLE))[0];

        java.util.Date reportRunDate = null;
        java.util.Date beginDate = null;
        java.util.Date endDate = null;
        try {
            reportRunDate = dateTimeService.convertToDate(((String[]) urlParameters.get(
                    KFSConstants.CustomerOpenItemReport.REPORT_RUN_DATE))[0]);
            if (!StringUtils.equals(columnTitle, KFSConstants.CustomerOpenItemReport.ALL_DAYS)) {
                endDate = dateTimeService.convertToDate(((String[]) urlParameters.get(
                        KFSConstants.CustomerOpenItemReport.REPORT_END_DATE))[0]);
                final String strBeginDate = ((String[]) urlParameters.get(
                        KFSConstants.CustomerOpenItemReport.REPORT_BEGIN_DATE))[0];
                if (StringUtils.isNotEmpty(strBeginDate)) {
                    beginDate = dateTimeService.convertToDate(strBeginDate);
                }
            }
        } catch (final ParseException e) {
            LOG.error("problem during CustomerOpenItemReportServiceImpl.getPopulatedReportDetails()", e);
        }

        // Billing Organization
        if (StringUtils.equals(reportOption, ArConstants.ReportOptionFieldValues.BILLING_ORG)) {
            // All days
            // 1. invoice open amount > 0
            // 2. billingDate <= reportRunDate
            // 3. billByChartOfAccountsCode = processingOrBillingChartCode
            // 4. billbyOrganizationCode = orgCode
            final String chartCode = ((String[]) urlParameters.get(
                    KFSConstants.CustomerOpenItemReport.PROCESSING_OR_BILLING_CHART_CODE))[0];
            final String orgCode = ((String[]) urlParameters.get(KFSConstants.CustomerOpenItemReport.ORGANIZATION_CODE))[0];
            if (StringUtils.equals(columnTitle, KFSConstants.CustomerOpenItemReport.ALL_DAYS)) {
                for (final CustomerInvoiceDocument invoice : invoices) {
                    // get only invoices with open amounts
                    if (ObjectUtils.isNull(invoice.getClosedDate()) && ObjectUtils.isNotNull(invoice.getBillingDate())
                            && !reportRunDate.before(invoice.getBillingDate())
                            && StringUtils.equals(chartCode, invoice.getBillByChartOfAccountCode())
                            && StringUtils.equals(orgCode, invoice.getBilledByOrganizationCode())) {
                        selectedInvoices.add(invoice);
                    }
                }
            } else {
                // *days
                // 1. invoice open amount > 0
                // 2. beginDate <= invoice billing date <= endDate
                // 3. billByChartOfAccountsCode = chartCode
                // 4. billbyOrganizationCode = orgCode
                for (final CustomerInvoiceDocument invoice : invoices) {
                    if (ObjectUtils.isNull(invoice.getClosedDate())
                            && ObjectUtils.isNotNull(invoice.getBillingDate())
                            && StringUtils.equals(chartCode, invoice.getBillByChartOfAccountCode())
                            && StringUtils.equals(orgCode, invoice.getBilledByOrganizationCode())) {
                        if (ObjectUtils.isNotNull(beginDate)
                            && !beginDate.after(invoice.getBillingDate())
                            && !endDate.before(invoice.getBillingDate())
                            || ObjectUtils.isNull(beginDate)
                            && !endDate.before(invoice.getBillingDate())) {
                            selectedInvoices.add(invoice);
                        }
                    }
                }
            }
        } else {
            // Processing Organization or Account
            // All days
            // 1. invoice open amount > 0
            // 2. invoice billing dates <= reportRunDate
            if (StringUtils.equals(columnTitle, KFSConstants.CustomerOpenItemReport.ALL_DAYS)) {
                for (final CustomerInvoiceDocument invoice : invoices) {
                    if (ObjectUtils.isNull(invoice.getClosedDate()) && ObjectUtils.isNotNull(invoice.getBillingDate())
                            && !reportRunDate.before(invoice.getBillingDate())) {
                        selectedInvoices.add(invoice);
                    }
                }
            } else {
                // *days
                // 1. invoice open amount > 0
                // 2. beginDate <= invoice billing date <= endDate
                for (final CustomerInvoiceDocument invoice : invoices) {
                    if (ObjectUtils.isNull(invoice.getClosedDate())
                        && ObjectUtils.isNotNull(invoice.getBillingDate())
                        && ObjectUtils.isNotNull(beginDate) && !beginDate.after(invoice.getBillingDate())
                        && !endDate.before(invoice.getBillingDate())
                        || ObjectUtils.isNull(beginDate) && !endDate.before(invoice.getBillingDate())) {
                        selectedInvoices.add(invoice);
                    }
                }
            }
        }

        if (selectedInvoices.size() == 0) {
            return results;
        }

        if (StringUtils.equals(reportOption, ArConstants.CustomerAgingReportFields.ACCT)) {
            populateReporDetails(selectedInvoices, results, details);
        } else {
            populateReportDetails(selectedInvoices, results);
        }

        return results;
    }

    /**
     * This method populates CustomerOpenItemReportDetails for CustomerInvoiceDocuments (Customer History Report).
     *
     * @param invoiceIds
     * @param results CustomerOpenItemReportDetails to display in the report
     * @param details <key = documentNumber, value = customerOpenItemReportDetail>
     */
    protected void populateReportDetailsForInvoices(
            final List<String> invoiceIds,
            final List<CustomerOpenItemReportDetail> results,
            final HashMap<String, CustomerOpenItemReportDetail> details
    ) {
        if (invoiceIds.isEmpty()) {
            return;
        }

        final List<CustomerInvoiceDocument> invoices =
                (List<CustomerInvoiceDocument>) getDocuments(CustomerInvoiceDocument.class, invoiceIds);
        for (final CustomerInvoiceDocument invoice: invoices) {
            final String documentNumber = invoice.getDocumentNumber();
            final CustomerOpenItemReportDetail detail = details.get(documentNumber);

            final String documentDescription = invoice.getDocumentHeader().getDocumentDescription();
            detail.setDocumentDescription(StringUtils.defaultIfEmpty(documentDescription, ""));
            detail.setBillingDate(invoice.getBillingDate());

            // Due/Approved Date -> for invoice it is Due Date, and for all other documents Approved Date
            detail.setDueApprovedDate(invoice.getInvoiceDueDate());

            detail.setDocumentPaymentAmount(
                    invoice.getDocumentHeader().getFinancialDocumentTotalAmount());

            detail.setUnpaidUnappliedAmount(
                    customerInvoiceDocumentService.getOpenAmountForCustomerInvoiceDocument(invoice));

            results.add(detail);
        }
    }

    /**
     * This method populates CustomerOpenItemReportDetails for PaymentApplicationAdjustmentDocuments (Customer History Report).
     *
     * @param paymentApplicationAdjustmentIds documentNumbers of PaymentApplicationAdjustmentDocuments
     * @param results                         CustomerOpenItemReportDetails to display in the report
     * @param details                         <key = documentNumber, value = customerOpenItemReportDetail>
     */
    protected void populateReportDetailsForPaymentApplicationAdjustments(
            final List<String> paymentApplicationAdjustmentIds,
            final String customerNumber,
            final List results,
            final HashMap<String, CustomerOpenItemReportDetail> details
    ) {
        if (paymentApplicationAdjustmentIds.isEmpty()) {
            return;
        }

        final List<PaymentApplicationAdjustmentDocument> appaDocuments =
                (List<PaymentApplicationAdjustmentDocument>)
                        getDocuments(PaymentApplicationAdjustmentDocument.class, paymentApplicationAdjustmentIds);

        appaDocuments.forEach(appaDoc -> {
            final CustomerOpenItemReportDetail detail = details.get(appaDoc.getDocumentNumber());

            final String documentDescription = appaDoc.getDocumentHeader().getDocumentDescription();
            detail.setDocumentDescription(StringUtils.defaultIfEmpty(documentDescription, ""));

            final KualiDecimal nonAppliedAmountForCustomer = appaDoc.getNonAppliedHoldings().stream()
                    .filter(nonAppliedHolding -> nonAppliedHolding.getCustomerNumber().equals(customerNumber))
                    .map(NonAppliedHolding::getAvailableUnappliedAmount)
                    .reduce(KualiDecimal.ZERO, KualiDecimal::add);
            detail.setUnpaidUnappliedAmount(nonAppliedAmountForCustomer);

            final KualiDecimal documentPaymentAmount = appaDoc.getInvoicePaidApplieds().stream()
                    .filter(invoicePaidApplied -> invoicePaidApplied.getCustomerInvoiceDocument().getCustomer().getCustomerNumber().equals(customerNumber))
                    .map(InvoicePaidApplied::getInvoiceItemAppliedAmount)
                    .reduce(nonAppliedAmountForCustomer, KualiDecimal::add);
            detail.setDocumentPaymentAmount(documentPaymentAmount);

            results.add(detail);
        });
    }

    /**
     * This method populates CustomerOpenItemReportDetails for PaymentApplicationDocuments (Customer History Report).
     *
     * @param paymentApplicationIds documentNumbers of PaymentApplicationDocuments
     * @param results               CustomerOpenItemReportDetails to display in the report
     * @param details               <key = documentNumber, value = customerOpenItemReportDetail>
     */
    protected void populateReportDetailsForPaymentApplications(
            final List<String> paymentApplicationIds,
            final String customerNumber,
            final List<CustomerOpenItemReportDetail> results,
            final HashMap<String, CustomerOpenItemReportDetail> details
    ) {
        if (paymentApplicationIds.isEmpty()) {
            return;
        }

        final List<PaymentApplicationDocument> paymentApplications =
                (List<PaymentApplicationDocument>) getDocuments(PaymentApplicationDocument.class, paymentApplicationIds);

        for (final PaymentApplicationDocument paymentApplication : paymentApplications) {
            final String documentNumber = paymentApplication.getDocumentNumber();
            final CustomerOpenItemReportDetail detail = details.get(documentNumber);

            // populate Document Description
            final String documentDescription = paymentApplication.getDocumentHeader().getDocumentDescription();
            detail.setDocumentDescription(StringUtils.defaultIfEmpty(documentDescription, ""));

            // populate Document Payment Amount
            detail.setDocumentPaymentAmount(paymentApplication.getDocumentHeader().getFinancialDocumentTotalAmount());

            // populate Unpaid/Unapplied Amount if the customer number is the same
            KualiDecimal unAppliedAmount = KualiDecimal.ZERO;
            if (ObjectUtils.isNotNull(paymentApplication.getNonAppliedHolding())) {
                if (paymentApplication.getNonAppliedHolding().getCustomerNumber().equals(customerNumber)) {
                    unAppliedAmount = paymentApplication.getNonAppliedHolding().getAvailableUnappliedAmount();
                }
            }
            detail.setUnpaidUnappliedAmount(unAppliedAmount);
            results.add(detail);
        }
    }

    /**
     * This method populates CustomerOpenItemReportDetails for CustomerCreditMemoDocuments and WriteOffDocuments <=> all documents
     * but CustomerInvoiceDocument and PaymentApplicationDocument (Customer History Report).
     *
     * @param docHeaderIds documentNumbers of DocumentHeaders
     * @param results            CustomerOpenItemReportDetails to display in the report
     * @param details            <key = documentNumber, value = customerOpenItemReportDetail>
     */
    public void populateReportDetails(final List<String> docHeaderIds, final List results, final HashMap details) {
        if (docHeaderIds.isEmpty()) {
            return;
        }

        final Collection<DocumentHeader> financialSystemDocHeaders = new ArrayList<>();
        for (final String documentNumber : docHeaderIds) {
            final DocumentHeader header = businessObjectService.findBySinglePrimaryKey(
                    DocumentHeader.class, documentNumber);
            if (header != null) {
                financialSystemDocHeaders.add(header);
            }
        }

        for (final DocumentHeader fsDocumentHeader : financialSystemDocHeaders) {
            final CustomerOpenItemReportDetail detail = (CustomerOpenItemReportDetail) details.get(fsDocumentHeader.getDocumentNumber());

            // populate Document Description
            detail.setDocumentDescription(StringUtils.trimToEmpty(fsDocumentHeader.getDocumentDescription()));

            // populate Document Payment Amount
            detail.setDocumentPaymentAmount(fsDocumentHeader.getFinancialDocumentTotalAmount());

            // Unpaid/Unapplied Amount
            detail.setUnpaidUnappliedAmount(KualiDecimal.ZERO);

            results.add(detail);
        }
    }

    /**
     * @param invoices
     * @param results
     */
    protected void populateReportDetails(final List<CustomerInvoiceDocument> invoices, final List results) {
        for (final CustomerInvoiceDocument invoice : invoices) {
            final CustomerOpenItemReportDetail detail = new CustomerOpenItemReportDetail();
            detail.setDocumentType(invoice.getDocumentHeader().getWorkflowDocument().getDocumentTypeName());
            detail.setDocumentNumber(invoice.getDocumentNumber());
            final String documentDescription = invoice.getDocumentHeader().getDocumentDescription();
            if (ObjectUtils.isNotNull(documentDescription)) {
                detail.setDocumentDescription(documentDescription);
            } else {
                detail.setDocumentDescription("");
            }
            detail.setBillingDate(invoice.getBillingDate());
            detail.setDueApprovedDate(invoice.getInvoiceDueDate());
            detail.setDocumentPaymentAmount(invoice.getDocumentHeader()
                    .getFinancialDocumentTotalAmount());
            detail.setUnpaidUnappliedAmount(
                    customerInvoiceDocumentService.getOpenAmountForCustomerInvoiceDocument(invoice));
            results.add(detail);
        }
    }

    private Document getDocumentWithId(final String docId) {
        return documentService.getByDocumentHeaderId(docId);
    }

    /**
     * This method returns collection of documents of type classToSearchFrom Note: can be used for documents only, not for
     * *DocumentHeaders @param documentNumbers
     */
    public Collection getDocuments(final Class classToSearchFrom, final List documentNumbers) {
        return documentService.getDocumentsByListOfDocumentHeaderIds(classToSearchFrom, documentNumbers);
    }

    /**
     * This method retrieves ARDocumentHeader objects for "Customer Open Item Report"
     *
     * @param urlParameters
     * @return ARDocumentHeader objects meeting the search criteria
     */
    protected Collection getARDocumentHeaders(final Map urlParameters) {
        final Collection arDocumentHeaders;

        final String reportOption = ((String[]) urlParameters.get(KFSConstants.CustomerOpenItemReport.REPORT_OPTION))[0];
        final String customerNumber = ((String[]) urlParameters.get(KFSConstants.CustomerOpenItemReport.CUSTOMER_NUMBER))[0];

        if (StringUtils.equals(reportOption, ArConstants.ReportOptionFieldValues.PROCESSING_ORG)) {
            final String processingChartCode = ((String[]) urlParameters.get(KFSConstants.CustomerOpenItemReport.PROCESSING_OR_BILLING_CHART_CODE))[0];
            final String processingOrganizationCode = ((String[]) urlParameters.get(KFSConstants.CustomerOpenItemReport.ORGANIZATION_CODE))[0];
            arDocumentHeaders = accountsReceivableDocumentHeaderDao.getARDocumentHeadersByCustomerNumberByProcessingOrgCodeAndChartCode(customerNumber, processingChartCode, processingOrganizationCode);
        } else {
            // reportOption is "Billing Organization" or "Account"
            arDocumentHeaders = accountsReceivableDocumentHeaderDao.getARDocumentHeadersIncludingHiddenApplicationByCustomerNumber(customerNumber);
        }
        return arDocumentHeaders;
    }

    /**
     * This method gets called only if reportOption is Account Gets invoices based on the selected invoice details <=> invoice
     * details meeting search criteria i.e. the accountNumber and the list of documentIds
     *
     * @param accountNumber
     * @param arDocumentHeaderIds
     * @param details             (will get populated here)
     * @return invoices
     */
    protected Collection<CustomerInvoiceDocument> getInvoicesByAccountNumberByDocumentIds(
            final String accountNumber,
            final List arDocumentHeaderIds, final Collection<CustomerInvoiceDetail> details) {
        Collection<CustomerInvoiceDocument> invoices = null;

        if (ObjectUtils.isNull(details) | details.size() == 0) {
            return invoices;
        }

        // get list of invoice document ids (eliminate duplicate invoice document ids)
        final List<String> documentIds = new ArrayList<>();
        for (final CustomerInvoiceDetail detail : details) {
            final String documentNumber = detail.getDocumentNumber();
            if (!documentIds.contains(documentNumber)) {
                documentIds.add(documentNumber);
            }
        }

        // get invoices for the document ids list
        if (documentIds.size() != 0) {
            invoices = getDocuments(CustomerInvoiceDocument.class, documentIds);
        }

        return invoices;
    }

    /**
     * @param selectedInvoices
     * @param results
     * @param invoiceDetails
     */
    protected void populateReporDetails(
            final List<CustomerInvoiceDocument> selectedInvoices, final List results,
            final Collection<CustomerInvoiceDetail> invoiceDetails) {
        for (final CustomerInvoiceDocument invoice : selectedInvoices) {
            final String documentNumber = invoice.getDocumentNumber();

            KualiDecimal amount = KualiDecimal.ZERO;
            KualiDecimal taxAmount = KualiDecimal.ZERO;
            KualiDecimal openAmount = KualiDecimal.ZERO;

            boolean foundFlag = false;

            for (final CustomerInvoiceDetail invoiceDetail : invoiceDetails) {
                final String tempDocumentNumber = invoiceDetail.getDocumentNumber();
                if (!StringUtils.equals(documentNumber, tempDocumentNumber)) {
                    continue;
                }
                foundFlag = true;

                final KualiDecimal itemAmount = invoiceDetail.getAmount();
                if (ObjectUtils.isNotNull(itemAmount)) {
                    amount = amount.add(itemAmount);
                }

                final KualiDecimal itemTaxAmount = invoiceDetail.getInvoiceItemTaxAmount();
                if (ObjectUtils.isNotNull(itemTaxAmount)) {
                    taxAmount = taxAmount.add(itemTaxAmount);
                }

                final KualiDecimal openItemAmount = invoiceDetail.getAmountOpen();
                if (ObjectUtils.isNotNull(openItemAmount)) {
                    openAmount = openAmount.add(openItemAmount);
                }
            }
            // How is this possible?
            // invoiceDetails are for the list of invoices(invoices) meeting seach criteria including accountNumber
            // and selected arDocumentHeaders
            // -> list of invoices gets modified based on report run date and chosen date bucket -> selectedInvoices
            // selectedInvoices.size() <= invoices.size()
            if (!foundFlag) {
                continue;
            }

            final CustomerOpenItemReportDetail detail = new CustomerOpenItemReportDetail();
            detail.setDocumentType(invoice.getDocumentHeader().getWorkflowDocument().getDocumentTypeName());
            detail.setDocumentNumber(documentNumber);
            final String documentDescription = invoice.getDocumentHeader().getDocumentDescription();
            if (ObjectUtils.isNotNull(documentDescription)) {
                detail.setDocumentDescription(documentDescription);
            } else {
                detail.setDocumentDescription("");
            }
            // Billing Date
            detail.setBillingDate(invoice.getBillingDate());
            // Due Date
            detail.setDueApprovedDate(invoice.getInvoiceDueDate());
            // Document Payment Amount
            detail.setDocumentPaymentAmount(amount.add(taxAmount));
            // Unpaid/Unapplied Amount
            detail.setUnpaidUnappliedAmount(openAmount);
            results.add(detail);
        }
    }

    @Override
    public Collection<String> getDocumentNumbersOfReferenceReports(final String customerNumber) {
        final Collection<String> documentNumbers = new ArrayList<>();

        final Collection<AccountsReceivableDocumentHeader> arDocumentHeaders = accountsReceivableDocumentHeaderDao
                .getARDocumentHeadersIncludingHiddenApplicationByCustomerNumber(customerNumber);
        final String userId = GlobalVariables.getUserSession().getPrincipalId();

        final List<String> paymentApplicationAdjustableIds = new ArrayList();
        final List<String> creditMemoIds = new ArrayList();
        final List<String> writeOffIds = new ArrayList();
        WorkflowDocument workflowDocument;

        for (final AccountsReceivableDocumentHeader documentHeader: arDocumentHeaders) {

            // populate workflow document
            workflowDocument = WorkflowDocumentFactory.loadDocument(userId, documentHeader.getDocumentNumber());

            // do not display not approved documents
            if (ObjectUtils.isNull(workflowDocument.getDateApproved())) {
                continue;
            }

            final String documentType = workflowDocument.getDocumentTypeName();
            final String documentNumber = documentHeader.getDocumentNumber();

            switch (documentType) {
                case ArConstants.ArDocumentTypeCodes.PAYMENT_APPLICATION_DOCUMENT_TYPE_CODE:
                case ArConstants.ArDocumentTypeCodes.PAYMENT_APPLICATION_ADJUSTMENT_DOCUMENT_TYPE_CODE:
                    paymentApplicationAdjustableIds.add(documentNumber);
                    break;
                case ArConstants.ArDocumentTypeCodes.CUSTOMER_CREDIT_MEMO_DOCUMENT_TYPE_CODE:
                    creditMemoIds.add(documentNumber);
                    break;
                case ArConstants.ArDocumentTypeCodes.INVOICE_WRITEOFF_DOCUMENT_TYPE_CODE:
                    writeOffIds.add(documentNumber);
                    break;
                default:
                    break;
            }
        }

        paymentApplicationAdjustableIds.stream()
            .map(this::getDocumentWithId)
            .filter(PaymentApplicationAdjustableDocument.class::isInstance)
                .map(PaymentApplicationAdjustableDocument.class::cast)
                .map(PaymentApplicationAdjustableDocument::getInvoicePaidApplieds)
                .flatMap(Collection::stream)
                .map(InvoicePaidApplied::getFinancialDocumentReferenceInvoiceNumber)
                .forEach(documentNumbers::add);

        if (creditMemoIds.size() > 0) {
            final Collection<CustomerCreditMemoDocument> creditMemoDetailDocs =
                    getDocuments(CustomerCreditMemoDocument.class, creditMemoIds);
            for (final CustomerCreditMemoDocument creditMemo : creditMemoDetailDocs) {
                documentNumbers.add(creditMemo.getFinancialDocumentReferenceInvoiceNumber());
            }
        }

        if (writeOffIds.size() > 0) {
            final Collection<CustomerInvoiceWriteoffDocument> customerInvoiceWriteoffDocs =
                    getDocuments(CustomerInvoiceWriteoffDocument.class, writeOffIds);
            for (final CustomerInvoiceWriteoffDocument invoiceWriteoffDocument : customerInvoiceWriteoffDocs) {
                documentNumbers.add(invoiceWriteoffDocument.getFinancialDocumentReferenceInvoiceNumber());
            }
        }

        return documentNumbers;
    }

    @Override
    public List getPopulatedUnpaidUnappliedAmountReportDetails(final String customerNumber, final String refDocumentNumber) {
        final List results = new ArrayList();

        final Collection<AccountsReceivableDocumentHeader> arDocumentHeaders = accountsReceivableDocumentHeaderDao
                .getARDocumentHeadersIncludingHiddenApplicationByCustomerNumber(customerNumber);
        final String userId = GlobalVariables.getUserSession().getPrincipalId();

        final HashMap details = new HashMap();
        final List paymentApplicationIds = new ArrayList();
        final List creditMemoIds = new ArrayList();
        final List writeOffIds = new ArrayList();

        WorkflowDocument workflowDocument;

        for (final AccountsReceivableDocumentHeader documentHeader : arDocumentHeaders) {
            final CustomerOpenItemReportDetail detail = new CustomerOpenItemReportDetail();

            // populate workflow document
            workflowDocument = WorkflowDocumentFactory.loadDocument(userId, documentHeader.getDocumentNumber());

            // do not display not approved documents
            if (ObjectUtils.isNull(workflowDocument.getDateApproved())) {
                continue;
            }
            final Date approvedDate = workflowDocument.getDateApproved().withTimeAtStartOfDay().toDate();

            final String documentType = workflowDocument.getDocumentTypeName();
            detail.setDocumentType(documentType);

            final String documentNumber = documentHeader.getDocumentNumber();
            detail.setDocumentNumber(documentNumber);

            // Approved Date -> for invoices Due Date, for all other documents Approved Date
            detail.setDueApprovedDate(approvedDate);

            switch (documentType) {
                case ArConstants.ArDocumentTypeCodes.PAYMENT_APPLICATION_DOCUMENT_TYPE_CODE:
                case ArConstants.ArDocumentTypeCodes.PAYMENT_APPLICATION_ADJUSTMENT_DOCUMENT_TYPE_CODE:
                    paymentApplicationIds.add(documentNumber);
                    break;
                case ArConstants.ArDocumentTypeCodes.CUSTOMER_CREDIT_MEMO_DOCUMENT_TYPE_CODE:
                    creditMemoIds.add(documentNumber);
                    break;
                case ArConstants.ArDocumentTypeCodes.INVOICE_WRITEOFF_DOCUMENT_TYPE_CODE:
                    writeOffIds.add(documentNumber);
                    break;
                default:
                    break;
            }

            details.put(documentNumber, detail);
        }

        populateReportDetailsForUnpaidUnappliedPaymentApplications(paymentApplicationIds, results, details,
                customerNumber);

        if (creditMemoIds.size() > 0) {
            populateReportDetailsForCreditMemo(creditMemoIds, results, details, refDocumentNumber);
        }

        if (writeOffIds.size() > 0) {
            populateReportDetailsForWriteOff(writeOffIds, results, details, refDocumentNumber);
        }

        return results;
    }

    protected void populateReportDetailsForUnpaidUnappliedPaymentApplications(
            final List<String> paymentApplicationIds,
            final List results,
            final HashMap details,
            final String customerNumber
    ) {
        paymentApplicationIds.stream()
                .map(this::getDocumentWithId)
                .filter(document -> document instanceof PaymentApplicationAdjustableDocument)
                .map(PaymentApplicationAdjustableDocument.class::cast)
                .forEach(paymentApplicationAdjustableDocument -> {
                    final CustomerOpenItemReportDetail detail = (CustomerOpenItemReportDetail)
                            details.get(paymentApplicationAdjustableDocument.getDocumentNumber());
                    final String documentDescription = paymentApplicationAdjustableDocument
                            .getDocumentHeader().getDocumentDescription();

                    detail.setDocumentDescription(StringUtils.defaultIfEmpty(documentDescription, ""));
                    detail.setDocumentPaymentAmount(paymentApplicationAdjustableDocument
                            .getDocumentHeader().getFinancialDocumentTotalAmount());

                    final Optional<NonAppliedHolding> nonAppliedHoldingForCustomer = paymentApplicationAdjustableDocument
                            .getNonAppliedHoldings().stream()
                            .filter(nonAppliedHolding -> nonAppliedHolding.getCustomerNumber().equals(customerNumber))
                            .findAny();

                    KualiDecimal unappliedAmount = KualiDecimal.ZERO;
                    if (nonAppliedHoldingForCustomer.isPresent()) {
                        unappliedAmount = nonAppliedHoldingForCustomer.get().getFinancialDocumentLineAmount();
                    }
                    detail.setUnpaidUnappliedAmount(unappliedAmount);
                    results.add(detail);
                }
            );
    }

    protected void populateReportDetailsForCreditMemo(
            final List creditMemoIds, final List results, final HashMap details,
            final String refDocumentNumber) {
        final Collection<CustomerCreditMemoDocument> creditMemoDetailDocs = getDocuments(CustomerCreditMemoDocument.class, creditMemoIds);
        for (final CustomerCreditMemoDocument creditMemo : creditMemoDetailDocs) {
            if (creditMemo.getFinancialDocumentReferenceInvoiceNumber().equals(refDocumentNumber)) {
                final String documentNumber = creditMemo.getDocumentNumber();

                final CustomerOpenItemReportDetail detail = (CustomerOpenItemReportDetail) details.get(documentNumber);

                // populate Document Description
                final String documentDescription = creditMemo.getDocumentHeader().getDocumentDescription();
                if (ObjectUtils.isNotNull(documentDescription)) {
                    detail.setDocumentDescription(documentDescription);
                } else {
                    detail.setDocumentDescription("");
                }

                // populate Document Payment Amount
                detail.setDocumentPaymentAmount(creditMemo.getDocumentHeader().getFinancialDocumentTotalAmount());

                // Unpaid/Unapplied Amount
                detail.setUnpaidUnappliedAmount(KualiDecimal.ZERO);
                results.add(detail);
            }
        }
    }

    protected void populateReportDetailsForWriteOff(
            final List writeOffIds, final List results, final HashMap details,
            final String refDocumentNumber) {
        final Collection<CustomerInvoiceWriteoffDocument> customerInvoiceWriteoffDocs = getDocuments(CustomerInvoiceWriteoffDocument.class, writeOffIds);
        for (final CustomerInvoiceWriteoffDocument invoiceWriteoffDocument : customerInvoiceWriteoffDocs) {
            if (invoiceWriteoffDocument.getFinancialDocumentReferenceInvoiceNumber().equals(refDocumentNumber)) {
                final String documentNumber = invoiceWriteoffDocument.getDocumentNumber();

                final CustomerOpenItemReportDetail detail = (CustomerOpenItemReportDetail) details.get(documentNumber);

                // populate Document Description
                final String documentDescription = invoiceWriteoffDocument.getDocumentHeader().getDocumentDescription();
                if (ObjectUtils.isNotNull(documentDescription)) {
                    detail.setDocumentDescription(documentDescription);
                } else {
                    detail.setDocumentDescription("");
                }

                // populate Document Payment Amount
                detail.setDocumentPaymentAmount(invoiceWriteoffDocument.getDocumentHeader().getFinancialDocumentTotalAmount());

                // Unpaid/Unapplied Amount
                detail.setUnpaidUnappliedAmount(KualiDecimal.ZERO);
                results.add(detail);
            }
        }
    }

    public void setAccountsReceivableDocumentHeaderDao(final AccountsReceivableDocumentHeaderDao accountsReceivableDocumentHeaderDao) {
        this.accountsReceivableDocumentHeaderDao = accountsReceivableDocumentHeaderDao;
    }

    public void setCustomerInvoiceDocumentService(final CustomerInvoiceDocumentService customerInvoiceDocumentService) {
        this.customerInvoiceDocumentService = customerInvoiceDocumentService;
    }

    public void setDocumentService(final DocumentService documentService) {
        this.documentService = documentService;
    }

    public void setDateTimeService(final DateTimeService dateTimeService) {
        this.dateTimeService = dateTimeService;
    }

    public void setCustomerInvoiceDetailDao(final CustomerInvoiceDetailDao customerInvoiceDetailDao) {
        this.customerInvoiceDetailDao = customerInvoiceDetailDao;
    }

    public void setNonAppliedHoldingDao(final NonAppliedHoldingDao nonAppliedHoldingDao) {
        this.nonAppliedHoldingDao = nonAppliedHoldingDao;
    }

    public void setBusinessObjectService(final BusinessObjectService businessObjectService) {
        this.businessObjectService = businessObjectService;
    }
}
