/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document.validation.impl;

import org.kuali.kfs.core.api.config.property.ConfigurationService;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.kim.api.identity.PersonService;
import org.kuali.kfs.kim.impl.identity.Person;
import org.kuali.kfs.kns.rules.PromptBeforeValidationBase;
import org.kuali.kfs.krad.bo.Note;
import org.kuali.kfs.krad.document.Document;
import org.kuali.kfs.krad.service.NoteService;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ar.ArConstants;
import org.kuali.kfs.module.ar.ArKeyConstants;
import org.kuali.kfs.module.ar.businessobject.ContractsGrantsLetterOfCreditReviewDetail;
import org.kuali.kfs.module.ar.document.ContractsGrantsLetterOfCreditReviewDocument;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.context.SpringContext;

import java.text.MessageFormat;

public class ContractsGrantsLetterOfCreditReviewDocumentPreRules extends PromptBeforeValidationBase {

    private ConfigurationService configurationService;
    private PersonService personService;
    private NoteService noteService;

    @Override
    public boolean doPrompts(final Document document) {
        return conditionallyAskQuestion(document);
    }

    private boolean conditionallyAskQuestion(final Document document) {
        final ContractsGrantsLetterOfCreditReviewDocument letterOfCreditReviewDocument =
                (ContractsGrantsLetterOfCreditReviewDocument) document;

        for (final ContractsGrantsLetterOfCreditReviewDetail detail :
                letterOfCreditReviewDocument.getAccountReviewDetails()) {
            // If the amount to draw on the Letter of Credit Review document exceeds the maximum, check with the user to
            // confirm they want to continue to submit the document and have the subsequent CINV doc created.
            if (amountToDrawExceedsMaximum(detail)) {
                final String questionText = getConfigurationService().getPropertyValueAsString(
                        ArKeyConstants.WARNING_LOC_REVIEW_DOCUMENT_AMOUNT_TO_DRAW_EXCEEDS_MAXIMUM);
                final boolean confirm = askOrAnalyzeYesNoQuestion(
                        ArConstants.LETTER_OF_CREDIT_REVIEW_AMOUNT_TO_DRAW_QUESTION_ID,
                        questionText
                );
                if (confirm) {
                    createAndSaveNote(detail.getProposalNumber(), detail.getAccountNumber(),
                            letterOfCreditReviewDocument);
                } else {
                    abortRulesCheck();
                    return false;
                }
            }
        }
        return true;
    }

    private static boolean amountToDrawExceedsMaximum(final ContractsGrantsLetterOfCreditReviewDetail detail) {
        final KualiDecimal fundsNotDrawn = detail.getHiddenAmountToDraw().subtract(detail.getAmountToDraw());
        return fundsNotDrawn.isNegative();
    }

    private void createAndSaveNote(
            final String proposalNumber,
            final String accountNumber,
            final ContractsGrantsLetterOfCreditReviewDocument letterOfCreditReviewDocument
    ) {
        Note note = new Note();
        final String noteTemplate = getConfigurationService()
                .getPropertyValueAsString(ArKeyConstants.LOC_REVIEW_DOCUMENT_AMOUNT_TO_DRAW_EXCEEDED_NOTE);
        final Person initiator = GlobalVariables.getUserSession().getPerson();
        note.setNoteText(MessageFormat.format(noteTemplate, proposalNumber, accountNumber, initiator.getName()));
        note.setNotePostedTimestampToCurrent();
        note.setNoteTypeCode(KFSConstants.NoteTypeEnum.BUSINESS_OBJECT_NOTE_TYPE.getCode());
        final Person systemUser = getPersonService().getPersonByPrincipalName(KFSConstants.SYSTEM_USER);
        note = getNoteService().createNote(note, letterOfCreditReviewDocument.getNoteTarget(),
                systemUser.getPrincipalId());
        getNoteService().save(note);
    }

    private ConfigurationService getConfigurationService() {
        if (ObjectUtils.isNull(configurationService)) {
            configurationService = SpringContext.getBean(ConfigurationService.class);
        }

        return configurationService;
    }

    private NoteService getNoteService() {
        if (ObjectUtils.isNull(noteService)) {
            noteService = SpringContext.getBean(NoteService.class);
        }

        return noteService;
    }

    private PersonService getPersonService() {
        if (ObjectUtils.isNull(personService)) {
            personService = SpringContext.getBean(PersonService.class);
        }

        return personService;
    }
}
