/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document.validation.impl;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.kns.document.MaintenanceDocument;
import org.kuali.kfs.kns.rules.PromptBeforeValidationBase;
import org.kuali.kfs.krad.document.Document;
import org.kuali.kfs.krad.exception.UnknownDocumentIdException;
import org.kuali.kfs.krad.service.DocumentService;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ar.businessobject.InvoiceRecurrence;
import org.kuali.kfs.module.ar.document.CustomerInvoiceDocument;
import org.kuali.kfs.module.ar.document.CustomerInvoiceRecurrenceUtils;
import org.kuali.kfs.sys.context.SpringContext;

import java.sql.Date;
import java.time.LocalDate;

public class InvoiceRecurrencePreRules extends PromptBeforeValidationBase {

    private static final Logger LOG = LogManager.getLogger();

    @Override
    public boolean doPrompts(final Document document) {
        boolean preRulesOK = setCustomerNumberIfInvoiceIsEntered(document);
        preRulesOK &= setEndDateIfTotalRecurrenceNumberIsEntered(document);
        preRulesOK &= setTotalRecurrenceNumberIfEndDateIsEntered(document);
        return preRulesOK;
    }

    protected boolean setCustomerNumberIfInvoiceIsEntered(final Document document) {
        final MaintenanceDocument maintenanceDocument = (MaintenanceDocument) document;
        final InvoiceRecurrence newInvoiceRecurrence =
                (InvoiceRecurrence) maintenanceDocument.getNewMaintainableObject().getBusinessObject();

        if (ObjectUtils.isNull(newInvoiceRecurrence.getInvoiceNumber()) ||
            ObjectUtils.isNotNull(newInvoiceRecurrence.getCustomerNumber())) {
            return true;
        }

        try {
            if (SpringContext.getBean(DocumentService.class).documentExists(newInvoiceRecurrence.getInvoiceNumber())) {
                final CustomerInvoiceDocument customerInvoiceDocument =
                        (CustomerInvoiceDocument) SpringContext.getBean(DocumentService.class)
                                .getByDocumentHeaderId(newInvoiceRecurrence.getInvoiceNumber());
                newInvoiceRecurrence.setCustomerNumber(customerInvoiceDocument.getCustomer().getCustomerNumber());
            }
        } catch (final UnknownDocumentIdException ex) {
            LOG.error("Document {} does not exist.", newInvoiceRecurrence::getInvoiceNumber);
        }

        return true;
    }

    /**
     * This method checks if there is another customer with the same name and generates yes/no question
     *
     * @param document the maintenance document
     * @return
     */
    protected boolean setEndDateIfTotalRecurrenceNumberIsEntered(final Document document) {
        final MaintenanceDocument maintenanceDocument = (MaintenanceDocument) document;
        final InvoiceRecurrence newInvoiceRecurrence =
                (InvoiceRecurrence) maintenanceDocument.getNewMaintainableObject().getBusinessObject();

        if (ObjectUtils.isNull(newInvoiceRecurrence.getDocumentTotalRecurrenceNumber())) {
            return true;
        }
        if (ObjectUtils.isNotNull(newInvoiceRecurrence.getDocumentRecurrenceEndDate())
                && ObjectUtils.isNotNull(newInvoiceRecurrence.getDocumentTotalRecurrenceNumber())) {
            return true;
        }

        final LocalDate beginDate = newInvoiceRecurrence.getDocumentRecurrenceBeginDate().toLocalDate();
        final Integer documentTotalRecurrenceNumber = newInvoiceRecurrence.getDocumentTotalRecurrenceNumber();
        final String intervalCode = newInvoiceRecurrence.getDocumentRecurrenceIntervalCode();
        final LocalDate endDate = CustomerInvoiceRecurrenceUtils.calculateEndDateByTotalRecurrences(beginDate, intervalCode,
                documentTotalRecurrenceNumber);
        newInvoiceRecurrence.setDocumentRecurrenceEndDate(Date.valueOf(endDate));

        return true;
    }

    /**
     * This method calculates the total number of recurrences when a begin date and end date is entered.
     *
     * @param document the maintenance document
     * @return
     */
    protected boolean setTotalRecurrenceNumberIfEndDateIsEntered(final Document document) {
        final MaintenanceDocument maintenanceDocument = (MaintenanceDocument) document;
        final InvoiceRecurrence newInvoiceRecurrence =
                (InvoiceRecurrence) maintenanceDocument.getNewMaintainableObject().getBusinessObject();

        if (ObjectUtils.isNull(newInvoiceRecurrence.getDocumentRecurrenceEndDate())) {
            return true;
        }
        if (ObjectUtils.isNotNull(newInvoiceRecurrence.getDocumentRecurrenceEndDate())
                && ObjectUtils.isNotNull(newInvoiceRecurrence.getDocumentTotalRecurrenceNumber())) {
            return true;
        }

        final LocalDate beginDate = newInvoiceRecurrence.getDocumentRecurrenceBeginDate().toLocalDate();
        final LocalDate endDate = newInvoiceRecurrence.getDocumentRecurrenceEndDate().toLocalDate();
        final String intervalCode = newInvoiceRecurrence.getDocumentRecurrenceIntervalCode();
        final int totalRecurrences = CustomerInvoiceRecurrenceUtils.calculateTotalRecurrences(beginDate, endDate,
                intervalCode);

        if (totalRecurrences > 0) {
            newInvoiceRecurrence.setDocumentTotalRecurrenceNumber(totalRecurrences);
        }
        return true;
    }

}
