/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2019 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cam.document.validation.impl;

import org.kuali.kfs.kns.service.DataDictionaryService;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.module.cam.CamsKeyConstants;
import org.kuali.kfs.module.cam.CamsPropertyConstants;
import org.kuali.kfs.module.cam.businessobject.AssetPaymentDetail;
import org.kuali.kfs.module.cam.document.AssetPaymentDocument;
import org.kuali.kfs.module.cam.document.service.AssetPaymentService;
import org.kuali.kfs.sys.KFSKeyConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.businessobject.AccountingLine;
import org.kuali.kfs.sys.businessobject.UniversityDate;
import org.kuali.kfs.sys.document.validation.GenericValidation;
import org.kuali.kfs.sys.document.validation.event.AttributedDocumentEvent;
import org.kuali.kfs.sys.service.UniversityDateService;
import org.kuali.rice.core.api.datetime.DateTimeService;

import java.sql.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * This class validates asset payment posting date
 */
public class AssetPaymentPostingDateValidation extends GenericValidation {

    private AccountingLine accountingLineForValidation;
    private DateTimeService dateTimeService;
    private BusinessObjectService businessObjectService;
    private DataDictionaryService dataDictionaryService;
    private UniversityDateService universityDateService;
    private AssetPaymentService assetPaymentService;


    /**
     * Validates asset payment posting date
     */
    public boolean validate(AttributedDocumentEvent event) {
        // skip check if accounting line is from CAB
        AssetPaymentDocument assetPaymentDocument = (AssetPaymentDocument) event.getDocument();
        if (assetPaymentDocument.isCapitalAssetBuilderOriginIndicator()) {
            return true;
        }

        boolean valid = true;
        AssetPaymentDetail assetPaymentDetail = (AssetPaymentDetail) getAccountingLineForValidation();
        Date expenditureFinancialDocumentPostedDate = assetPaymentDetail.getExpenditureFinancialDocumentPostedDate();
        if (expenditureFinancialDocumentPostedDate != null) {
            // check if date is after today
            java.util.Date today = dateTimeService.getCurrentDate();
            if (expenditureFinancialDocumentPostedDate.after(today)) {
                GlobalVariables.getMessageMap().putError(CamsPropertyConstants.AssetPaymentDetail.DOCUMENT_POSTING_DATE,
                        CamsKeyConstants.Payment.ERROR_POSTING_DATE_FUTURE_NOT_ALLOWED);
                return false;
            }
            Map<String, Object> keyToFind = new HashMap<>();
            keyToFind.put(KFSPropertyConstants.UNIVERSITY_DATE, expenditureFinancialDocumentPostedDate);
            UniversityDate universityDate = businessObjectService.findByPrimaryKey(UniversityDate.class, keyToFind);

            if (universityDate == null) {
                String label = dataDictionaryService.getDataDictionary().getBusinessObjectEntry(
                        AssetPaymentDetail.class.getName()).getAttributeDefinition(
                        CamsPropertyConstants.AssetPaymentDetail.DOCUMENT_POSTING_DATE).getLabel();
                GlobalVariables.getMessageMap().putError(CamsPropertyConstants.AssetPaymentDetail.DOCUMENT_POSTING_DATE,
                        KFSKeyConstants.ERROR_EXISTENCE, label);
                return false;
            }

            // Validating the fiscal year extracted from posted document date is not greater than the current fiscal year.
            Integer currentFiscalYear = universityDateService.getCurrentFiscalYear();
            if (universityDate.getUniversityFiscalYear().compareTo(currentFiscalYear) > 0) {
                GlobalVariables.getMessageMap().putError(CamsPropertyConstants.AssetPaymentDetail.DOCUMENT_POSTING_DATE,
                        CamsKeyConstants.Payment.ERROR_INVALID_DOC_POST_DATE);
                valid = false;
            }
            if (valid) {
                assetPaymentService.extractPostedDatePeriod(assetPaymentDetail);
            }
        } else {
            String label = dataDictionaryService.getDataDictionary().getBusinessObjectEntry(
                    AssetPaymentDetail.class.getName()).getAttributeDefinition(
                    CamsPropertyConstants.AssetPaymentDetail.DOCUMENT_POSTING_DATE).getLabel();
            GlobalVariables.getMessageMap().putError(CamsPropertyConstants.AssetPaymentDetail.DOCUMENT_POSTING_DATE,
                    KFSKeyConstants.ERROR_REQUIRED, label);
            valid = false;
        }
        return valid;
    }

    public AccountingLine getAccountingLineForValidation() {
        return accountingLineForValidation;
    }

    public void setAccountingLineForValidation(AccountingLine accountingLine) {
        this.accountingLineForValidation = accountingLine;
    }

    public DateTimeService getDateTimeService() {
        return dateTimeService;
    }

    public void setDateTimeService(DateTimeService dateTimeService) {
        this.dateTimeService = dateTimeService;
    }

    public BusinessObjectService getBusinessObjectService() {
        return businessObjectService;
    }

    public void setBusinessObjectService(BusinessObjectService businessObjectService) {
        this.businessObjectService = businessObjectService;
    }

    public DataDictionaryService getDataDictionaryService() {
        return dataDictionaryService;
    }

    public void setDataDictionaryService(DataDictionaryService dataDictionaryService) {
        this.dataDictionaryService = dataDictionaryService;
    }

    public UniversityDateService getUniversityDateService() {
        return universityDateService;
    }

    public void setUniversityDateService(UniversityDateService universityDateService) {
        this.universityDateService = universityDateService;
    }

    public AssetPaymentService getAssetPaymentService() {
        return assetPaymentService;
    }

    public void setAssetPaymentService(AssetPaymentService assetPaymentService) {
        this.assetPaymentService = assetPaymentService;
    }
}
