/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2019 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cam.businessobject;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.coa.businessobject.Account;
import org.kuali.kfs.coa.businessobject.AccountingPeriod;
import org.kuali.kfs.coa.businessobject.Chart;
import org.kuali.kfs.coa.businessobject.ObjectSubType;
import org.kuali.kfs.integration.cg.ContractsAndGrantsAgency;
import org.kuali.kfs.krad.bo.DocumentHeader;
import org.kuali.kfs.krad.bo.PersistableBusinessObject;
import org.kuali.kfs.krad.bo.PersistableBusinessObjectBase;
import org.kuali.kfs.krad.service.KualiModuleService;
import org.kuali.kfs.krad.util.KRADConstants;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.krad.util.UrlFactory;
import org.kuali.kfs.module.cam.CamsConstants;
import org.kuali.kfs.module.cam.CamsPropertyConstants;
import org.kuali.kfs.module.cam.document.service.PaymentSummaryService;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.businessobject.Building;
import org.kuali.kfs.sys.businessobject.Campus;
import org.kuali.kfs.sys.businessobject.Room;
import org.kuali.kfs.sys.businessobject.UniversityDate;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.sys.service.UniversityDateService;
import org.kuali.rice.core.api.config.property.ConfigurationService;
import org.kuali.rice.core.api.datetime.DateTimeService;
import org.kuali.rice.core.api.util.type.KualiDecimal;
import org.kuali.rice.kew.api.KewApiConstants;
import org.kuali.rice.kim.api.identity.Person;

import java.sql.Date;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class AssetBase extends PersistableBusinessObjectBase {

    protected Long capitalAssetNumber;
    protected String capitalAssetDescription;
    protected String capitalAssetTypeCode;
    protected String conditionCode;
    protected Date createDate;
    protected Date receiveDate;
    protected Date loanReturnDate;
    protected Date loanDate;
    protected Date expectedReturnDate;
    protected String financialDocumentPostingPeriodCode;
    protected Integer financialDocumentPostingYear;
    protected String organizationOwnerAccountNumber;
    protected String organizationOwnerChartOfAccountsCode;
    protected String vendorName;
    protected String acquisitionTypeCode;
    protected KualiDecimal totalCostAmount;
    protected KualiDecimal replacementAmount;
    protected KualiDecimal salePrice;
    protected KualiDecimal estimatedSellingPrice;
    protected KualiDecimal salvageAmount;
    protected String campusCode;
    protected String buildingCode;
    protected String buildingRoomNumber;
    protected String buildingSubRoomNumber;
    protected String retirementChartOfAccountsCode;
    protected String retirementAccountNumber;
    protected String retirementReasonCode;
    protected String retirementPeriodCode;
    protected Integer retirementFiscalYear;
    protected String cashReceiptFinancialDocumentNumber;
    protected String primaryDepreciationMethodCode;
    protected Date estimatedFabricationCompletionDate;
    protected KualiDecimal fabricationEstimatedTotalAmount;
    protected String transferOfFundsFinancialDocumentNumber;
    protected String inventoryStatusCode;
    protected String campusTagNumber;
    protected Timestamp lastInventoryDate;
    protected String organizationInventoryName;
    protected String oldTagNumber;
    protected String manufacturerName;
    protected String manufacturerModelNumber;
    protected String serialNumber;
    protected String representativeUniversalIdentifier;
    protected String agencyNumber;
    protected String campusPoliceDepartmentCaseNumber;
    protected String inventoryScannedCode;
    protected boolean active;
    protected Date capitalAssetInServiceDate;
    protected String governmentTagNumber;
    protected String nationalStockNumber;
    protected String landCountyName;
    protected Integer landAcreageSize;
    protected String landParcelNumber;
    protected Date depreciationDate;
    protected String financialObjectSubTypeCode;
    protected Integer fabricationEstimatedRetentionYears;

    protected AssetType capitalAssetType;
    protected Account organizationOwnerAccount;
    protected Chart organizationOwnerChartOfAccounts;
    protected Campus campus;
    protected Room buildingRoom;
    protected Account retirementAccount;
    protected Chart retirementChartOfAccounts;
    protected AccountingPeriod financialDocumentPostingPeriod;
    protected Building building;
    protected AccountingPeriod retirementPeriod;
    protected AssetRetirementReason retirementReason;
    protected DocumentHeader cashReceiptFinancialDocument;
    protected DocumentHeader transferOfFundsFinancialDocument;
    protected AssetCondition condition;
    protected AssetStatus inventoryStatus;
    protected List<AssetPayment> assetPayments;
    protected Person assetRepresentative;
    protected Person borrowerPerson;
    protected AssetOrganization assetOrganization;
    protected String organizationTagNumber;
    protected List<AssetRepairHistory> assetRepairHistory;
    protected AssetWarranty assetWarranty;
    protected List<AssetComponent> assetComponents;
    protected List<AssetLocation> assetLocations;
    protected List<AssetRetirementGlobalDetail> assetRetirementHistory;
    protected AssetDepreciationMethod assetPrimaryDepreciationMethod;
    protected List<AssetRetirementGlobal> retirementGlobals;
    protected ObjectSubType financialObjectSubType;
    protected AssetAcquisitionType acquisitionType;
    protected ContractsAndGrantsAgency agency;


    // Non-persisted attributes:
    protected KualiDecimal paymentTotalCost;
    protected transient AssetGlobal separateHistory;
    protected List<AssetRetirementGlobalDetail> mergeHistory;
    protected KualiDecimal federalContribution;
    protected AssetRetirementGlobalDetail retirementInfo;
    protected AssetLocation offCampusLocation;
    protected AssetLocation borrowerLocation;
    protected AssetLocation borrowerStorageLocation;
    // calculated depreciation amounts
    protected KualiDecimal accumulatedDepreciation;
    protected KualiDecimal baseAmount;
    protected KualiDecimal bookValue;
    protected KualiDecimal prevYearDepreciation;
    protected KualiDecimal yearToDateDepreciation;
    protected KualiDecimal currentMonthDepreciation;
    protected Date depreciationDateCopy;
    protected transient Integer quantity;
    protected String lookup;
    protected String assetTransferDocumentLookup;
    protected String assetMaintenanceDocumentLookup;
    protected String assetFabricationDocumentLookup;
    protected String assetCreateOrSeparateDocumentLookup;
    protected String assetPaymentDocumentLookup;
    protected String assetEquipmentLoanOrReturnDocumentLookup;
    protected String assetLocationDocumentLookup;
    protected String assetMergeOrRetirementDocumentLookup;
    protected String camsComplexMaintenanceDocumentLookup;
    protected boolean tagged;
    protected String lastInventoryDateUpdateButton;

    public AssetBase() {
        this.assetPayments = new ArrayList<>();
        this.assetRepairHistory = new ArrayList<>();
        this.assetComponents = new ArrayList<>();
        this.assetLocations = new ArrayList<>();
        this.assetRetirementHistory = new ArrayList<>();
        this.retirementGlobals = new ArrayList<>();
        this.mergeHistory = new ArrayList<>();
    }

    /**
     * Constructs a Asset object. Includes logic to properly set fields depending if it's creating a new asset or
     * separating an asset
     *
     * @param assetGlobal
     * @param assetGlobalDetail
     * @param separate          if it's separate an asset
     */
    public AssetBase(AssetGlobal assetGlobal, AssetGlobalDetail assetGlobalDetail, boolean separate) {
        this();

        UniversityDateService universityDateService = SpringContext.getBean(UniversityDateService.class);

        UniversityDate date = universityDateService.getCurrentUniversityDate();
        this.setFinancialDocumentPostingYear(date.getUniversityFiscalYear());
        this.setFinancialDocumentPostingPeriodCode(date.getUniversityFiscalAccountingPeriod());
        this.setLastInventoryDate(new Timestamp(SpringContext.getBean(DateTimeService.class).getCurrentSqlDate()
                .getTime()));

        this.setPrimaryDepreciationMethodCode(CamsConstants.Asset.DEPRECIATION_METHOD_STRAIGHT_LINE_CODE);

        this.setInventoryStatusCode(assetGlobal.getInventoryStatusCode());
        this.setConditionCode(assetGlobal.getConditionCode());
        this.setAcquisitionTypeCode(assetGlobal.getAcquisitionTypeCode());
        this.setLandCountyName(assetGlobal.getLandCountyName());
        this.setLandAcreageSize(assetGlobal.getLandAcreageSize());
        this.setLandParcelNumber(assetGlobal.getLandParcelNumber());
        this.setVendorName(assetGlobal.getVendorName());
        this.setOrganizationOwnerAccountNumber(assetGlobal.getOrganizationOwnerAccountNumber());
        this.setOrganizationOwnerChartOfAccountsCode(assetGlobal.getOrganizationOwnerChartOfAccountsCode());
        this.setAgencyNumber(assetGlobal.getAgencyNumber());
        this.setCapitalAssetInServiceDate(assetGlobal.getCapitalAssetInServiceDate());
        this.setDepreciationDate(assetGlobal.getCapitalAssetDepreciationDate());
        this.setCreateDate(assetGlobal.getCreateDate());

        this.setCapitalAssetNumber(assetGlobalDetail.getCapitalAssetNumber());
        this.setCampusCode(assetGlobalDetail.getCampusCode());
        this.setBuildingCode(assetGlobalDetail.getBuildingCode());
        this.setBuildingRoomNumber(assetGlobalDetail.getBuildingRoomNumber());
        this.setBuildingSubRoomNumber(assetGlobalDetail.getBuildingSubRoomNumber());
        this.setSerialNumber(assetGlobalDetail.getSerialNumber());
        this.setOrganizationInventoryName(assetGlobalDetail.getOrganizationInventoryName());
        this.setGovernmentTagNumber(assetGlobalDetail.getGovernmentTagNumber());
        this.setCampusTagNumber(assetGlobalDetail.getCampusTagNumber());
        this.setNationalStockNumber(assetGlobalDetail.getNationalStockNumber());

        AssetOrganization assetOrganization = new AssetOrganization();
        assetOrganization.setCapitalAssetNumber(assetGlobalDetail.getCapitalAssetNumber());
        assetOrganization.setOrganizationAssetTypeIdentifier(assetGlobalDetail.getOrganizationAssetTypeIdentifier());
        this.setAssetOrganization(assetOrganization);

        this.setActive(true);

        if (separate) {
            this.setRepresentativeUniversalIdentifier(assetGlobalDetail.getRepresentativeUniversalIdentifier());
            this.setCapitalAssetTypeCode(assetGlobalDetail.getCapitalAssetTypeCode());
            this.setCapitalAssetDescription(assetGlobalDetail.getCapitalAssetDescription());
            this.setManufacturerName(assetGlobalDetail.getManufacturerName());
            this.setManufacturerModelNumber(assetGlobalDetail.getManufacturerModelNumber());

            this.assetOrganization.setOrganizationText(assetGlobalDetail.getOrganizationText());
        } else {
            this.setRepresentativeUniversalIdentifier(assetGlobalDetail.getRepresentativeUniversalIdentifier());
            this.setCapitalAssetTypeCode(assetGlobal.getCapitalAssetTypeCode());
            this.setCapitalAssetDescription(assetGlobal.getCapitalAssetDescription());
            this.setManufacturerName(assetGlobal.getManufacturerName());
            this.setManufacturerModelNumber(assetGlobal.getManufacturerModelNumber());

            this.assetOrganization.setOrganizationText(assetGlobal.getOrganizationText());
        }
    }

    public KualiDecimal getCurrentMonthDepreciation() {
        return currentMonthDepreciation;
    }

    public void setCurrentMonthDepreciation(KualiDecimal currentMonthDepreciation) {
        this.currentMonthDepreciation = currentMonthDepreciation;
    }

    public KualiDecimal getAccumulatedDepreciation() {

        // Calculates payment summary and depreciation summary based on available payment records
        // The value can be null due to it being used as a non-singleton on the AssetRetirementGlobal
        // page (a list of retired assets). If it were a singleton, each value would get overridden
        // by the next use on the same page
        if (accumulatedDepreciation == null) {
            SpringContext.getBean(PaymentSummaryService.class).calculateAndSetPaymentSummary((Asset) this);
        }

        return accumulatedDepreciation;
    }

    public void setAccumulatedDepreciation(KualiDecimal accumulatedDepreciation) {
        this.accumulatedDepreciation = accumulatedDepreciation;
    }

    public KualiDecimal getBaseAmount() {
        return baseAmount;
    }

    public void setBaseAmount(KualiDecimal baseAmount) {
        this.baseAmount = baseAmount;
    }

    public KualiDecimal getBookValue() {
        // Calculates payment summary and depreciation summary based on available payment records
        // The value can be null due to it being used as a non-singleton on the AssetRetirementGlobal
        // page (a list of retired assets). If it were a singleton, each value would get overridden
        // by the next use on the same page
        if (bookValue == null) {
            SpringContext.getBean(PaymentSummaryService.class).calculateAndSetPaymentSummary((Asset) this);
        }

        return bookValue;
    }

    public void setBookValue(KualiDecimal bookValue) {
        this.bookValue = bookValue;
    }

    public KualiDecimal getPrevYearDepreciation() {
        return prevYearDepreciation;
    }

    public void setPrevYearDepreciation(KualiDecimal prevYearDepreciation) {
        this.prevYearDepreciation = prevYearDepreciation;
    }

    public KualiDecimal getYearToDateDepreciation() {
        return yearToDateDepreciation;
    }

    public void setYearToDateDepreciation(KualiDecimal yearToDateDepreciation) {
        this.yearToDateDepreciation = yearToDateDepreciation;
    }

    public Long getCapitalAssetNumber() {
        return capitalAssetNumber;
    }

    public void setCapitalAssetNumber(Long capitalAssetNumber) {
        this.capitalAssetNumber = capitalAssetNumber;
    }

    public String getCapitalAssetDescription() {
        return capitalAssetDescription;
    }

    public void setCapitalAssetDescription(String capitalAssetDescription) {
        this.capitalAssetDescription = capitalAssetDescription;
    }

    public String getCapitalAssetTypeCode() {
        return capitalAssetTypeCode;
    }

    public void setCapitalAssetTypeCode(String capitalAssetTypeCode) {
        this.capitalAssetTypeCode = capitalAssetTypeCode;
    }

    public String getConditionCode() {
        return conditionCode;
    }

    public void setConditionCode(String conditionCode) {
        this.conditionCode = conditionCode;
    }

    public Date getCreateDate() {
        return createDate;
    }

    public void setCreateDate(Date createDate) {
        this.createDate = createDate;
    }

    public Date getReceiveDate() {
        return receiveDate;
    }

    public void setReceiveDate(Date receiveDate) {
        this.receiveDate = receiveDate;
    }

    public Date getLoanReturnDate() {
        return loanReturnDate;
    }

    public void setLoanReturnDate(Date loanReturnDate) {
        this.loanReturnDate = loanReturnDate;
    }

    public Date getLoanDate() {
        return loanDate;
    }

    public void setLoanDate(Date loanDate) {
        this.loanDate = loanDate;
    }

    public Date getExpectedReturnDate() {
        return expectedReturnDate;
    }

    public void setExpectedReturnDate(Date expectedReturnDate) {
        this.expectedReturnDate = expectedReturnDate;
    }

    public String getFinancialDocumentPostingPeriodCode() {
        return financialDocumentPostingPeriodCode;
    }

    public void setFinancialDocumentPostingPeriodCode(String financialDocumentPostingPeriodCode) {
        this.financialDocumentPostingPeriodCode = financialDocumentPostingPeriodCode;
    }

    public Integer getFinancialDocumentPostingYear() {
        return financialDocumentPostingYear;
    }

    public void setFinancialDocumentPostingYear(Integer financialDocumentPostingYear) {
        this.financialDocumentPostingYear = financialDocumentPostingYear;
    }

    public String getOrganizationOwnerAccountNumber() {
        return organizationOwnerAccountNumber;
    }

    public void setOrganizationOwnerAccountNumber(String organizationOwnerAccountNumber) {
        this.organizationOwnerAccountNumber = organizationOwnerAccountNumber;
    }

    public String getOrganizationOwnerChartOfAccountsCode() {
        return organizationOwnerChartOfAccountsCode;
    }

    public void setOrganizationOwnerChartOfAccountsCode(String organizationOwnerChartOfAccountsCode) {
        this.organizationOwnerChartOfAccountsCode = organizationOwnerChartOfAccountsCode;
    }

    public String getVendorName() {
        return vendorName;
    }

    public void setVendorName(String vendorName) {
        this.vendorName = vendorName;
    }

    public String getAcquisitionTypeCode() {
        return acquisitionTypeCode;
    }

    public void setAcquisitionTypeCode(String acquisitionTypeCode) {
        this.acquisitionTypeCode = acquisitionTypeCode;
    }

    public KualiDecimal getTotalCostAmount() {
        return totalCostAmount;
    }

    public void setTotalCostAmount(KualiDecimal totalCostAmount) {
        this.totalCostAmount = totalCostAmount;
    }

    public KualiDecimal getReplacementAmount() {
        return replacementAmount;
    }

    public void setReplacementAmount(KualiDecimal replacementAmount) {
        this.replacementAmount = replacementAmount;
    }

    public KualiDecimal getSalePrice() {
        return salePrice;
    }

    public void setSalePrice(KualiDecimal salePrice) {
        this.salePrice = salePrice;
    }

    public KualiDecimal getEstimatedSellingPrice() {
        return estimatedSellingPrice;
    }

    public void setEstimatedSellingPrice(KualiDecimal estimatedSellingPrice) {
        this.estimatedSellingPrice = estimatedSellingPrice;
    }

    public KualiDecimal getSalvageAmount() {
        if (salvageAmount == null) {
            salvageAmount = KualiDecimal.ZERO;
        }
        return salvageAmount;
    }

    public void setSalvageAmount(KualiDecimal salvageAmount) {
        this.salvageAmount = salvageAmount;
    }

    public String getCampusCode() {
        return campusCode;
    }

    public void setCampusCode(String campusCode) {
        this.campusCode = campusCode;
    }

    public String getBuildingCode() {
        return buildingCode;
    }

    public void setBuildingCode(String buildingCode) {
        this.buildingCode = buildingCode;
    }

    public String getBuildingRoomNumber() {
        return buildingRoomNumber;
    }

    public void setBuildingRoomNumber(String buildingRoomNumber) {
        this.buildingRoomNumber = buildingRoomNumber;
    }

    public String getBuildingSubRoomNumber() {
        return buildingSubRoomNumber;
    }

    public void setBuildingSubRoomNumber(String buildingSubRoomNumber) {
        this.buildingSubRoomNumber = buildingSubRoomNumber;
    }

    public String getRetirementChartOfAccountsCode() {
        return retirementChartOfAccountsCode;
    }

    public void setRetirementChartOfAccountsCode(String retirementChartOfAccountsCode) {
        this.retirementChartOfAccountsCode = retirementChartOfAccountsCode;
    }

    public String getRetirementAccountNumber() {
        return retirementAccountNumber;
    }

    public void setRetirementAccountNumber(String retirementAccountNumber) {
        this.retirementAccountNumber = retirementAccountNumber;
    }

    public String getRetirementReasonCode() {
        return retirementReasonCode;
    }

    public void setRetirementReasonCode(String retirementReasonCode) {
        this.retirementReasonCode = retirementReasonCode;
    }

    public String getRetirementPeriodCode() {
        return retirementPeriodCode;
    }

    public void setRetirementPeriodCode(String retirementPeriodCode) {
        this.retirementPeriodCode = retirementPeriodCode;
    }

    public Integer getRetirementFiscalYear() {
        return retirementFiscalYear;
    }

    public void setRetirementFiscalYear(Integer retirementFiscalYear) {
        this.retirementFiscalYear = retirementFiscalYear;
    }

    public String getCashReceiptFinancialDocumentNumber() {
        return cashReceiptFinancialDocumentNumber;
    }

    public void setCashReceiptFinancialDocumentNumber(String cashReceiptFinancialDocumentNumber) {
        this.cashReceiptFinancialDocumentNumber = cashReceiptFinancialDocumentNumber;
    }

    public String getPrimaryDepreciationMethodCode() {
        return primaryDepreciationMethodCode;
    }

    public void setPrimaryDepreciationMethodCode(String primaryDepreciationMethodCode) {
        this.primaryDepreciationMethodCode = primaryDepreciationMethodCode;
    }

    public Date getEstimatedFabricationCompletionDate() {
        return estimatedFabricationCompletionDate;
    }

    public void setEstimatedFabricationCompletionDate(Date estimatedFabricationCompletionDate) {
        this.estimatedFabricationCompletionDate = estimatedFabricationCompletionDate;
    }

    public KualiDecimal getFabricationEstimatedTotalAmount() {
        return fabricationEstimatedTotalAmount;
    }

    public void setFabricationEstimatedTotalAmount(KualiDecimal fabricationEstimatedTotalAmount) {
        this.fabricationEstimatedTotalAmount = fabricationEstimatedTotalAmount;
    }

    public String getTransferOfFundsFinancialDocumentNumber() {
        return transferOfFundsFinancialDocumentNumber;
    }

    public void setTransferOfFundsFinancialDocumentNumber(String transferOfFundsFinancialDocumentNumber) {
        this.transferOfFundsFinancialDocumentNumber = transferOfFundsFinancialDocumentNumber;
    }

    public String getInventoryStatusCode() {
        return inventoryStatusCode;
    }

    public void setInventoryStatusCode(String inventoryStatusCode) {
        this.inventoryStatusCode = inventoryStatusCode;
    }

    public String getCampusTagNumber() {
        return campusTagNumber;
    }

    public void setCampusTagNumber(String campusTagNumber) {
        this.campusTagNumber = campusTagNumber;
    }

    public Timestamp getLastInventoryDate() {
        return lastInventoryDate;
    }

    public void setLastInventoryDate(Timestamp lastInventoryDate) {
        this.lastInventoryDate = lastInventoryDate;
    }

    public String getOrganizationInventoryName() {
        return organizationInventoryName;
    }

    public void setOrganizationInventoryName(String organizationInventoryName) {
        this.organizationInventoryName = organizationInventoryName;
    }

    public String getOldTagNumber() {
        return oldTagNumber;
    }

    public void setOldTagNumber(String oldTagNumber) {
        this.oldTagNumber = oldTagNumber;
    }

    public String getManufacturerName() {
        return manufacturerName;
    }

    public void setManufacturerName(String manufacturerName) {
        this.manufacturerName = manufacturerName;
    }

    public String getManufacturerModelNumber() {
        return manufacturerModelNumber;
    }

    public void setManufacturerModelNumber(String manufacturerModelNumber) {
        this.manufacturerModelNumber = manufacturerModelNumber;
    }

    public String getSerialNumber() {
        return serialNumber;
    }

    public void setSerialNumber(String serialNumber) {
        this.serialNumber = serialNumber;
    }

    public String getRepresentativeUniversalIdentifier() {
        return representativeUniversalIdentifier;
    }

    public void setRepresentativeUniversalIdentifier(String representativeUniversalIdentifier) {
        this.representativeUniversalIdentifier = representativeUniversalIdentifier;
    }

    public String getAgencyNumber() {
        return agencyNumber;
    }

    public void setAgencyNumber(String agencyNumber) {
        this.agencyNumber = agencyNumber;
    }

    public String getCampusPoliceDepartmentCaseNumber() {
        return campusPoliceDepartmentCaseNumber;
    }

    public void setCampusPoliceDepartmentCaseNumber(String campusPoliceDepartmentCaseNumber) {
        this.campusPoliceDepartmentCaseNumber = campusPoliceDepartmentCaseNumber;
    }

    public String getInventoryScannedCode() {
        return inventoryScannedCode;
    }

    public void setInventoryScannedCode(String inventoryScannedCode) {
        this.inventoryScannedCode = inventoryScannedCode;
    }

    public Date getCapitalAssetInServiceDate() {
        return capitalAssetInServiceDate;
    }

    public void setCapitalAssetInServiceDate(Date capitalAssetInServiceDate) {
        this.capitalAssetInServiceDate = capitalAssetInServiceDate;
    }

    public String getGovernmentTagNumber() {
        return governmentTagNumber;
    }

    public void setGovernmentTagNumber(String governmentTagNumber) {
        this.governmentTagNumber = governmentTagNumber;
    }

    public String getNationalStockNumber() {
        return nationalStockNumber;
    }

    public void setNationalStockNumber(String nationalStockNumber) {
        this.nationalStockNumber = nationalStockNumber;
    }

    public Integer getLandAcreageSize() {
        return landAcreageSize;
    }

    public void setLandAcreageSize(Integer landAcreageSize) {
        this.landAcreageSize = landAcreageSize;
    }

    public String getLandCountyName() {
        return landCountyName;
    }

    public void setLandCountyName(String landCountyName) {
        this.landCountyName = landCountyName;
    }

    public String getLandParcelNumber() {
        return landParcelNumber;
    }

    public void setLandParcelNumber(String landParcelNumber) {
        this.landParcelNumber = landParcelNumber;
    }

    public Date getDepreciationDate() {
        return depreciationDate;
    }

    public void setDepreciationDate(Date depreciationDate) {
        this.depreciationDate = depreciationDate;
    }

    public AssetType getCapitalAssetType() {
        return capitalAssetType;
    }

    @Deprecated
    public void setCapitalAssetType(AssetType capitalAssetType) {
        this.capitalAssetType = capitalAssetType;
    }

    public Account getOrganizationOwnerAccount() {
        return organizationOwnerAccount;
    }

    @Deprecated
    public void setOrganizationOwnerAccount(Account organizationOwnerAccount) {
        this.organizationOwnerAccount = organizationOwnerAccount;
    }

    public Chart getOrganizationOwnerChartOfAccounts() {
        return organizationOwnerChartOfAccounts;
    }

    @Deprecated
    public void setOrganizationOwnerChartOfAccounts(Chart organizationOwnerChartOfAccounts) {
        this.organizationOwnerChartOfAccounts = organizationOwnerChartOfAccounts;
    }

    public Campus getCampus() {
        return campus;
    }

    @Deprecated
    public void setCampus(Campus campus) {
        this.campus = campus;
    }

    public Room getBuildingRoom() {
        return buildingRoom;
    }

    @Deprecated
    public void setBuildingRoom(Room buildingRoom) {
        this.buildingRoom = buildingRoom;
    }

    public Account getRetirementAccount() {
        return retirementAccount;
    }

    @Deprecated
    public void setRetirementAccount(Account retirementAccount) {
        this.retirementAccount = retirementAccount;
    }

    public Chart getRetirementChartOfAccounts() {
        return retirementChartOfAccounts;
    }

    @Deprecated
    public void setRetirementChartOfAccounts(Chart retirementChartOfAccounts) {
        this.retirementChartOfAccounts = retirementChartOfAccounts;
    }

    public Building getBuilding() {
        return building;
    }

    @Deprecated
    public void setBuilding(Building building) {
        this.building = building;
    }

    public DocumentHeader getCashReceiptFinancialDocument() {
        return cashReceiptFinancialDocument;
    }

    @Deprecated
    public void setCashReceiptFinancialDocument(DocumentHeader cashReceiptFinancialDocument) {
        this.cashReceiptFinancialDocument = cashReceiptFinancialDocument;
    }

    public AccountingPeriod getRetirementPeriod() {
        return retirementPeriod;
    }

    @Deprecated
    public void setRetirementPeriod(AccountingPeriod retirementPeriod) {
        this.retirementPeriod = retirementPeriod;
    }

    public AssetRetirementReason getRetirementReason() {
        return retirementReason;
    }

    @Deprecated
    public void setRetirementReason(AssetRetirementReason retirementReason) {
        this.retirementReason = retirementReason;
    }

    public DocumentHeader getTransferOfFundsFinancialDocument() {
        return transferOfFundsFinancialDocument;
    }

    @Deprecated
    public void setTransferOfFundsFinancialDocument(DocumentHeader transferOfFundsFinancialDocument) {
        this.transferOfFundsFinancialDocument = transferOfFundsFinancialDocument;
    }

    public AccountingPeriod getFinancialDocumentPostingPeriod() {
        return financialDocumentPostingPeriod;
    }

    @Deprecated
    public void setFinancialDocumentPostingPeriod(AccountingPeriod financialDocumentPostingPeriod) {
        this.financialDocumentPostingPeriod = financialDocumentPostingPeriod;
    }

    public AssetCondition getCondition() {
        return condition;
    }

    @Deprecated
    public void setCondition(AssetCondition condition) {
        this.condition = condition;
    }

    public boolean isActive() {
        return active;
    }

    public void setActive(boolean active) {
        this.active = active;
    }

    public List<AssetPayment> getAssetPayments() {
        return assetPayments;
    }

    public void setAssetPayments(List<AssetPayment> assetPayments) {
        this.assetPayments = assetPayments;
    }

    public AssetStatus getInventoryStatus() {
        return inventoryStatus;
    }

    public void setInventoryStatus(AssetStatus inventoryStatus) {
        this.inventoryStatus = inventoryStatus;
    }

    public Person getAssetRepresentative() {
        assetRepresentative = SpringContext.getBean(org.kuali.rice.kim.api.identity.PersonService.class)
                .updatePersonIfNecessary(representativeUniversalIdentifier, assetRepresentative);
        return assetRepresentative;
    }

    @Deprecated
    public void setAssetRepresentative(Person assetRepresentative) {
        this.assetRepresentative = assetRepresentative;
    }

    public Person getBorrowerPerson() {
        if (ObjectUtils.isNotNull(borrowerLocation)) {
            borrowerPerson = SpringContext.getBean(org.kuali.rice.kim.api.identity.PersonService.class)
                    .updatePersonIfNecessary(borrowerLocation.getAssetLocationContactIdentifier(), borrowerPerson);
        }
        return borrowerPerson;
    }

    public void setBorrowerPerson(Person borrowerPerson) {
        this.borrowerPerson = borrowerPerson;
    }

    public AssetOrganization getAssetOrganization() {
        return assetOrganization;
    }

    public void setAssetOrganization(AssetOrganization assetOrganization) {
        this.assetOrganization = assetOrganization;
    }

    public String getOrganizationTagNumber() {
        return organizationTagNumber;
    }

    public void setOrganizationTagNumber(String organizationTagNumber) {
        this.organizationTagNumber = organizationTagNumber;
    }

    public List<AssetRepairHistory> getAssetRepairHistory() {
        return assetRepairHistory;
    }

    public void setAssetRepairHistory(List<AssetRepairHistory> assetRepairHistory) {
        this.assetRepairHistory = assetRepairHistory;
    }

    public AssetWarranty getAssetWarranty() {
        return assetWarranty;
    }

    public void setAssetWarranty(AssetWarranty assetWarranty) {
        this.assetWarranty = assetWarranty;
    }

    public KualiDecimal getPaymentTotalCost() {
        return paymentTotalCost;
    }

    public void setPaymentTotalCost(KualiDecimal paymentTotalCost) {
        this.paymentTotalCost = paymentTotalCost;
    }

    public List<AssetComponent> getAssetComponents() {
        return assetComponents;
    }

    public void setAssetComponents(List<AssetComponent> assetComponents) {
        this.assetComponents = assetComponents;
    }

    public KualiDecimal getFederalContribution() {
        return federalContribution;
    }

    public void setFederalContribution(KualiDecimal federalContribution) {
        this.federalContribution = federalContribution;
    }

    public AssetRetirementGlobalDetail getRetirementInfo() {
        return retirementInfo;
    }

    public void setRetirementInfo(AssetRetirementGlobalDetail retirementInfo) {
        this.retirementInfo = retirementInfo;
    }

    public List<AssetLocation> getAssetLocations() {
        return assetLocations;
    }

    public void setAssetLocations(List<AssetLocation> assetLocations) {
        this.assetLocations = assetLocations;
    }

    public AssetLocation getOffCampusLocation() {
        return offCampusLocation;
    }

    public void setOffCampusLocation(AssetLocation offCampusLocation) {
        this.offCampusLocation = offCampusLocation;
    }

    public List<AssetRetirementGlobalDetail> getAssetRetirementHistory() {
        return assetRetirementHistory;
    }

    public void setAssetRetirementHistory(List<AssetRetirementGlobalDetail> assetRetirementHistory) {
        this.assetRetirementHistory = assetRetirementHistory;
    }

    public AssetDepreciationMethod getAssetPrimaryDepreciationMethod() {
        return assetPrimaryDepreciationMethod;
    }

    public void setAssetPrimaryDepreciationMethod(AssetDepreciationMethod assetPrimaryDepreciationMethod) {
        this.assetPrimaryDepreciationMethod = assetPrimaryDepreciationMethod;
    }

    public Integer getFabricationEstimatedRetentionYears() {
        return fabricationEstimatedRetentionYears;
    }

    public void setFabricationEstimatedRetentionYears(Integer fabricationEstimatedRetentionYears) {
        this.fabricationEstimatedRetentionYears = fabricationEstimatedRetentionYears;
    }

    public List<AssetRetirementGlobal> getRetirementGlobals() {
        return retirementGlobals;
    }

    public void setRetirementGlobals(List<AssetRetirementGlobal> retirementGlobals) {
        this.retirementGlobals = retirementGlobals;
    }

    public AssetGlobal getSeparateHistory() {
        return separateHistory;
    }

    public void setSeparateHistory(AssetGlobal separateHistory) {
        this.separateHistory = separateHistory;
    }

    public List<AssetRetirementGlobalDetail> getMergeHistory() {
        return mergeHistory;
    }

    public void setMergeHistory(List<AssetRetirementGlobalDetail> mergeHistory) {
        this.mergeHistory = mergeHistory;
    }

    public String getFinancialObjectSubTypeCode() {
        return financialObjectSubTypeCode;
    }

    public void setFinancialObjectSubTypeCode(String financialObjectSubTypeCode) {
        this.financialObjectSubTypeCode = financialObjectSubTypeCode;
    }

    public ObjectSubType getFinancialObjectSubType() {
        return financialObjectSubType;
    }

    @Deprecated
    public void setFinancialObjectSubType(ObjectSubType financialObjectSubType) {
        this.financialObjectSubType = financialObjectSubType;
    }

    public Date getDepreciationDateCopy() {
        return depreciationDate;
    }

    @Deprecated
    public void setDepreciationDateCopy(Date depreciationDateCopy) {
        this.depreciationDateCopy = depreciationDateCopy;
    }

    public AssetAcquisitionType getAcquisitionType() {
        return acquisitionType;
    }

    public void setAcquisitionType(AssetAcquisitionType acquisitionType) {
        this.acquisitionType = acquisitionType;
    }

    public ContractsAndGrantsAgency getAgency() {
        agency = SpringContext.getBean(KualiModuleService.class).getResponsibleModuleService(
                ContractsAndGrantsAgency.class).retrieveExternalizableBusinessObjectIfNecessary(this, agency, "agency");
        return agency;
    }

    /**
     * Technically this is obsolete but necessary because MaintenanceDocumentBase.populateXmlDocumentContentsFromMaintainables has
     * the following hack:<br>
     * ObjectUtils.materializeAllSubObjects(oldBo); // hack to resolve XStream not dealing well with Proxies<br>
     * so as long as that is there we need this setter otherwise a NoSuchMethodException occurs.
     *
     * @deprecated
     */
    @Deprecated
    public void setAgency(ContractsAndGrantsAgency agency) {
        this.agency = agency;
    }

    public Integer getQuantity() {
        return quantity;
    }

    public boolean isTagged() {
        return tagged;
    }

    public void setTagged() {
        this.tagged = !StringUtils.isBlank(campusTagNumber);
    }

    public AssetLocation getBorrowerLocation() {
        return borrowerLocation;
    }

    public void setBorrowerLocation(AssetLocation borrowerLocation) {
        this.borrowerLocation = borrowerLocation;
    }

    public AssetLocation getBorrowerStorageLocation() {
        return borrowerStorageLocation;
    }

    public void setBorrowerStorageLocation(AssetLocation borrowerStorageLocation) {
        this.borrowerStorageLocation = borrowerStorageLocation;
    }

    public String getLastInventoryDateUpdateButton() {
        return lastInventoryDateUpdateButton;
    }

    public void setLastInventoryDateUpdateButton(String lastInventoryDateUpdateButton) {
        this.lastInventoryDateUpdateButton = lastInventoryDateUpdateButton;
    }

    /**
     * @return the link for payment lookup
     */
    public String getLookup() {
        if (getCapitalAssetNumber() == null) {
            return "";
        }

        String basePath = SpringContext.getBean(ConfigurationService.class)
                .getPropertyValueAsString(KFSConstants.APPLICATION_URL_KEY);

        Map<String, String> params = new HashMap<>();
        params.put(KRADConstants.DISPATCH_REQUEST_PARAMETER, KRADConstants.SEARCH_METHOD);
        params.put(KRADConstants.DOC_FORM_KEY, "88888888");
        params.put(KRADConstants.HIDE_LOOKUP_RETURN_LINK, "true");
        params.put(CamsPropertyConstants.Asset.CAPITAL_ASSET_NUMBER, getCapitalAssetNumber().toString());
        params.put(KRADConstants.RETURN_LOCATION_PARAMETER, "portal.do");
        params.put(KRADConstants.BUSINESS_OBJECT_CLASS_ATTRIBUTE, AssetPayment.class.getName());

        return UrlFactory.parameterizeUrl(basePath + "/" + KFSConstants.LOOKUP_ACTION, params);
    }

    protected String getUrlForAssetDocumentLookup(String documentTypeName) {
        if (getCapitalAssetNumber() == null) {
            return "";
        }
        Map<String, String> params = new HashMap<>();
        params.put(KFSPropertyConstants.DOCUMENT_TYPE_NAME, documentTypeName);
        params.put(KewApiConstants.DOCUMENT_ATTRIBUTE_FIELD_PREFIX + CamsPropertyConstants.Asset.CAPITAL_ASSET_NUMBER,
                getCapitalAssetNumber().toString());
        params.put(KRADConstants.DISPATCH_REQUEST_PARAMETER, KRADConstants.SEARCH_METHOD);
        return UrlFactory.parameterizeUrl(SpringContext.getBean(ConfigurationService.class).getPropertyValueAsString(
                KRADConstants.WORKFLOW_DOCUMENTSEARCH_URL_KEY), params);
    }

    /**
     * @return the link for asset transfer document lookup for given capital asset number.
     */
    public String getAssetTransferDocumentLookup() {
        return getUrlForAssetDocumentLookup(CamsConstants.DocumentTypeName.ASSET_TRANSFER);
    }

    /**
     * @return the link for asset maintenance document lookup for given capital asset number.
     */
    public String getAssetMaintenanceDocumentLookup() {
        return getUrlForAssetDocumentLookup(CamsConstants.DocumentTypeName.ASSET_EDIT);
    }

    /**
     * @return the link for asset fabrication document lookup for given capital asset number.
     */
    public String getAssetFabricationDocumentLookup() {
        return getUrlForAssetDocumentLookup(CamsConstants.DocumentTypeName.ASSET_FABRICATION);
    }

    /**
     * @return the link for asset create or separate global document lookup for given capital asset number.
     */
    public String getAssetCreateOrSeparateDocumentLookup() {
        return getUrlForAssetDocumentLookup(CamsConstants.DocumentTypeName.ASSET_ADD_GLOBAL);
    }

    /**
     * @return the link for asset payment document lookup for given capital asset number.
     */
    public String getAssetPaymentDocumentLookup() {
        return getUrlForAssetDocumentLookup(CamsConstants.DocumentTypeName.ASSET_PAYMENT);
    }

    /**
     * @return the link for asset equipment loan or return document lookup for given capital asset number.
     */
    public String getAssetEquipmentLoanOrReturnDocumentLookup() {
        return getUrlForAssetDocumentLookup(CamsConstants.DocumentTypeName.ASSET_EQUIPMENT_LOAN_OR_RETURN);
    }

    /**
     * @return the link for asset location global document lookup for given capital asset number.
     */
    public String getAssetLocationDocumentLookup() {
        return getUrlForAssetDocumentLookup(CamsConstants.DocumentTypeName.ASSET_LOCATION_GLOBAL);
    }

    /**
     * @return the link for asset retirement or merge document lookup for given capital asset number.
     */
    public String getAssetMergeOrRetirementDocumentLookup() {
        return getUrlForAssetDocumentLookup(CamsConstants.DocumentTypeName.ASSET_RETIREMENT_GLOBAL);
    }

    public String getCamsComplexMaintenanceDocumentLookup() {
        return getUrlForAssetDocumentLookup(CamsConstants.DocumentTypeName.COMPLEX_MAINTENANCE_DOC_BASE);
    }

    /**
     * override this method so we can remove the offcampus location
     */
    @Override
    public List<Collection<PersistableBusinessObject>> buildListOfDeletionAwareLists() {
        List<Collection<PersistableBusinessObject>> managedLists = new ArrayList<>();
        managedLists.add(new ArrayList<>(getAssetLocations()));
        return managedLists;
    }

}
