/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2020 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cam.businessobject.options;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.coa.businessobject.AccountingPeriod;
import org.kuali.kfs.coa.service.AccountingPeriodService;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.krad.keyvalues.KeyValuesBase;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.sys.service.impl.KfsParameterConstants;
import org.kuali.rice.core.api.datetime.DateTimeService;
import org.kuali.rice.core.api.util.ConcreteKeyValue;
import org.kuali.rice.core.api.util.KeyValue;

import java.sql.Date;
import java.util.ArrayList;
import java.util.List;

/**
 * returns list of accounting period value pairs valid for year end posting
 */
public class CamsYearEndAccountingPeriodValuesFinder extends KeyValuesBase {

    private DateTimeService dateTimeService;
    private ParameterService parameterService;

    public List<KeyValue> getKeyValues() {
        List<KeyValue> accountingPeriodCodes = new ArrayList<>();

        Date date = dateTimeService.getCurrentSqlDate();
        AccountingPeriod currentAccountingPeriod = getAccountingPeriodService().getByDate(date);

        if (currentAccountingPeriod.isOpen()) {
            // add the current period with blank value, so scrubber will set entries when posting
            accountingPeriodCodes.add(new ConcreteKeyValue(currentAccountingPeriod.getUniversityFiscalPeriodCode()
                    + currentAccountingPeriod.getUniversityFiscalYear(),
                    currentAccountingPeriod.getUniversityFiscalPeriodName()));
        }

        String numberOfPostBackPeriodsParamVal = parameterService.getParameterValueAsString(
                KFSConstants.CoreModuleNamespaces.KFS,
                KfsParameterConstants.YEAR_END_ACCOUNTING_PERIOD_PARAMETER_NAMES.DETAIL_PARAMETER_TYPE,
                KfsParameterConstants.YEAR_END_ACCOUNTING_PERIOD_PARAMETER_NAMES.NUMBER_OF_POSTBACK_PERIODS);
        if (StringUtils.isNotBlank(numberOfPostBackPeriodsParamVal)
                && Integer.parseInt(numberOfPostBackPeriodsParamVal) > 0) {
            for (int i = 1; i <= Integer.parseInt(numberOfPostBackPeriodsParamVal); i++) {
                int currentFiscalYear = currentAccountingPeriod.getUniversityFiscalYear();
                int currentFiscalPeriod = Integer.parseInt(currentAccountingPeriod.getUniversityFiscalPeriodCode());

                if (currentFiscalPeriod == 1) {
                    currentFiscalYear = currentFiscalYear - 1;
                    currentFiscalPeriod = 13;
                } else {
                    currentFiscalPeriod = currentFiscalPeriod - 1;
                }

                currentAccountingPeriod = getAccountingPeriodService()
                        .getByPeriod(StringUtils.leftPad(Integer.toString(currentFiscalPeriod), 2, "0"),
                                currentFiscalYear);
                if (currentAccountingPeriod.isOpen()) {
                    accountingPeriodCodes.add(new ConcreteKeyValue(currentAccountingPeriod
                            .getUniversityFiscalPeriodCode() + currentAccountingPeriod.getUniversityFiscalYear(),
                            currentAccountingPeriod.getUniversityFiscalPeriodName()));
                }
            }
        }

        return accountingPeriodCodes;
    }

    private AccountingPeriodService getAccountingPeriodService() {
        // since this is coming from COA and defined in spring-coa.xml currently, we cannot inject it yet
        return SpringContext.getBean(AccountingPeriodService.class);
    }

    public void setDateTimeService(DateTimeService dateTimeService) {
        this.dateTimeService = dateTimeService;
    }

    public void setParameterService(ParameterService parameterService) {
        this.parameterService = parameterService;
    }
}
