/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cam.businessobject;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.coa.businessobject.Account;
import org.kuali.kfs.coa.businessobject.AccountingPeriod;
import org.kuali.kfs.coa.businessobject.Chart;
import org.kuali.kfs.coa.businessobject.ObjectSubType;
import org.kuali.kfs.integration.cg.ContractsAndGrantsAgency;
import org.kuali.kfs.kim.impl.identity.Person;
import org.kuali.kfs.krad.bo.PersistableBusinessObject;
import org.kuali.kfs.krad.bo.PersistableBusinessObjectBase;
import org.kuali.kfs.krad.service.KualiModuleService;
import org.kuali.kfs.krad.util.KRADConstants;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.krad.util.UrlFactory;
import org.kuali.kfs.module.cam.CamsConstants;
import org.kuali.kfs.module.cam.CamsPropertyConstants;
import org.kuali.kfs.module.cam.document.service.PaymentSummaryService;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.businessobject.Building;
import org.kuali.kfs.sys.businessobject.Campus;
import org.kuali.kfs.sys.businessobject.DocumentHeader;
import org.kuali.kfs.sys.businessobject.Room;
import org.kuali.kfs.sys.businessobject.UniversityDate;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.sys.service.UniversityDateService;
import org.kuali.kfs.core.api.config.property.ConfigurationService;
import org.kuali.kfs.core.api.datetime.DateTimeService;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.kew.api.KewApiConstants;

import java.sql.Date;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class AssetBase extends PersistableBusinessObjectBase {

    protected Long capitalAssetNumber;
    protected String capitalAssetDescription;
    protected String capitalAssetTypeCode;
    protected String conditionCode;
    protected Date createDate;
    protected Date receiveDate;
    protected Date loanReturnDate;
    protected Date loanDate;
    protected Date expectedReturnDate;
    protected String financialDocumentPostingPeriodCode;
    protected Integer financialDocumentPostingYear;
    protected String organizationOwnerAccountNumber;
    protected String organizationOwnerChartOfAccountsCode;
    protected String vendorName;
    protected String acquisitionTypeCode;
    protected KualiDecimal totalCostAmount;
    protected KualiDecimal replacementAmount;
    protected KualiDecimal salePrice;
    protected KualiDecimal estimatedSellingPrice;
    protected KualiDecimal salvageAmount;
    protected String campusCode;
    protected String buildingCode;
    protected String buildingRoomNumber;
    protected String buildingSubRoomNumber;
    protected String retirementChartOfAccountsCode;
    protected String retirementAccountNumber;
    protected String retirementReasonCode;
    protected String retirementPeriodCode;
    protected Integer retirementFiscalYear;
    protected String cashReceiptFinancialDocumentNumber;
    protected String primaryDepreciationMethodCode;
    protected Date estimatedFabricationCompletionDate;
    protected KualiDecimal fabricationEstimatedTotalAmount;
    protected String transferOfFundsFinancialDocumentNumber;
    protected String inventoryStatusCode;
    protected String campusTagNumber;
    protected Timestamp lastInventoryDate;
    protected String organizationInventoryName;
    protected String oldTagNumber;
    protected String manufacturerName;
    protected String manufacturerModelNumber;
    protected String serialNumber;
    protected String representativeUniversalIdentifier;
    protected String agencyNumber;
    protected String campusPoliceDepartmentCaseNumber;
    protected String inventoryScannedCode;
    protected boolean active;
    protected Date capitalAssetInServiceDate;
    protected String governmentTagNumber;
    protected String nationalStockNumber;
    protected String landCountyName;
    protected Integer landAcreageSize;
    protected String landParcelNumber;
    protected Date depreciationDate;
    protected String financialObjectSubTypeCode;
    protected Integer fabricationEstimatedRetentionYears;

    protected AssetType capitalAssetType;
    protected Account organizationOwnerAccount;
    protected Chart organizationOwnerChartOfAccounts;
    protected Campus campus;
    protected Room buildingRoom;
    protected Account retirementAccount;
    protected Chart retirementChartOfAccounts;
    protected AccountingPeriod financialDocumentPostingPeriod;
    protected Building building;
    protected AccountingPeriod retirementPeriod;
    protected AssetRetirementReason retirementReason;
    protected DocumentHeader cashReceiptFinancialDocument;
    protected DocumentHeader transferOfFundsFinancialDocument;
    protected AssetCondition condition;
    protected AssetStatus inventoryStatus;
    protected List<AssetPayment> assetPayments;
    protected Person assetRepresentative;
    protected Person borrowerPerson;
    protected AssetOrganization assetOrganization;
    protected String organizationTagNumber;
    protected List<AssetRepairHistory> assetRepairHistory;
    protected AssetWarranty assetWarranty;
    protected List<AssetComponent> assetComponents;
    protected List<AssetLocation> assetLocations;
    protected List<AssetRetirementGlobalDetail> assetRetirementHistory;
    protected AssetDepreciationMethod assetPrimaryDepreciationMethod;
    protected List<AssetRetirementGlobal> retirementGlobals;
    protected ObjectSubType financialObjectSubType;
    protected AssetAcquisitionType acquisitionType;
    protected ContractsAndGrantsAgency agency;


    // Non-persisted attributes:
    protected KualiDecimal paymentTotalCost;
    protected transient AssetGlobal separateHistory;
    protected List<AssetRetirementGlobalDetail> mergeHistory;
    protected KualiDecimal federalContribution;
    protected AssetRetirementGlobalDetail retirementInfo;
    protected AssetLocation offCampusLocation;
    protected AssetLocation borrowerLocation;
    protected AssetLocation borrowerStorageLocation;
    // calculated depreciation amounts
    protected KualiDecimal accumulatedDepreciation;
    protected KualiDecimal baseAmount;
    protected KualiDecimal bookValue;
    protected KualiDecimal prevYearDepreciation;
    protected KualiDecimal yearToDateDepreciation;
    protected KualiDecimal currentMonthDepreciation;
    protected Date depreciationDateCopy;
    protected transient Integer quantity;
    protected String lookup;
    protected String assetTransferDocumentLookup;
    protected String assetMaintenanceDocumentLookup;
    protected String assetFabricationDocumentLookup;
    protected String assetCreateOrSeparateDocumentLookup;
    protected String assetPaymentDocumentLookup;
    protected String assetEquipmentLoanOrReturnDocumentLookup;
    protected String assetLocationDocumentLookup;
    protected String assetMergeOrRetirementDocumentLookup;
    protected String camsComplexMaintenanceDocumentLookup;
    protected boolean tagged;
    protected String lastInventoryDateUpdateButton;

    public AssetBase() {
        assetPayments = new ArrayList<>();
        assetRepairHistory = new ArrayList<>();
        assetComponents = new ArrayList<>();
        assetLocations = new ArrayList<>();
        assetRetirementHistory = new ArrayList<>();
        retirementGlobals = new ArrayList<>();
        mergeHistory = new ArrayList<>();
    }

    /**
     * Constructs a Asset object. Includes logic to properly set fields depending if it's creating a new asset or
     * separating an asset
     *
     * @param assetGlobal
     * @param assetGlobalDetail
     * @param separate          if it's separate an asset
     */
    public AssetBase(final AssetGlobal assetGlobal, final AssetGlobalDetail assetGlobalDetail, final boolean separate) {
        this();

        final UniversityDateService universityDateService = SpringContext.getBean(UniversityDateService.class);

        final UniversityDate date = universityDateService.getCurrentUniversityDate();

        if (assetGlobal.getFinancialDocumentPostingYear() != null) {
            setFinancialDocumentPostingYear(assetGlobal.getFinancialDocumentPostingYear());
        } else {
            setFinancialDocumentPostingYear(date.getUniversityFiscalYear());
        }
        if (assetGlobal.getFinancialDocumentPostingPeriodCode() != null) {
            setFinancialDocumentPostingPeriodCode(assetGlobal.getFinancialDocumentPostingPeriodCode());
        } else {
            setFinancialDocumentPostingPeriodCode(date.getUniversityFiscalAccountingPeriod());
        }

        setLastInventoryDate(new Timestamp(SpringContext.getBean(DateTimeService.class).getCurrentSqlDate()
                .getTime()));

        setPrimaryDepreciationMethodCode(CamsConstants.Asset.DEPRECIATION_METHOD_STRAIGHT_LINE_CODE);

        setInventoryStatusCode(assetGlobal.getInventoryStatusCode());
        setConditionCode(assetGlobal.getConditionCode());
        setAcquisitionTypeCode(assetGlobal.getAcquisitionTypeCode());
        setLandCountyName(assetGlobal.getLandCountyName());
        setLandAcreageSize(assetGlobal.getLandAcreageSize());
        setLandParcelNumber(assetGlobal.getLandParcelNumber());
        setVendorName(assetGlobal.getVendorName());
        setOrganizationOwnerAccountNumber(assetGlobal.getOrganizationOwnerAccountNumber());
        setOrganizationOwnerChartOfAccountsCode(assetGlobal.getOrganizationOwnerChartOfAccountsCode());
        setAgencyNumber(assetGlobal.getAgencyNumber());
        setCapitalAssetInServiceDate(assetGlobal.getCapitalAssetInServiceDate());
        setDepreciationDate(assetGlobal.getCapitalAssetDepreciationDate());
        setCreateDate(assetGlobal.getCreateDate());

        setCapitalAssetNumber(assetGlobalDetail.getCapitalAssetNumber());
        setCampusCode(assetGlobalDetail.getCampusCode());
        setBuildingCode(assetGlobalDetail.getBuildingCode());
        setBuildingRoomNumber(assetGlobalDetail.getBuildingRoomNumber());
        setBuildingSubRoomNumber(assetGlobalDetail.getBuildingSubRoomNumber());
        setSerialNumber(assetGlobalDetail.getSerialNumber());
        setOrganizationInventoryName(assetGlobalDetail.getOrganizationInventoryName());
        setGovernmentTagNumber(assetGlobalDetail.getGovernmentTagNumber());
        setCampusTagNumber(assetGlobalDetail.getCampusTagNumber());
        setNationalStockNumber(assetGlobalDetail.getNationalStockNumber());

        final AssetOrganization assetOrganization = new AssetOrganization();
        assetOrganization.setCapitalAssetNumber(assetGlobalDetail.getCapitalAssetNumber());
        assetOrganization.setOrganizationAssetTypeIdentifier(assetGlobalDetail.getOrganizationAssetTypeIdentifier());
        setAssetOrganization(assetOrganization);

        setActive(true);

        if (separate) {
            setRepresentativeUniversalIdentifier(assetGlobalDetail.getRepresentativeUniversalIdentifier());
            setCapitalAssetTypeCode(assetGlobalDetail.getCapitalAssetTypeCode());
            setCapitalAssetDescription(assetGlobalDetail.getCapitalAssetDescription());
            setManufacturerName(assetGlobalDetail.getManufacturerName());
            setManufacturerModelNumber(assetGlobalDetail.getManufacturerModelNumber());

            this.assetOrganization.setOrganizationText(assetGlobalDetail.getOrganizationText());
        } else {
            setRepresentativeUniversalIdentifier(assetGlobalDetail.getRepresentativeUniversalIdentifier());
            setCapitalAssetTypeCode(assetGlobal.getCapitalAssetTypeCode());
            setCapitalAssetDescription(assetGlobal.getCapitalAssetDescription());
            setManufacturerName(assetGlobal.getManufacturerName());
            setManufacturerModelNumber(assetGlobal.getManufacturerModelNumber());

            this.assetOrganization.setOrganizationText(assetGlobal.getOrganizationText());
        }
    }

    public KualiDecimal getCurrentMonthDepreciation() {
        return currentMonthDepreciation;
    }

    public void setCurrentMonthDepreciation(final KualiDecimal currentMonthDepreciation) {
        this.currentMonthDepreciation = currentMonthDepreciation;
    }

    public KualiDecimal getAccumulatedDepreciation() {

        // Calculates payment summary and depreciation summary based on available payment records
        // The value can be null due to it being used as a non-singleton on the AssetRetirementGlobal
        // page (a list of retired assets). If it were a singleton, each value would get overridden
        // by the next use on the same page
        if (accumulatedDepreciation == null) {
            SpringContext.getBean(PaymentSummaryService.class).calculateAndSetPaymentSummary((Asset) this);
        }

        return accumulatedDepreciation;
    }

    public void setAccumulatedDepreciation(final KualiDecimal accumulatedDepreciation) {
        this.accumulatedDepreciation = accumulatedDepreciation;
    }

    public KualiDecimal getBaseAmount() {
        return baseAmount;
    }

    public void setBaseAmount(final KualiDecimal baseAmount) {
        this.baseAmount = baseAmount;
    }

    public KualiDecimal getBookValue() {
        // Calculates payment summary and depreciation summary based on available payment records
        // The value can be null due to it being used as a non-singleton on the AssetRetirementGlobal
        // page (a list of retired assets). If it were a singleton, each value would get overridden
        // by the next use on the same page
        if (bookValue == null) {
            SpringContext.getBean(PaymentSummaryService.class).calculateAndSetPaymentSummary((Asset) this);
        }

        return bookValue;
    }

    public void setBookValue(final KualiDecimal bookValue) {
        this.bookValue = bookValue;
    }

    public KualiDecimal getPrevYearDepreciation() {
        return prevYearDepreciation;
    }

    public void setPrevYearDepreciation(final KualiDecimal prevYearDepreciation) {
        this.prevYearDepreciation = prevYearDepreciation;
    }

    public KualiDecimal getYearToDateDepreciation() {
        return yearToDateDepreciation;
    }

    public void setYearToDateDepreciation(final KualiDecimal yearToDateDepreciation) {
        this.yearToDateDepreciation = yearToDateDepreciation;
    }

    public Long getCapitalAssetNumber() {
        return capitalAssetNumber;
    }

    public void setCapitalAssetNumber(final Long capitalAssetNumber) {
        this.capitalAssetNumber = capitalAssetNumber;
    }

    public String getCapitalAssetDescription() {
        return capitalAssetDescription;
    }

    public void setCapitalAssetDescription(final String capitalAssetDescription) {
        this.capitalAssetDescription = capitalAssetDescription;
    }

    public String getCapitalAssetTypeCode() {
        return capitalAssetTypeCode;
    }

    public void setCapitalAssetTypeCode(final String capitalAssetTypeCode) {
        this.capitalAssetTypeCode = capitalAssetTypeCode;
    }

    public String getConditionCode() {
        return conditionCode;
    }

    public void setConditionCode(final String conditionCode) {
        this.conditionCode = conditionCode;
    }

    public Date getCreateDate() {
        return createDate;
    }

    public void setCreateDate(final Date createDate) {
        this.createDate = createDate;
    }

    public Date getReceiveDate() {
        return receiveDate;
    }

    public void setReceiveDate(final Date receiveDate) {
        this.receiveDate = receiveDate;
    }

    public Date getLoanReturnDate() {
        return loanReturnDate;
    }

    public void setLoanReturnDate(final Date loanReturnDate) {
        this.loanReturnDate = loanReturnDate;
    }

    public Date getLoanDate() {
        return loanDate;
    }

    public void setLoanDate(final Date loanDate) {
        this.loanDate = loanDate;
    }

    public Date getExpectedReturnDate() {
        return expectedReturnDate;
    }

    public void setExpectedReturnDate(final Date expectedReturnDate) {
        this.expectedReturnDate = expectedReturnDate;
    }

    public String getFinancialDocumentPostingPeriodCode() {
        return financialDocumentPostingPeriodCode;
    }

    public void setFinancialDocumentPostingPeriodCode(final String financialDocumentPostingPeriodCode) {
        this.financialDocumentPostingPeriodCode = financialDocumentPostingPeriodCode;
    }

    public Integer getFinancialDocumentPostingYear() {
        return financialDocumentPostingYear;
    }

    public void setFinancialDocumentPostingYear(final Integer financialDocumentPostingYear) {
        this.financialDocumentPostingYear = financialDocumentPostingYear;
    }

    public String getOrganizationOwnerAccountNumber() {
        return organizationOwnerAccountNumber;
    }

    public void setOrganizationOwnerAccountNumber(final String organizationOwnerAccountNumber) {
        this.organizationOwnerAccountNumber = organizationOwnerAccountNumber;
    }

    public String getOrganizationOwnerChartOfAccountsCode() {
        return organizationOwnerChartOfAccountsCode;
    }

    public void setOrganizationOwnerChartOfAccountsCode(final String organizationOwnerChartOfAccountsCode) {
        this.organizationOwnerChartOfAccountsCode = organizationOwnerChartOfAccountsCode;
    }

    public String getVendorName() {
        return vendorName;
    }

    public void setVendorName(final String vendorName) {
        this.vendorName = vendorName;
    }

    public String getAcquisitionTypeCode() {
        return acquisitionTypeCode;
    }

    public void setAcquisitionTypeCode(final String acquisitionTypeCode) {
        this.acquisitionTypeCode = acquisitionTypeCode;
    }

    public KualiDecimal getTotalCostAmount() {
        return totalCostAmount;
    }

    public void setTotalCostAmount(final KualiDecimal totalCostAmount) {
        this.totalCostAmount = totalCostAmount;
    }

    public KualiDecimal getReplacementAmount() {
        return replacementAmount;
    }

    public void setReplacementAmount(final KualiDecimal replacementAmount) {
        this.replacementAmount = replacementAmount;
    }

    public KualiDecimal getSalePrice() {
        return salePrice;
    }

    public void setSalePrice(final KualiDecimal salePrice) {
        this.salePrice = salePrice;
    }

    public KualiDecimal getEstimatedSellingPrice() {
        return estimatedSellingPrice;
    }

    public void setEstimatedSellingPrice(final KualiDecimal estimatedSellingPrice) {
        this.estimatedSellingPrice = estimatedSellingPrice;
    }

    public KualiDecimal getSalvageAmount() {
        if (salvageAmount == null) {
            salvageAmount = KualiDecimal.ZERO;
        }
        return salvageAmount;
    }

    public void setSalvageAmount(final KualiDecimal salvageAmount) {
        this.salvageAmount = salvageAmount;
    }

    public String getCampusCode() {
        return campusCode;
    }

    public void setCampusCode(final String campusCode) {
        this.campusCode = campusCode;
    }

    public String getBuildingCode() {
        return buildingCode;
    }

    public void setBuildingCode(final String buildingCode) {
        this.buildingCode = buildingCode;
    }

    public String getBuildingRoomNumber() {
        return buildingRoomNumber;
    }

    public void setBuildingRoomNumber(final String buildingRoomNumber) {
        this.buildingRoomNumber = buildingRoomNumber;
    }

    public String getBuildingSubRoomNumber() {
        return buildingSubRoomNumber;
    }

    public void setBuildingSubRoomNumber(final String buildingSubRoomNumber) {
        this.buildingSubRoomNumber = buildingSubRoomNumber;
    }

    public String getRetirementChartOfAccountsCode() {
        return retirementChartOfAccountsCode;
    }

    public void setRetirementChartOfAccountsCode(final String retirementChartOfAccountsCode) {
        this.retirementChartOfAccountsCode = retirementChartOfAccountsCode;
    }

    public String getRetirementAccountNumber() {
        return retirementAccountNumber;
    }

    public void setRetirementAccountNumber(final String retirementAccountNumber) {
        this.retirementAccountNumber = retirementAccountNumber;
    }

    public String getRetirementReasonCode() {
        return retirementReasonCode;
    }

    public void setRetirementReasonCode(final String retirementReasonCode) {
        this.retirementReasonCode = retirementReasonCode;
    }

    public String getRetirementPeriodCode() {
        return retirementPeriodCode;
    }

    public void setRetirementPeriodCode(final String retirementPeriodCode) {
        this.retirementPeriodCode = retirementPeriodCode;
    }

    public Integer getRetirementFiscalYear() {
        return retirementFiscalYear;
    }

    public void setRetirementFiscalYear(final Integer retirementFiscalYear) {
        this.retirementFiscalYear = retirementFiscalYear;
    }

    public String getCashReceiptFinancialDocumentNumber() {
        return cashReceiptFinancialDocumentNumber;
    }

    public void setCashReceiptFinancialDocumentNumber(final String cashReceiptFinancialDocumentNumber) {
        this.cashReceiptFinancialDocumentNumber = cashReceiptFinancialDocumentNumber;
    }

    public String getPrimaryDepreciationMethodCode() {
        return primaryDepreciationMethodCode;
    }

    public void setPrimaryDepreciationMethodCode(final String primaryDepreciationMethodCode) {
        this.primaryDepreciationMethodCode = primaryDepreciationMethodCode;
    }

    public Date getEstimatedFabricationCompletionDate() {
        return estimatedFabricationCompletionDate;
    }

    public void setEstimatedFabricationCompletionDate(final Date estimatedFabricationCompletionDate) {
        this.estimatedFabricationCompletionDate = estimatedFabricationCompletionDate;
    }

    public KualiDecimal getFabricationEstimatedTotalAmount() {
        return fabricationEstimatedTotalAmount;
    }

    public void setFabricationEstimatedTotalAmount(final KualiDecimal fabricationEstimatedTotalAmount) {
        this.fabricationEstimatedTotalAmount = fabricationEstimatedTotalAmount;
    }

    public String getTransferOfFundsFinancialDocumentNumber() {
        return transferOfFundsFinancialDocumentNumber;
    }

    public void setTransferOfFundsFinancialDocumentNumber(final String transferOfFundsFinancialDocumentNumber) {
        this.transferOfFundsFinancialDocumentNumber = transferOfFundsFinancialDocumentNumber;
    }

    public String getInventoryStatusCode() {
        return inventoryStatusCode;
    }

    public void setInventoryStatusCode(final String inventoryStatusCode) {
        this.inventoryStatusCode = inventoryStatusCode;
    }

    public String getCampusTagNumber() {
        return campusTagNumber;
    }

    public void setCampusTagNumber(final String campusTagNumber) {
        this.campusTagNumber = campusTagNumber;
    }

    public Timestamp getLastInventoryDate() {
        return lastInventoryDate;
    }

    public void setLastInventoryDate(final Timestamp lastInventoryDate) {
        this.lastInventoryDate = lastInventoryDate;
    }

    public String getOrganizationInventoryName() {
        return organizationInventoryName;
    }

    public void setOrganizationInventoryName(final String organizationInventoryName) {
        this.organizationInventoryName = organizationInventoryName;
    }

    public String getOldTagNumber() {
        return oldTagNumber;
    }

    public void setOldTagNumber(final String oldTagNumber) {
        this.oldTagNumber = oldTagNumber;
    }

    public String getManufacturerName() {
        return manufacturerName;
    }

    public void setManufacturerName(final String manufacturerName) {
        this.manufacturerName = manufacturerName;
    }

    public String getManufacturerModelNumber() {
        return manufacturerModelNumber;
    }

    public void setManufacturerModelNumber(final String manufacturerModelNumber) {
        this.manufacturerModelNumber = manufacturerModelNumber;
    }

    public String getSerialNumber() {
        return serialNumber;
    }

    public void setSerialNumber(final String serialNumber) {
        this.serialNumber = serialNumber;
    }

    public String getRepresentativeUniversalIdentifier() {
        return representativeUniversalIdentifier;
    }

    public void setRepresentativeUniversalIdentifier(final String representativeUniversalIdentifier) {
        this.representativeUniversalIdentifier = representativeUniversalIdentifier;
    }

    public String getAgencyNumber() {
        return agencyNumber;
    }

    public void setAgencyNumber(final String agencyNumber) {
        this.agencyNumber = agencyNumber;
    }

    public String getCampusPoliceDepartmentCaseNumber() {
        return campusPoliceDepartmentCaseNumber;
    }

    public void setCampusPoliceDepartmentCaseNumber(final String campusPoliceDepartmentCaseNumber) {
        this.campusPoliceDepartmentCaseNumber = campusPoliceDepartmentCaseNumber;
    }

    public String getInventoryScannedCode() {
        return inventoryScannedCode;
    }

    public void setInventoryScannedCode(final String inventoryScannedCode) {
        this.inventoryScannedCode = inventoryScannedCode;
    }

    public Date getCapitalAssetInServiceDate() {
        return capitalAssetInServiceDate;
    }

    public void setCapitalAssetInServiceDate(final Date capitalAssetInServiceDate) {
        this.capitalAssetInServiceDate = capitalAssetInServiceDate;
    }

    public String getGovernmentTagNumber() {
        return governmentTagNumber;
    }

    public void setGovernmentTagNumber(final String governmentTagNumber) {
        this.governmentTagNumber = governmentTagNumber;
    }

    public String getNationalStockNumber() {
        return nationalStockNumber;
    }

    public void setNationalStockNumber(final String nationalStockNumber) {
        this.nationalStockNumber = nationalStockNumber;
    }

    public Integer getLandAcreageSize() {
        return landAcreageSize;
    }

    public void setLandAcreageSize(final Integer landAcreageSize) {
        this.landAcreageSize = landAcreageSize;
    }

    public String getLandCountyName() {
        return landCountyName;
    }

    public void setLandCountyName(final String landCountyName) {
        this.landCountyName = landCountyName;
    }

    public String getLandParcelNumber() {
        return landParcelNumber;
    }

    public void setLandParcelNumber(final String landParcelNumber) {
        this.landParcelNumber = landParcelNumber;
    }

    public Date getDepreciationDate() {
        return depreciationDate;
    }

    public void setDepreciationDate(final Date depreciationDate) {
        this.depreciationDate = depreciationDate;
    }

    public AssetType getCapitalAssetType() {
        return capitalAssetType;
    }

    @Deprecated
    public void setCapitalAssetType(final AssetType capitalAssetType) {
        this.capitalAssetType = capitalAssetType;
    }

    public Account getOrganizationOwnerAccount() {
        return organizationOwnerAccount;
    }

    @Deprecated
    public void setOrganizationOwnerAccount(final Account organizationOwnerAccount) {
        this.organizationOwnerAccount = organizationOwnerAccount;
    }

    public Chart getOrganizationOwnerChartOfAccounts() {
        return organizationOwnerChartOfAccounts;
    }

    @Deprecated
    public void setOrganizationOwnerChartOfAccounts(final Chart organizationOwnerChartOfAccounts) {
        this.organizationOwnerChartOfAccounts = organizationOwnerChartOfAccounts;
    }

    public Campus getCampus() {
        return campus;
    }

    @Deprecated
    public void setCampus(final Campus campus) {
        this.campus = campus;
    }

    public Room getBuildingRoom() {
        return buildingRoom;
    }

    @Deprecated
    public void setBuildingRoom(final Room buildingRoom) {
        this.buildingRoom = buildingRoom;
    }

    public Account getRetirementAccount() {
        return retirementAccount;
    }

    @Deprecated
    public void setRetirementAccount(final Account retirementAccount) {
        this.retirementAccount = retirementAccount;
    }

    public Chart getRetirementChartOfAccounts() {
        return retirementChartOfAccounts;
    }

    @Deprecated
    public void setRetirementChartOfAccounts(final Chart retirementChartOfAccounts) {
        this.retirementChartOfAccounts = retirementChartOfAccounts;
    }

    public Building getBuilding() {
        return building;
    }

    @Deprecated
    public void setBuilding(final Building building) {
        this.building = building;
    }

    public DocumentHeader getCashReceiptFinancialDocument() {
        return cashReceiptFinancialDocument;
    }

    @Deprecated
    public void setCashReceiptFinancialDocument(final DocumentHeader cashReceiptFinancialDocument) {
        this.cashReceiptFinancialDocument = cashReceiptFinancialDocument;
    }

    public AccountingPeriod getRetirementPeriod() {
        return retirementPeriod;
    }

    @Deprecated
    public void setRetirementPeriod(final AccountingPeriod retirementPeriod) {
        this.retirementPeriod = retirementPeriod;
    }

    public AssetRetirementReason getRetirementReason() {
        return retirementReason;
    }

    @Deprecated
    public void setRetirementReason(final AssetRetirementReason retirementReason) {
        this.retirementReason = retirementReason;
    }

    public DocumentHeader getTransferOfFundsFinancialDocument() {
        return transferOfFundsFinancialDocument;
    }

    @Deprecated
    public void setTransferOfFundsFinancialDocument(final DocumentHeader transferOfFundsFinancialDocument) {
        this.transferOfFundsFinancialDocument = transferOfFundsFinancialDocument;
    }

    public AccountingPeriod getFinancialDocumentPostingPeriod() {
        return financialDocumentPostingPeriod;
    }

    @Deprecated
    public void setFinancialDocumentPostingPeriod(final AccountingPeriod financialDocumentPostingPeriod) {
        this.financialDocumentPostingPeriod = financialDocumentPostingPeriod;
    }

    public AssetCondition getCondition() {
        return condition;
    }

    @Deprecated
    public void setCondition(final AssetCondition condition) {
        this.condition = condition;
    }

    public boolean isActive() {
        return active;
    }

    public void setActive(final boolean active) {
        this.active = active;
    }

    public List<AssetPayment> getAssetPayments() {
        return assetPayments;
    }

    public void setAssetPayments(final List<AssetPayment> assetPayments) {
        this.assetPayments = assetPayments;
    }

    public AssetStatus getInventoryStatus() {
        return inventoryStatus;
    }

    public void setInventoryStatus(final AssetStatus inventoryStatus) {
        this.inventoryStatus = inventoryStatus;
    }

    public Person getAssetRepresentative() {
        assetRepresentative = SpringContext.getBean(org.kuali.kfs.kim.api.identity.PersonService.class)
                .updatePersonIfNecessary(representativeUniversalIdentifier, assetRepresentative);
        return assetRepresentative;
    }

    @Deprecated
    public void setAssetRepresentative(final Person assetRepresentative) {
        this.assetRepresentative = assetRepresentative;
    }

    public Person getBorrowerPerson() {
        if (ObjectUtils.isNotNull(borrowerLocation)) {
            borrowerPerson = SpringContext.getBean(org.kuali.kfs.kim.api.identity.PersonService.class)
                    .updatePersonIfNecessary(borrowerLocation.getAssetLocationContactIdentifier(), borrowerPerson);
        }
        return borrowerPerson;
    }

    public void setBorrowerPerson(final Person borrowerPerson) {
        this.borrowerPerson = borrowerPerson;
    }

    public AssetOrganization getAssetOrganization() {
        return assetOrganization;
    }

    public void setAssetOrganization(final AssetOrganization assetOrganization) {
        this.assetOrganization = assetOrganization;
    }

    public String getOrganizationTagNumber() {
        return organizationTagNumber;
    }

    public void setOrganizationTagNumber(final String organizationTagNumber) {
        this.organizationTagNumber = organizationTagNumber;
    }

    public List<AssetRepairHistory> getAssetRepairHistory() {
        return assetRepairHistory;
    }

    public void setAssetRepairHistory(final List<AssetRepairHistory> assetRepairHistory) {
        this.assetRepairHistory = assetRepairHistory;
    }

    public AssetWarranty getAssetWarranty() {
        return assetWarranty;
    }

    public void setAssetWarranty(final AssetWarranty assetWarranty) {
        this.assetWarranty = assetWarranty;
    }

    public KualiDecimal getPaymentTotalCost() {
        return paymentTotalCost;
    }

    public void setPaymentTotalCost(final KualiDecimal paymentTotalCost) {
        this.paymentTotalCost = paymentTotalCost;
    }

    public List<AssetComponent> getAssetComponents() {
        return assetComponents;
    }

    public void setAssetComponents(final List<AssetComponent> assetComponents) {
        this.assetComponents = assetComponents;
    }

    public KualiDecimal getFederalContribution() {
        return federalContribution;
    }

    public void setFederalContribution(final KualiDecimal federalContribution) {
        this.federalContribution = federalContribution;
    }

    public AssetRetirementGlobalDetail getRetirementInfo() {
        return retirementInfo;
    }

    public void setRetirementInfo(final AssetRetirementGlobalDetail retirementInfo) {
        this.retirementInfo = retirementInfo;
    }

    public List<AssetLocation> getAssetLocations() {
        return assetLocations;
    }

    public void setAssetLocations(final List<AssetLocation> assetLocations) {
        this.assetLocations = assetLocations;
    }

    public AssetLocation getOffCampusLocation() {
        return offCampusLocation;
    }

    public void setOffCampusLocation(final AssetLocation offCampusLocation) {
        this.offCampusLocation = offCampusLocation;
    }

    public List<AssetRetirementGlobalDetail> getAssetRetirementHistory() {
        return assetRetirementHistory;
    }

    public void setAssetRetirementHistory(final List<AssetRetirementGlobalDetail> assetRetirementHistory) {
        this.assetRetirementHistory = assetRetirementHistory;
    }

    public AssetDepreciationMethod getAssetPrimaryDepreciationMethod() {
        return assetPrimaryDepreciationMethod;
    }

    public void setAssetPrimaryDepreciationMethod(final AssetDepreciationMethod assetPrimaryDepreciationMethod) {
        this.assetPrimaryDepreciationMethod = assetPrimaryDepreciationMethod;
    }

    public Integer getFabricationEstimatedRetentionYears() {
        return fabricationEstimatedRetentionYears;
    }

    public void setFabricationEstimatedRetentionYears(final Integer fabricationEstimatedRetentionYears) {
        this.fabricationEstimatedRetentionYears = fabricationEstimatedRetentionYears;
    }

    public List<AssetRetirementGlobal> getRetirementGlobals() {
        return retirementGlobals;
    }

    public void setRetirementGlobals(final List<AssetRetirementGlobal> retirementGlobals) {
        this.retirementGlobals = retirementGlobals;
    }

    public AssetGlobal getSeparateHistory() {
        return separateHistory;
    }

    public void setSeparateHistory(final AssetGlobal separateHistory) {
        this.separateHistory = separateHistory;
    }

    public List<AssetRetirementGlobalDetail> getMergeHistory() {
        return mergeHistory;
    }

    public void setMergeHistory(final List<AssetRetirementGlobalDetail> mergeHistory) {
        this.mergeHistory = mergeHistory;
    }

    public String getFinancialObjectSubTypeCode() {
        return financialObjectSubTypeCode;
    }

    public void setFinancialObjectSubTypeCode(final String financialObjectSubTypeCode) {
        this.financialObjectSubTypeCode = financialObjectSubTypeCode;
    }

    public ObjectSubType getFinancialObjectSubType() {
        return financialObjectSubType;
    }

    @Deprecated
    public void setFinancialObjectSubType(final ObjectSubType financialObjectSubType) {
        this.financialObjectSubType = financialObjectSubType;
    }

    public Date getDepreciationDateCopy() {
        return depreciationDate;
    }

    @Deprecated
    public void setDepreciationDateCopy(final Date depreciationDateCopy) {
        this.depreciationDateCopy = depreciationDateCopy;
    }

    public AssetAcquisitionType getAcquisitionType() {
        return acquisitionType;
    }

    public void setAcquisitionType(final AssetAcquisitionType acquisitionType) {
        this.acquisitionType = acquisitionType;
    }

    public ContractsAndGrantsAgency getAgency() {
        agency = SpringContext.getBean(KualiModuleService.class).getResponsibleModuleService(
                ContractsAndGrantsAgency.class).retrieveExternalizableBusinessObjectIfNecessary(this, agency, "agency");
        return agency;
    }

    /**
     * Technically this is obsolete but necessary because MaintenanceDocumentBase.populateXmlDocumentContentsFromMaintainables has
     * the following hack:<br>
     * ObjectUtils.materializeAllSubObjects(oldBo); // hack to resolve XStream not dealing well with Proxies<br>
     * so as long as that is there we need this setter otherwise a NoSuchMethodException occurs.
     *
     * @deprecated
     */
    @Deprecated
    public void setAgency(final ContractsAndGrantsAgency agency) {
        this.agency = agency;
    }

    public Integer getQuantity() {
        return quantity;
    }

    public boolean isTagged() {
        return tagged;
    }

    public void setTagged() {
        tagged = StringUtils.isNotBlank(campusTagNumber);
    }

    public AssetLocation getBorrowerLocation() {
        return borrowerLocation;
    }

    public void setBorrowerLocation(final AssetLocation borrowerLocation) {
        this.borrowerLocation = borrowerLocation;
    }

    public AssetLocation getBorrowerStorageLocation() {
        return borrowerStorageLocation;
    }

    public void setBorrowerStorageLocation(final AssetLocation borrowerStorageLocation) {
        this.borrowerStorageLocation = borrowerStorageLocation;
    }

    public String getLastInventoryDateUpdateButton() {
        return lastInventoryDateUpdateButton;
    }

    public void setLastInventoryDateUpdateButton(final String lastInventoryDateUpdateButton) {
        this.lastInventoryDateUpdateButton = lastInventoryDateUpdateButton;
    }

    /**
     * @return the link for payment lookup
     */
    public String getLookup() {
        if (getCapitalAssetNumber() == null) {
            return "";
        }

        final String basePath = SpringContext.getBean(ConfigurationService.class)
                .getPropertyValueAsString(KFSConstants.APPLICATION_URL_KEY);

        final Map<String, String> params = new HashMap<>();
        params.put(KRADConstants.DISPATCH_REQUEST_PARAMETER, KRADConstants.SEARCH_METHOD);
        params.put(KRADConstants.DOC_FORM_KEY, "88888888");
        params.put(KRADConstants.HIDE_LOOKUP_RETURN_LINK, "true");
        params.put(CamsPropertyConstants.Asset.CAPITAL_ASSET_NUMBER, getCapitalAssetNumber().toString());
        params.put(KRADConstants.RETURN_LOCATION_PARAMETER, "portal.do");
        params.put(KRADConstants.BUSINESS_OBJECT_CLASS_ATTRIBUTE, AssetPayment.class.getName());

        return UrlFactory.parameterizeUrl(basePath + "/" + KFSConstants.LOOKUP_ACTION, params);
    }

    protected String getUrlForAssetDocumentLookup(final String documentTypeName) {
        if (getCapitalAssetNumber() == null) {
            return "";
        }
        final Map<String, String> params = new HashMap<>();
        params.put(KFSPropertyConstants.DOCUMENT_TYPE_NAME, documentTypeName);
        params.put(KewApiConstants.DOCUMENT_ATTRIBUTE_FIELD_PREFIX + CamsPropertyConstants.Asset.CAPITAL_ASSET_NUMBER,
                getCapitalAssetNumber().toString());
        return UrlFactory.parameterizeUrl(SpringContext.getBean(ConfigurationService.class).getPropertyValueAsString(
                KRADConstants.WORKFLOW_DOCUMENTSEARCH_URL_KEY), params);
    }

    /**
     * @return the link for asset transfer document lookup for given capital asset number.
     */
    public String getAssetTransferDocumentLookup() {
        return getUrlForAssetDocumentLookup(CamsConstants.DocumentTypeName.ASSET_TRANSFER);
    }

    /**
     * @return the link for asset maintenance document lookup for given capital asset number.
     */
    public String getAssetMaintenanceDocumentLookup() {
        return getUrlForAssetDocumentLookup(CamsConstants.DocumentTypeName.ASSET_EDIT);
    }

    /**
     * @return the link for asset fabrication document lookup for given capital asset number.
     */
    public String getAssetFabricationDocumentLookup() {
        return getUrlForAssetDocumentLookup(CamsConstants.DocumentTypeName.ASSET_FABRICATION);
    }

    /**
     * @return the link for asset create or separate global document lookup for given capital asset number.
     */
    public String getAssetCreateOrSeparateDocumentLookup() {
        return getUrlForAssetDocumentLookup(CamsConstants.DocumentTypeName.ASSET_ADD_GLOBAL);
    }

    /**
     * @return the link for asset payment document lookup for given capital asset number.
     */
    public String getAssetPaymentDocumentLookup() {
        return getUrlForAssetDocumentLookup(CamsConstants.DocumentTypeName.ASSET_PAYMENT);
    }

    /**
     * @return the link for asset equipment loan or return document lookup for given capital asset number.
     */
    public String getAssetEquipmentLoanOrReturnDocumentLookup() {
        return getUrlForAssetDocumentLookup(CamsConstants.DocumentTypeName.ASSET_EQUIPMENT_LOAN_OR_RETURN);
    }

    /**
     * @return the link for asset location global document lookup for given capital asset number.
     */
    public String getAssetLocationDocumentLookup() {
        return getUrlForAssetDocumentLookup(CamsConstants.DocumentTypeName.ASSET_LOCATION_GLOBAL);
    }

    /**
     * @return the link for asset retirement or merge document lookup for given capital asset number.
     */
    public String getAssetMergeOrRetirementDocumentLookup() {
        return getUrlForAssetDocumentLookup(CamsConstants.DocumentTypeName.ASSET_RETIREMENT_GLOBAL);
    }

    public String getCamsComplexMaintenanceDocumentLookup() {
        return getUrlForAssetDocumentLookup(CamsConstants.DocumentTypeName.COMPLEX_MAINTENANCE_DOC_BASE);
    }

    /**
     * override this method so we can remove the offcampus location
     */
    @Override
    public List<Collection<PersistableBusinessObject>> buildListOfDeletionAwareLists() {
        final List<Collection<PersistableBusinessObject>> managedLists = new ArrayList<>();
        managedLists.add(new ArrayList<>(getAssetLocations()));
        return managedLists;
    }

}
