/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cam.businessobject;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.coa.businessobject.Account;
import org.kuali.kfs.coa.businessobject.AccountingPeriod;
import org.kuali.kfs.coa.businessobject.Chart;
import org.kuali.kfs.coa.service.AccountingPeriodService;
import org.kuali.kfs.core.api.datetime.DateTimeService;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.integration.cg.ContractsAndGrantsAgency;
import org.kuali.kfs.kew.api.WorkflowDocument;
import org.kuali.kfs.kew.api.WorkflowDocumentFactory;
import org.kuali.kfs.kim.impl.identity.Person;
import org.kuali.kfs.kns.maintenance.rules.MaintenanceDocumentRuleBase;
import org.kuali.kfs.krad.bo.GlobalBusinessObject;
import org.kuali.kfs.krad.bo.GlobalBusinessObjectDetail;
import org.kuali.kfs.krad.bo.PersistableBusinessObject;
import org.kuali.kfs.krad.bo.PersistableBusinessObjectBase;
import org.kuali.kfs.krad.service.KualiModuleService;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.cam.CamsKeyConstants;
import org.kuali.kfs.module.cam.CamsPropertyConstants;
import org.kuali.kfs.module.cam.document.service.AssetGlobalService;
import org.kuali.kfs.sys.businessobject.DocumentHeader;
import org.kuali.kfs.sys.businessobject.GeneralLedgerPendingEntry;
import org.kuali.kfs.sys.context.SpringContext;

import java.sql.Date;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

public class AssetGlobal extends PersistableBusinessObjectBase implements GlobalBusinessObject {

    private static final Logger LOG = LogManager.getLogger();

    private String documentNumber;
    private String acquisitionTypeCode;
    private String capitalAssetDescription;
    private String inventoryStatusCode;
    private String conditionCode;
    private String capitalAssetTypeCode;
    private String manufacturerName;
    private String manufacturerModelNumber;
    private KualiDecimal totalCostAmount;
    private String landCountyName;
    private Integer landAcreageSize;
    private String landParcelNumber;
    private String vendorName;
    private String organizationText;
    private Date createDate;
    private Date capitalAssetInServiceDate;
    private Date capitalAssetDepreciationDate;
    private String representativeUniversalIdentifier;
    private String organizationOwnerChartOfAccountsCode;
    private String organizationOwnerAccountNumber;
    private String agencyNumber;
    private Integer financialDocumentNextLineNumber;
    private Asset separateSourceCapitalAsset;
    private Integer separateSourcePaymentSequenceNumber;
    private boolean capitalAssetBuilderOriginIndicator;

    protected String financialDocumentPostingPeriodCode;
    protected Integer financialDocumentPostingYear;

    // Not Persisted
    private AccountingPeriod accountingPeriod;
    private Date lastInventoryDate;
    private ContractsAndGrantsAgency agency;
    private Person assetRepresentative;
    private AssetType capitalAssetType;
    private AssetCondition assetCondition;
    private AssetStatus inventoryStatus;
    private List<AssetGlobalDetail> assetGlobalDetails;
    private List<AssetGlobalDetail> assetSharedDetails;
    private List<AssetPaymentDetail> assetPaymentDetails;
    private AssetAcquisitionType acquisitionType;
    private Chart organizationOwnerChartOfAccounts;
    private Account organizationOwnerAccount;

    // field is here so that AssetLookupableHelperServiceImpl can pass action information
    private String financialDocumentTypeCode;
    private Long separateSourceCapitalAssetNumber;

    // Calculate Equal Source Amounts button
    private String calculateEqualSourceAmountsButton;

    // calculate remaining source amount
    private KualiDecimal separateSourceRemainingAmount;
    private KualiDecimal separateSourceTotalAmount;
    private String calculateSeparateSourceRemainingAmountButton;

    private List<GeneralLedgerPendingEntry> generalLedgerPendingEntries;
    private DocumentHeader documentHeader;
    private KualiDecimal totalAssetPaymentAmount;

    //
    private KualiDecimal minAssetTotalAmount;
    private KualiDecimal maxAssetTotalAmount;

    protected static transient AccountingPeriodService accountingPeriodService;

    public AssetGlobal() {
        assetGlobalDetails = new ArrayList<>();
        assetSharedDetails = new ArrayList<>();
        assetPaymentDetails = new ArrayList<>();
        generalLedgerPendingEntries = new ArrayList<>();

        // Guard clause is necessary for unit tests
        if (SpringContext.isInitialized()) {
            // Handle the situation where the DB does not contain the current Fiscal Year.
            final DateTimeService dateTimeService = SpringContext.getBean(DateTimeService.class);
            final Date currentDate = dateTimeService.getCurrentSqlDate();
            AccountingPeriod accountingPeriod = null;
            try {
                accountingPeriod = getAccountingPeriodService().getByDate(currentDate);
            } catch (final Exception e) {
                LOG.warn("AssetGlobal() - Could not retrieve the AccountingPeriod based on the date; run "
                         + "fiscalYearMakerJob : currentDate={}", currentDate);
            }
            setAccountingPeriod(accountingPeriod);
        }
    }

    @Override
    public String getDocumentNumber() {
        return documentNumber;
    }

    @Override
    public void setDocumentNumber(final String documentNumber) {
        this.documentNumber = documentNumber;
    }

    public String getAcquisitionTypeCode() {
        return acquisitionTypeCode;
    }

    public void setAcquisitionTypeCode(final String acquisitionTypeCode) {
        this.acquisitionTypeCode = acquisitionTypeCode;
    }

    public String getCapitalAssetDescription() {
        return capitalAssetDescription;
    }

    public void setCapitalAssetDescription(final String capitalAssetDescription) {
        this.capitalAssetDescription = capitalAssetDescription;
    }

    public String getInventoryStatusCode() {
        return inventoryStatusCode;
    }

    public void setInventoryStatusCode(final String inventoryStatusCode) {
        this.inventoryStatusCode = inventoryStatusCode;
    }

    public String getConditionCode() {
        return conditionCode;
    }

    public void setConditionCode(final String conditionCode) {
        this.conditionCode = conditionCode;
    }

    public String getCapitalAssetTypeCode() {
        return capitalAssetTypeCode;
    }

    public void setCapitalAssetTypeCode(final String capitalAssetTypeCode) {
        this.capitalAssetTypeCode = capitalAssetTypeCode;
    }

    public String getManufacturerName() {
        return manufacturerName;
    }

    public void setManufacturerName(final String manufacturerName) {
        this.manufacturerName = manufacturerName;
    }

    public String getManufacturerModelNumber() {
        return manufacturerModelNumber;
    }

    public void setManufacturerModelNumber(final String manufacturerModelNumber) {
        this.manufacturerModelNumber = manufacturerModelNumber;
    }

    public KualiDecimal getTotalCostAmount() {
        return totalCostAmount;
    }

    public void setTotalCostAmount(final KualiDecimal totalCostAmount) {
        this.totalCostAmount = totalCostAmount;
    }

    public String getLandCountyName() {
        return landCountyName;
    }

    public void setLandCountyName(final String landCountyName) {
        this.landCountyName = landCountyName;
    }

    public Integer getLandAcreageSize() {
        return landAcreageSize;
    }

    public void setLandAcreageSize(final Integer landAcreageSize) {
        this.landAcreageSize = landAcreageSize;
    }

    public String getLandParcelNumber() {
        return landParcelNumber;
    }

    public void setLandParcelNumber(final String landParcelNumber) {
        this.landParcelNumber = landParcelNumber;
    }

    public String getVendorName() {
        return vendorName;
    }

    public void setVendorName(final String vendorName) {
        this.vendorName = vendorName;
    }

    public String getOrganizationText() {
        return organizationText;
    }

    public void setOrganizationText(final String organizationText) {
        this.organizationText = organizationText;
    }

    public Date getCreateDate() {
        if (createDate != null) {
            return createDate;
        } else {
            return SpringContext.getBean(DateTimeService.class).getCurrentSqlDate();
        }
    }

    public void setCreateDate(final Date createDate) {
        this.createDate = createDate;
    }

    public Date getCapitalAssetInServiceDate() {
        return capitalAssetInServiceDate;
    }

    public void setCapitalAssetInServiceDate(final Date capitalAssetInServiceDate) {
        this.capitalAssetInServiceDate = capitalAssetInServiceDate;
    }

    public Date getCapitalAssetDepreciationDate() {
        return capitalAssetDepreciationDate;
    }

    public void setCapitalAssetDepreciationDate(final Date capitalAssetDepreciationDate) {
        this.capitalAssetDepreciationDate = capitalAssetDepreciationDate;
    }

    public String getAgencyNumber() {
        return agencyNumber;
    }

    public void setAgencyNumber(final String agencyNumber) {
        this.agencyNumber = agencyNumber;
    }

    public String getOrganizationOwnerAccountNumber() {
        return organizationOwnerAccountNumber;
    }

    public void setOrganizationOwnerAccountNumber(final String organizationOwnerAccountNumber) {
        this.organizationOwnerAccountNumber = organizationOwnerAccountNumber;
    }

    public String getOrganizationOwnerChartOfAccountsCode() {
        return organizationOwnerChartOfAccountsCode;
    }

    public void setOrganizationOwnerChartOfAccountsCode(final String organizationOwnerChartOfAccountsCode) {
        this.organizationOwnerChartOfAccountsCode = organizationOwnerChartOfAccountsCode;
    }

    public String getRepresentativeUniversalIdentifier() {
        return representativeUniversalIdentifier;
    }

    public void setRepresentativeUniversalIdentifier(final String representativeUniversalIdentifier) {
        this.representativeUniversalIdentifier = representativeUniversalIdentifier;
    }

    public AssetCondition getAssetCondition() {
        return assetCondition;
    }

    public void setAssetCondition(final AssetCondition assetCondition) {
        this.assetCondition = assetCondition;
    }

    public AssetType getCapitalAssetType() {
        return capitalAssetType;
    }

    public void setCapitalAssetType(final AssetType capitalAssetType) {
        this.capitalAssetType = capitalAssetType;
    }

    public AssetStatus getInventoryStatus() {
        return inventoryStatus;
    }

    public void setInventoryStatus(final AssetStatus inventoryStatus) {
        this.inventoryStatus = inventoryStatus;
    }

    public Account getOrganizationOwnerAccount() {
        return organizationOwnerAccount;
    }

    public void setOrganizationOwnerAccount(final Account organizationOwnerAccount) {
        this.organizationOwnerAccount = organizationOwnerAccount;
    }

    public Chart getOrganizationOwnerChartOfAccounts() {
        return organizationOwnerChartOfAccounts;
    }

    public void setOrganizationOwnerChartOfAccounts(final Chart organizationOwnerChartOfAccounts) {
        this.organizationOwnerChartOfAccounts = organizationOwnerChartOfAccounts;
    }

    public List<AssetGlobalDetail> getAssetGlobalDetails() {
        return assetGlobalDetails;
    }

    public void setAssetGlobalDetails(final List<AssetGlobalDetail> assetGlobalDetails) {
        this.assetGlobalDetails = assetGlobalDetails;
    }

    public List<AssetPaymentDetail> getAssetPaymentDetails() {
        return assetPaymentDetails;
    }

    public void setAssetPaymentDetails(final List<AssetPaymentDetail> assetPaymentDetails) {
        this.assetPaymentDetails = assetPaymentDetails;
    }

    @Override
    public List<PersistableBusinessObject> generateDeactivationsToPersist() {
        return null;
    }

    /**
     * @return a list of Assets to Update and/or Add. Applicable to both create new and separate.
     */
    @Override
    public List<PersistableBusinessObject> generateGlobalChangesToPersist() {
        final List<PersistableBusinessObject> persistables;

        final AssetGlobalService assetGlobalService = SpringContext.getBean(AssetGlobalService.class);

        if (assetGlobalService.isAssetSeparate(this)) {
            persistables = assetGlobalService.getSeparateAssets(this);
        } else {
            persistables = assetGlobalService.getCreateNewAssets(this);
        }

        return persistables;
    }

    @Override
    public boolean isPersistable() {
        // This is needed because otherwise we could get DB Constraint violation error if user tries
        // to save Asset Global document with a non existing, inactive or expired account.
        refreshReferenceObject(CamsPropertyConstants.AssetGlobal.ORGANIZATION_OWNER_ACCOUNT);
        final AssetGlobalService assetGlobalService = SpringContext.getBean(AssetGlobalService.class);
        if (!assetGlobalService.isAssetSeparate(this)) {
            final Account organizationOwnerAccount = getOrganizationOwnerAccount();
            if (ObjectUtils.isNull(organizationOwnerAccount) || !organizationOwnerAccount.isActive()
                    || organizationOwnerAccount.isExpired()) {
                GlobalVariables.getMessageMap().putErrorWithoutFullErrorPath(
                        MaintenanceDocumentRuleBase.MAINTAINABLE_ERROR_PREFIX +
                                CamsPropertyConstants.AssetGlobal.ORGANIZATION_OWNER_ACCOUNT_NUMBER,
                        CamsKeyConstants.ORGANIZATION_OWNER_ACCOUNT_INVALID);
                return false;
            }
        }
        return true;
    }

    @Override
    public List<? extends GlobalBusinessObjectDetail> getAllDetailObjects() {
        return getAssetGlobalDetails();
    }

    public Date getLastInventoryDate() {
        return lastInventoryDate;
    }

    public void setLastInventoryDate(final Date lastInventoryDate) {
        this.lastInventoryDate = lastInventoryDate;
    }

    public String getFinancialDocumentPostingPeriodCode() {
        return financialDocumentPostingPeriodCode;
    }

    public void setFinancialDocumentPostingPeriodCode(final String financialDocumentPostingPeriodCode) {
        this.financialDocumentPostingPeriodCode = financialDocumentPostingPeriodCode;
    }

    public Integer getFinancialDocumentPostingYear() {
        return financialDocumentPostingYear;
    }

    public void setFinancialDocumentPostingYear(final Integer financialDocumentPostingYear) {
        this.financialDocumentPostingYear = financialDocumentPostingYear;
    }

    public Integer getFinancialDocumentNextLineNumber() {
        return financialDocumentNextLineNumber;
    }

    public void setFinancialDocumentNextLineNumber(final Integer financialDocumentNextLineNumber) {
        this.financialDocumentNextLineNumber = financialDocumentNextLineNumber;
    }

    public Integer incrementFinancialDocumentLineNumber() {
        if (financialDocumentNextLineNumber == null) {
            financialDocumentNextLineNumber = 0;
        }
        financialDocumentNextLineNumber += 1;
        return financialDocumentNextLineNumber;
    }

    public List<AssetGlobalDetail> getAssetSharedDetails() {
        return assetSharedDetails;
    }

    public void setAssetSharedDetails(final List<AssetGlobalDetail> assetSharedDetails) {
        this.assetSharedDetails = assetSharedDetails;
    }

    public AssetAcquisitionType getAcquisitionType() {
        return acquisitionType;
    }

    public void setAcquisitionType(final AssetAcquisitionType acquisitionType) {
        this.acquisitionType = acquisitionType;
    }

    public String getFinancialDocumentTypeCode() {
        return financialDocumentTypeCode;
    }

    public void setFinancialDocumentTypeCode(final String financialDocumentTypeCode) {
        this.financialDocumentTypeCode = financialDocumentTypeCode;
    }

    public Person getAssetRepresentative() {
        assetRepresentative = SpringContext.getBean(org.kuali.kfs.kim.api.identity.PersonService.class)
                .updatePersonIfNecessary(representativeUniversalIdentifier, assetRepresentative);
        return assetRepresentative;
    }

    public void setAssetRepresentative(final Person assetRepresentative) {
        this.assetRepresentative = assetRepresentative;
    }

    public ContractsAndGrantsAgency getAgency() {
        agency = SpringContext.getBean(KualiModuleService.class)
                .getResponsibleModuleService(ContractsAndGrantsAgency.class)
                .retrieveExternalizableBusinessObjectIfNecessary(this, agency, "agency");
        return agency;
    }

    /**
     * Technically this is obsolete but necessary because
     * MaintenanceDocumentBase.populateXmlDocumentContentsFromMaintainables has the following hack:<br>
     * ObjectUtils.materializeAllSubObjects(oldBo); // hack to resolve XStream not dealing well with Proxies<br>
     * so as long as that is there we need this setter otherwise a NoSuchMethodException occurs.
     */
    public void setAgency(final ContractsAndGrantsAgency agency) {
        this.agency = agency;
    }

    public List<GeneralLedgerPendingEntry> getGeneralLedgerPendingEntries() {
        return generalLedgerPendingEntries;
    }

    public void setGeneralLedgerPendingEntries(final List<GeneralLedgerPendingEntry> generalLedgerPendingEntries) {
        this.generalLedgerPendingEntries = generalLedgerPendingEntries;
    }

    public Asset getSeparateSourceCapitalAsset() {
        return separateSourceCapitalAsset;
    }

    public void setSeparateSourceCapitalAsset(final Asset separateSourceCapitalAsset) {
        this.separateSourceCapitalAsset = separateSourceCapitalAsset;
    }

    public Long getSeparateSourceCapitalAssetNumber() {
        return separateSourceCapitalAssetNumber;
    }

    public void setSeparateSourceCapitalAssetNumber(final Long separateSourceCapitalAssetNumber) {
        this.separateSourceCapitalAssetNumber = separateSourceCapitalAssetNumber;
    }

    public Integer getSeparateSourcePaymentSequenceNumber() {
        return separateSourcePaymentSequenceNumber;
    }

    public void setSeparateSourcePaymentSequenceNumber(final Integer separateSourcePaymentSequenceNumber) {
        this.separateSourcePaymentSequenceNumber = separateSourcePaymentSequenceNumber;
    }

    public DocumentHeader getDocumentHeader() {
        return documentHeader;
    }

    public void setDocumentHeader(final DocumentHeader documentHeader) {
        this.documentHeader = documentHeader;
    }

    /**
     * Small workaround to avoid KualiInquirableImpl.getInquiryUrl having think it needs to construct an inquiry url
     * for this date. This only returns a date if this is a separate.
     *
     * @return
     */
    public Date getSeparateDocumentHeaderFinalDate() {
        if (documentNumber == null || !SpringContext.getBean(AssetGlobalService.class).isAssetSeparate(this)) {
            return null;
        }
        final String userId = GlobalVariables.getUserSession().getPrincipalId();
        final WorkflowDocument workflowDocument = WorkflowDocumentFactory.loadDocument(userId, getDocumentNumber());

        // do not display not approved documents
        if (ObjectUtils.isNotNull(workflowDocument.getDateApproved())) {
            return new java.sql.Date(workflowDocument.getDateApproved().getMillis());
        }

        return null;
    }

    public boolean isCapitalAssetBuilderOriginIndicator() {
        return capitalAssetBuilderOriginIndicator;
    }

    public void setCapitalAssetBuilderOriginIndicator(final boolean capitalAssetBuilderOriginIndicator) {
        this.capitalAssetBuilderOriginIndicator = capitalAssetBuilderOriginIndicator;
    }

    public String getCalculateEqualSourceAmountsButton() {
        return calculateEqualSourceAmountsButton;
    }

    public void setCalculateEqualSourceAmountsButton(final String calculateEqualSourceAmountsButton) {
        this.calculateEqualSourceAmountsButton = calculateEqualSourceAmountsButton;
    }

    public String getCalculateSeparateSourceRemainingAmountButton() {
        return calculateSeparateSourceRemainingAmountButton;
    }

    public void setCalculateSeparateSourceRemainingAmountButton(final String calculateSeparateSourceRemainingAmountButton) {
        this.calculateSeparateSourceRemainingAmountButton = calculateSeparateSourceRemainingAmountButton;
    }

    public KualiDecimal getSeparateSourceRemainingAmount() {
        if (separateSourceRemainingAmount == null && getTotalCostAmount() != null) {
            return getTotalCostAmount().subtract(getSeparateSourceTotalAmount());
        }
        return separateSourceRemainingAmount;
    }

    public void setSeparateSourceRemainingAmount(final KualiDecimal separateSourceRemainingAmount) {
        this.separateSourceRemainingAmount = separateSourceRemainingAmount;
    }

    @Override
    public List<Collection<PersistableBusinessObject>> buildListOfDeletionAwareLists() {
        final List<Collection<PersistableBusinessObject>> managedLists = super.buildListOfDeletionAwareLists();
        managedLists.add(new ArrayList<>(getAssetGlobalDetails()));
        managedLists.add(new ArrayList<>(getAssetPaymentDetails()));
        return managedLists;
    }

    public KualiDecimal getTotalAssetPaymentAmount() {
        KualiDecimal totalAmount = KualiDecimal.ZERO;
        final List<AssetPaymentDetail> assetPaymentList = getAssetPaymentDetails();
        if (assetPaymentList != null && !assetPaymentList.isEmpty()) {
            for (final AssetPaymentDetail assetPaymentDetail : assetPaymentList) {
                totalAmount = totalAmount.add(assetPaymentDetail.getAmount());
            }
        }
        return totalAmount;
    }

    public void setTotalAssetPaymentAmount(final KualiDecimal totalAssetPaymentAmount) {
        this.totalAssetPaymentAmount = totalAssetPaymentAmount;
    }

    public KualiDecimal getSeparateSourceTotalAmount() {
        if (separateSourceTotalAmount == null) {
            separateSourceTotalAmount = KualiDecimal.ZERO;
            for (final AssetGlobalDetail detail : assetGlobalDetails) {
                final KualiDecimal separateSourceAmount = detail.getSeparateSourceAmount();
                if (separateSourceAmount != null) {
                    separateSourceTotalAmount = separateSourceTotalAmount.add(separateSourceAmount);
                }
            }
        }
        return separateSourceTotalAmount;
    }

    public void setSeparateSourceTotalAmount(final KualiDecimal separateSourceTotalAmount) {
        this.separateSourceTotalAmount = separateSourceTotalAmount;
    }

    public KualiDecimal getMinAssetTotalAmount() {
        return minAssetTotalAmount;
    }

    public void setMinAssetTotalAmount(final KualiDecimal minAssetTotalAmount) {
        this.minAssetTotalAmount = minAssetTotalAmount;
    }

    public KualiDecimal getMaxAssetTotalAmount() {
        return maxAssetTotalAmount;
    }

    public void setMaxAssetTotalAmount(final KualiDecimal maxAssetTotalAmount) {
        this.maxAssetTotalAmount = maxAssetTotalAmount;
    }

    public static AccountingPeriodService getAccountingPeriodService() {
        if (accountingPeriodService == null) {
            accountingPeriodService = SpringContext.getBean(AccountingPeriodService.class);
        }

        return accountingPeriodService;
    }

    public void setAccountingPeriod(final AccountingPeriod accountingPeriod) {
        this.accountingPeriod = accountingPeriod;

        if (ObjectUtils.isNotNull(accountingPeriod)) {
            setFinancialDocumentPostingYear(accountingPeriod.getUniversityFiscalYear());
            setFinancialDocumentPostingPeriodCode(accountingPeriod.getUniversityFiscalPeriodCode());
        }
    }

    public AccountingPeriod getAccountingPeriod() {
        return accountingPeriod;
    }

    /**
     * Creates a composite of postingPeriodCode and postingyear.
     *
     * @return composite or an empty string if either postingPeriodCode or postingYear is null
     */
    public String getAccountingPeriodCompositeString() {
        if (financialDocumentPostingPeriodCode == null || financialDocumentPostingYear == null) {
            return "";
        }
        return financialDocumentPostingPeriodCode + financialDocumentPostingYear;
    }

    public void setAccountingPeriodCompositeString(final String accountingPeriodString) {
        if (StringUtils.isNotBlank(accountingPeriodString)) {
            final String period = StringUtils.left(accountingPeriodString, 2);
            final Integer year = Integer.valueOf(StringUtils.right(accountingPeriodString, 4));
            final AccountingPeriod accountingPeriod = getAccountingPeriodService().getByPeriod(period, year);
            setAccountingPeriod(accountingPeriod);
        }
    }
}
