/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cam.businessobject;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.integration.purap.ItemCapitalAsset;
import org.kuali.kfs.krad.bo.PersistableBusinessObjectBase;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.krad.util.UrlFactory;
import org.kuali.kfs.module.cam.CamsConstants;
import org.kuali.kfs.module.cam.CamsPropertyConstants;
import org.kuali.kfs.module.cam.document.service.PurApLineService;
import org.kuali.kfs.module.purap.businessobject.PurchaseOrderItemCapitalAsset;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.core.api.util.type.KualiDecimal;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class PurchasingAccountsPayableItemAsset extends PersistableBusinessObjectBase implements
        Comparable<PurchasingAccountsPayableItemAsset> {

    private String documentNumber;
    private Integer accountsPayableLineItemIdentifier;
    private Integer capitalAssetBuilderLineNumber;
    private String accountsPayableLineItemDescription;
    private KualiDecimal accountsPayableItemQuantity;
    private String capitalAssetManagementDocumentNumber;
    private String activityStatusCode;

    private PurchasingAccountsPayableDocument purchasingAccountsPayableDocument;
    private List<PurchasingAccountsPayableLineAssetAccount> purchasingAccountsPayableLineAssetAccounts;

    // non persistent fields
    private boolean active;
    private Integer itemLineNumber;
    private boolean additionalChargeNonTradeInIndicator;
    private boolean tradeInAllowance;
    private boolean itemAssignedToTradeInIndicator;
    private KualiDecimal unitCost;
    private KualiDecimal totalCost;
    private String firstFincialObjectCode;
    private KualiDecimal splitQty;
    private boolean selectedValue;
    private String itemTypeCode;
    private String lockingInformation;
    // used for Capital Asset Transaction
    private String capitalAssetTransactionTypeCode;
    private List<ItemCapitalAsset> purApItemAssets;
    private Integer capitalAssetSystemIdentifier;

    private Integer purchaseOrderItemIdentifier;
    // used to control "create asset" and "apply payment" button display
    private boolean createAssetIndicator;
    private boolean applyPaymentIndicator;

    private String preTagInquiryUrl;
    private List<Long> approvedAssetNumbers;

    private Integer paymentRequestIdentifier;

    public PurchasingAccountsPayableItemAsset() {
        purchasingAccountsPayableLineAssetAccounts = new ArrayList<>();
        selectedValue = false;
        createAssetIndicator = false;
        applyPaymentIndicator = false;
        purApItemAssets = new ArrayList<>();
    }

    // constructor used for split
    public PurchasingAccountsPayableItemAsset(final PurchasingAccountsPayableItemAsset initialItemAsset) {
        documentNumber = initialItemAsset.documentNumber;
        accountsPayableLineItemIdentifier = initialItemAsset.getAccountsPayableLineItemIdentifier();
        accountsPayableLineItemDescription = initialItemAsset.getAccountsPayableLineItemDescription();
        itemLineNumber = initialItemAsset.getItemLineNumber();
        firstFincialObjectCode = initialItemAsset.getFirstFincialObjectCode();
        activityStatusCode = initialItemAsset.getActivityStatusCode();
        tradeInAllowance = initialItemAsset.isTradeInAllowance();
        itemAssignedToTradeInIndicator = initialItemAsset.isItemAssignedToTradeInIndicator();
        additionalChargeNonTradeInIndicator = initialItemAsset.isAdditionalChargeNonTradeInIndicator();
        purchasingAccountsPayableLineAssetAccounts = new ArrayList<>();
        selectedValue = false;
        createAssetIndicator = initialItemAsset.isCreateAssetIndicator();
        applyPaymentIndicator = initialItemAsset.isApplyPaymentIndicator();
        purchaseOrderItemIdentifier = initialItemAsset.getPurchaseOrderItemIdentifier();
        capitalAssetTransactionTypeCode = initialItemAsset.getCapitalAssetTransactionTypeCode();
        purApItemAssets = new ArrayList<>(initialItemAsset.getPurApItemAssets());
        capitalAssetSystemIdentifier = initialItemAsset.getCapitalAssetSystemIdentifier();
        purchasingAccountsPayableDocument = initialItemAsset.getPurchasingAccountsPayableDocument();
        lockingInformation = initialItemAsset.getLockingInformation();
    }

    public String getLockingInformation() {
        return lockingInformation;
    }

    public void setLockingInformation(final String lockingInformation) {
        this.lockingInformation = lockingInformation;
    }

    public Integer getCapitalAssetSystemIdentifier() {
        return capitalAssetSystemIdentifier;
    }

    public void setCapitalAssetSystemIdentifier(final Integer capitalAssetSystemIdentifier) {
        this.capitalAssetSystemIdentifier = capitalAssetSystemIdentifier;
    }

    public Integer getPurchaseOrderItemIdentifier() {
        return purchaseOrderItemIdentifier;
    }

    public void setPurchaseOrderItemIdentifier(final Integer purchaseOrderItemIdentifier) {
        this.purchaseOrderItemIdentifier = purchaseOrderItemIdentifier;
    }

    public List<ItemCapitalAsset> getPurApItemAssets() {
        return purApItemAssets;
    }

    public void setPurApItemAssets(final List<ItemCapitalAsset> capitalAssetNumbers) {
        purApItemAssets = capitalAssetNumbers;
    }

    public ItemCapitalAsset getPurApItemAsset(final int index) {
        while (getPurApItemAssets().size() <= index) {
            getPurApItemAssets().add(new PurchaseOrderItemCapitalAsset());
        }
        return getPurApItemAssets().get(index);
    }

    public boolean isCreateAssetIndicator() {
        return createAssetIndicator;
    }

    public void setCreateAssetIndicator(final boolean createAssetIndicator) {
        this.createAssetIndicator = createAssetIndicator;
    }

    public boolean isApplyPaymentIndicator() {
        return applyPaymentIndicator;
    }

    public void setApplyPaymentIndicator(final boolean applyPaymentIndicator) {
        this.applyPaymentIndicator = applyPaymentIndicator;
    }

    public boolean isSelectedValue() {
        return selectedValue;
    }

    public void setSelectedValue(final boolean selectedValue) {
        this.selectedValue = selectedValue;
    }

    public String getItemTypeCode() {
        return itemTypeCode;
    }

    public void setItemTypeCode(final String itemTypeCode) {
        this.itemTypeCode = itemTypeCode;
    }

    public String getCapitalAssetTransactionTypeCode() {
        return capitalAssetTransactionTypeCode;
    }

    public void setCapitalAssetTransactionTypeCode(final String capitalAssetTransactionTypeCode) {
        this.capitalAssetTransactionTypeCode = capitalAssetTransactionTypeCode;
    }

    public boolean isAdditionalChargeNonTradeInIndicator() {
        return additionalChargeNonTradeInIndicator;
    }

    public void setAdditionalChargeNonTradeInIndicator(final boolean additionalChargeNonTradeInIndicator) {
        this.additionalChargeNonTradeInIndicator = additionalChargeNonTradeInIndicator;
    }

    public boolean isTradeInAllowance() {
        return tradeInAllowance;
    }

    public void setTradeInAllowance(final boolean tradeInAllowance) {
        this.tradeInAllowance = tradeInAllowance;
    }

    public KualiDecimal getSplitQty() {
        return splitQty;
    }

    public void setSplitQty(final KualiDecimal splitQty) {
        this.splitQty = splitQty;
    }

    public List<PurchasingAccountsPayableLineAssetAccount> getPurchasingAccountsPayableLineAssetAccounts() {
        return purchasingAccountsPayableLineAssetAccounts;
    }

    public void setPurchasingAccountsPayableLineAssetAccounts(
            final List<PurchasingAccountsPayableLineAssetAccount> purchasingAccountsPayableLineAssetAccounts) {
        this.purchasingAccountsPayableLineAssetAccounts = purchasingAccountsPayableLineAssetAccounts;
    }

    public String getDocumentNumber() {
        return documentNumber;
    }

    public void setDocumentNumber(final String documentNumber) {
        this.documentNumber = documentNumber;
    }

    public Integer getAccountsPayableLineItemIdentifier() {
        return accountsPayableLineItemIdentifier;
    }

    public void setAccountsPayableLineItemIdentifier(final Integer accountsPayableLineItemIdentifier) {
        this.accountsPayableLineItemIdentifier = accountsPayableLineItemIdentifier;
    }

    public String getAccountsPayableLineItemDescription() {
        return accountsPayableLineItemDescription;
    }

    public void setAccountsPayableLineItemDescription(final String accountsPayableLineItemDescription) {
        this.accountsPayableLineItemDescription = accountsPayableLineItemDescription;
    }

    public KualiDecimal getAccountsPayableItemQuantity() {
        return accountsPayableItemQuantity;
    }

    public void setAccountsPayableItemQuantity(final KualiDecimal accountsPayableItemQuantity) {
        this.accountsPayableItemQuantity = accountsPayableItemQuantity;
    }

    public Integer getCapitalAssetBuilderLineNumber() {
        return capitalAssetBuilderLineNumber;
    }

    public void setCapitalAssetBuilderLineNumber(final Integer capitalAssetBuilderLineNumber) {
        this.capitalAssetBuilderLineNumber = capitalAssetBuilderLineNumber;
    }

    public String getCapitalAssetManagementDocumentNumber() {
        return capitalAssetManagementDocumentNumber;
    }

    public void setCapitalAssetManagementDocumentNumber(final String capitalAssetManagementDocumentNumber) {
        this.capitalAssetManagementDocumentNumber = capitalAssetManagementDocumentNumber;
    }

    public Integer getItemLineNumber() {
        return itemLineNumber;
    }

    public void setItemLineNumber(final Integer itemLineNumber) {
        this.itemLineNumber = itemLineNumber;
    }

    public boolean isActive() {
        return CamsConstants.ActivityStatusCode.NEW.equalsIgnoreCase(getActivityStatusCode()) ||
               CamsConstants.ActivityStatusCode.MODIFIED.equalsIgnoreCase(getActivityStatusCode());
    }

    public String getActivityStatusCode() {
        return activityStatusCode;
    }

    public void setActivityStatusCode(final String activityStatusCode) {
        this.activityStatusCode = activityStatusCode;
    }

    public PurchasingAccountsPayableDocument getPurchasingAccountsPayableDocument() {
        return purchasingAccountsPayableDocument;
    }

    public void setPurchasingAccountsPayableDocument(
            final PurchasingAccountsPayableDocument purchasingAccountsPayableDocument) {
        this.purchasingAccountsPayableDocument = purchasingAccountsPayableDocument;
    }

    public boolean isItemAssignedToTradeInIndicator() {
        return itemAssignedToTradeInIndicator;
    }

    public void setItemAssignedToTradeInIndicator(final boolean itemAssignedToTradeInIndicator) {
        this.itemAssignedToTradeInIndicator = itemAssignedToTradeInIndicator;
    }

    public KualiDecimal getUnitCost() {
        return unitCost;
    }

    public void setUnitCost(final KualiDecimal unitCost) {
        this.unitCost = unitCost;
    }

    public KualiDecimal getTotalCost() {
        return totalCost;
    }

    public void setTotalCost(final KualiDecimal totalCost) {
        this.totalCost = totalCost;
    }

    public String getFirstFincialObjectCode() {
        return firstFincialObjectCode;
    }

    public void setFirstFincialObjectCode(final String firstFincialObjectCode) {
        this.firstFincialObjectCode = firstFincialObjectCode;
    }

    public String getPreTagInquiryUrl() {
        if (StringUtils.isNotBlank(preTagInquiryUrl)) {
            return preTagInquiryUrl;
        }

        if (ObjectUtils.isNotNull(getPurchasingAccountsPayableDocument())) {
            final Integer purchaseOrderIdentifier = getPurchasingAccountsPayableDocument().getPurchaseOrderIdentifier();

            final PurApLineService purApLineService = SpringContext.getBean(PurApLineService.class);
            if (purApLineService.isPretaggingExisting(purApLineService.getPreTagLineItem(purchaseOrderIdentifier,
                    getItemLineNumber()))) {
                final String baseUrl = KFSConstants.INQUIRY_ACTION;
                final Map<String, String> parameters = new HashMap<>();
                parameters.put(KFSConstants.DISPATCH_REQUEST_PARAMETER, KFSConstants.START_METHOD);
                parameters.put(KFSConstants.BUSINESS_OBJECT_CLASS_ATTRIBUTE, Pretag.class.getName());
                parameters.put(CamsPropertyConstants.Pretag.PURCHASE_ORDER_NUMBER, purchaseOrderIdentifier.toString());
                parameters.put(CamsPropertyConstants.Pretag.ITEM_LINE_NUMBER, getItemLineNumber().toString());

                preTagInquiryUrl = UrlFactory.parameterizeUrl(baseUrl, parameters);

                return preTagInquiryUrl;
            }
        }
        return "";
    }

    @Override
    public int compareTo(final PurchasingAccountsPayableItemAsset o) {
        final boolean o1ItemTypeBelowTheLine = isAdditionalChargeNonTradeInIndicator() || isTradeInAllowance();
        final boolean o2ItemTypeBelowTheLine = o.isAdditionalChargeNonTradeInIndicator() || o.isTradeInAllowance();
        if (o1ItemTypeBelowTheLine && !o2ItemTypeBelowTheLine) {
            return 1;
        } else if (o2ItemTypeBelowTheLine && !o1ItemTypeBelowTheLine) {
            return -1;
        }
        return 0;
    }

    public List<Long> getApprovedAssetNumbers() {
        if (approvedAssetNumbers != null && !approvedAssetNumbers.isEmpty()) {
            return approvedAssetNumbers;
        } else {
            approvedAssetNumbers = new ArrayList<>();
            if (StringUtils.isNotEmpty(getCapitalAssetManagementDocumentNumber())) {
                final Map<String, String> fieldValues = new HashMap<>();
                if (CamsConstants.ActivityStatusCode.PROCESSED_IN_CAMS.equalsIgnoreCase(getActivityStatusCode())) {
                    // get asset number from asset global add doc
                    fieldValues.put(CamsPropertyConstants.AssetGlobalDetail.DOCUMENT_NUMBER,
                            getCapitalAssetManagementDocumentNumber());
                    final Collection<AssetGlobalDetail> assetGlobalDetails = SpringContext.getBean(
                            BusinessObjectService.class).findMatching(AssetGlobalDetail.class, fieldValues);
                    for (final AssetGlobalDetail detail : assetGlobalDetails) {
                        approvedAssetNumbers.add(detail.getCapitalAssetNumber());
                    }
                    if (assetGlobalDetails.isEmpty()) {
                        // get asset number from asset payment doc
                        fieldValues.clear();
                        fieldValues.put(CamsPropertyConstants.DOCUMENT_NUMBER,
                                getCapitalAssetManagementDocumentNumber());
                        final Collection<AssetPaymentAssetDetail> paymentAssetDetails = SpringContext.getBean(
                                BusinessObjectService.class).findMatching(AssetPaymentAssetDetail.class, fieldValues);
                        for (final AssetPaymentAssetDetail detail : paymentAssetDetails) {
                            approvedAssetNumbers.add(detail.getCapitalAssetNumber());
                        }
                    }
                }
            }
            return approvedAssetNumbers;
        }
    }

    public PurchasingAccountsPayableLineAssetAccount getPurchasingAccountsPayableLineAssetAccount(final int index) {
        int size = getPurchasingAccountsPayableLineAssetAccounts().size();
        while (size <= index || getPurchasingAccountsPayableLineAssetAccounts().get(index) == null) {
            getPurchasingAccountsPayableLineAssetAccounts().add(size++, new PurchasingAccountsPayableLineAssetAccount());
        }
        return getPurchasingAccountsPayableLineAssetAccounts().get(index);

    }

    public Integer getPaymentRequestIdentifier() {
        return paymentRequestIdentifier;
    }

    public void setPaymentRequestIdentifier(final Integer paymentRequestIdentifier) {
        this.paymentRequestIdentifier = paymentRequestIdentifier;
    }

    /**
     * KFSCNTRB-1676/FSKD-5487
     * Returns true if it is an active TRDI additional charge asset line.
     */
    public boolean isActiveAdditionalTRDI() {
        return isTradeInAllowance() && isActive();
    }

    /**
     * KFSCNTRB-1676/FSKD-5487
     * Returns true if it is an active non-TRDI additional charge asset line.
     */
    public boolean isActiveAdditionalNonTRDI() {
        return isAdditionalChargeNonTradeInIndicator() && isActive();
    }

    /**
     * KFSCNTRB-1676/FSKD-5487
     * Returns true if it is an active trade-in ITEM asset line.
     */
    public boolean isActiveItemTradeIn() {
        return !isAdditionalChargeNonTradeInIndicator() && !isTradeInAllowance() && isItemAssignedToTradeInIndicator()
                && isActive();
    }

    /**
     * KFSCNTRB-1676/FSKD-5487
     * Returns true if it is an active ITEM asset line.
     */
    public boolean isActiveItemLine() {
        return !isAdditionalChargeNonTradeInIndicator() && !isTradeInAllowance() && isActive();
    }

}
