/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cam.batch.service.impl;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.core.api.datetime.DateTimeService;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.datadictionary.legacy.DataDictionaryService;
import org.kuali.kfs.gl.service.impl.StringHelper;
import org.kuali.kfs.krad.bo.AdHocRouteRecipient;
import org.kuali.kfs.krad.document.Document;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.service.DocumentService;
import org.kuali.kfs.krad.service.KualiRuleService;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.module.cam.CamsConstants;
import org.kuali.kfs.module.cam.CamsParameterConstants;
import org.kuali.kfs.module.cam.CamsPropertyConstants;
import org.kuali.kfs.module.cam.batch.service.AssetBarcodeInventoryLoadService;
import org.kuali.kfs.module.cam.businessobject.Asset;
import org.kuali.kfs.module.cam.businessobject.AssetLocation;
import org.kuali.kfs.module.cam.businessobject.BarcodeInventoryErrorDetail;
import org.kuali.kfs.module.cam.document.BarcodeInventoryErrorDocument;
import org.kuali.kfs.module.cam.document.service.AssetService;
import org.kuali.kfs.module.cam.document.validation.event.ValidateBarcodeInventoryEvent;
import org.kuali.kfs.module.cam.document.web.struts.AssetBarCodeInventoryInputFileForm;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSKeyConstants;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.nio.charset.StandardCharsets;
import java.sql.Timestamp;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

/**
 * Implementation of the AssetBarcodeInventoryLoadService interface. Handles loading, parsing, and storing of incoming
 * barcode inventory files.
 */
public class AssetBarcodeInventoryLoadServiceImpl implements AssetBarcodeInventoryLoadService {

    private static final Logger LOG = LogManager.getLogger();

    // Used in CSU Overlay
    protected static final String MESSAGE_NO_DOCUMENT_CREATED = "NO barcode inventory error document was created.";
    // Used in UConn Overlay
    protected static final String DOCUMENT_EXPLANATION = "BARCODE ERROR INVENTORY";
    protected static final String DOCUMENTS_MSG = "The following barcode inventory error document were created";
    protected static final String TOTAL_RECORDS_UPLOADED_MSG = "Total records uploaded";
    protected static final String TOTAL_RECORDS_IN_ERROR_MSG = "Total records in error";

    private static final int MAX_NUMBER_OF_COLUMNS = 8;

    private BusinessObjectService businessObjectService;
    private DataDictionaryService dataDictionaryService;
    private KualiRuleService kualiRuleService;
    private DocumentService documentService;
    private ParameterService parameterService;
    private DateTimeService dateTimeService;
    private AssetService assetService;

    /**
     * Determines whether BarcodeInventoryError document has all its records corrected or deleted.
     */
    @Override
    public boolean isFullyProcessed(final Document document) {
        LOG.debug("isFullyProcessed() started");

        final BarcodeInventoryErrorDocument barcodeInventoryErrorDocument = (BarcodeInventoryErrorDocument) document;
        boolean result = true;
        final List<BarcodeInventoryErrorDetail> barcodeInventoryErrorDetails =
                barcodeInventoryErrorDocument.getBarcodeInventoryErrorDetail();

        for (final BarcodeInventoryErrorDetail detail : barcodeInventoryErrorDetails) {
            if (detail.getErrorCorrectionStatusCode().equals(CamsConstants.BarCodeInventoryError.STATUS_CODE_ERROR)) {
                result = false;
                break;
            }
        }
        return result;
    }

    @Override
    public boolean isCurrentUserInitiator(final Document document) {
        LOG.debug("isCurrentUserInitiator() started");

        if (document != null) {
            return GlobalVariables.getUserSession().getPerson().getPrincipalId().equalsIgnoreCase(
                    document.getDocumentHeader().getWorkflowDocument().getInitiatorPrincipalId());
        }
        return false;
    }

    @Override
    public boolean isFileFormatValid(final File file) {
        LOG.debug("isFileFormatValid(File file) - start");

        // Getting the length of each field that needs to be validated
        final Integer campusTagNumberMaxLength = dataDictionaryService.getAttributeMaxLength(Asset.class,
                CamsPropertyConstants.Asset.CAMPUS_TAG_NUMBER);
        final int inventoryScannedCodeMaxLength = 1;
        final Integer inventoryDateMaxLength = dataDictionaryService.getAttributeMaxLength(
                BarcodeInventoryErrorDetail.class, CamsPropertyConstants.BarcodeInventory.INVENTORY_DATE);
        final Integer campusCodeMaxLength = dataDictionaryService.getAttributeMaxLength(Asset.class,
                CamsPropertyConstants.Asset.CAMPUS_CODE);
        final Integer buildingCodeMaxLength = dataDictionaryService.getAttributeMaxLength(Asset.class,
                CamsPropertyConstants.Asset.BUILDING_CODE);
        final Integer buildingRoomNumberMaxLength = dataDictionaryService.getAttributeMaxLength(Asset.class,
                CamsPropertyConstants.Asset.BUILDING_ROOM_NUMBER);
        final Integer buildingSubRoomNumberMaxLength = dataDictionaryService.getAttributeMaxLength(Asset.class,
                CamsPropertyConstants.Asset.BUILDING_SUB_ROOM_NUMBER);
        final Integer conditionCodeMaxLength = dataDictionaryService.getAttributeMaxLength(Asset.class,
                CamsPropertyConstants.Asset.CONDITION_CODE);

        // Getting the label of each field from data dictionary.
        final String campusTagNumberLabel = dataDictionaryService.getAttributeLabel(Asset.class,
                CamsPropertyConstants.Asset.CAMPUS_TAG_NUMBER);
        final String inventoryScannedCodeLabel = dataDictionaryService.getAttributeLabel(Asset.class,
                CamsPropertyConstants.BarcodeInventory.UPLOAD_SCAN_INDICATOR);
        final String inventoryDateLabel = dataDictionaryService.getAttributeLabel(BarcodeInventoryErrorDetail.class,
                CamsPropertyConstants.BarcodeInventory.INVENTORY_DATE);
        final String campusCodeLabel = dataDictionaryService.getAttributeLabel(Asset.class,
                CamsPropertyConstants.Asset.CAMPUS_CODE);
        final String buildingCodeLabel = dataDictionaryService.getAttributeLabel(Asset.class,
                CamsPropertyConstants.Asset.BUILDING_CODE);
        final String buildingRoomNumberLabel = dataDictionaryService.getAttributeLabel(Asset.class,
                CamsPropertyConstants.Asset.BUILDING_ROOM_NUMBER);
        final String buildingSubRoomNumberLabel = dataDictionaryService.getAttributeLabel(Asset.class,
                CamsPropertyConstants.Asset.BUILDING_SUB_ROOM_NUMBER);
        final String conditionCodeLabel = dataDictionaryService.getAttributeLabel(Asset.class,
                CamsPropertyConstants.Asset.CONDITION_CODE);

        try (BufferedReader input = new BufferedReader(new FileReader(file, StandardCharsets.UTF_8))) {
            int recordCount = 0;
            String errorMsg;
            String errorMessage = "";
            boolean proceed = true;
            final String lengthError = "exceeds maximum length";

            String line;

            while ((line = input.readLine()) != null) {
                recordCount++;
                errorMsg = "";
                line = StringUtils.remove(line, "\"");

                final String[] column = line.split(",");

                if (MAX_NUMBER_OF_COLUMNS < column.length) {
                    // Error more columns that allowed. put it in the constants class.
                    errorMsg += "  Barcode inventory file has record(s) with more than " + MAX_NUMBER_OF_COLUMNS +
                            " columns\n";
                    proceed = false;
                } else if (MAX_NUMBER_OF_COLUMNS > column.length) {
                    errorMsg += "  Barcode inventory file has record(s) with less than " + MAX_NUMBER_OF_COLUMNS +
                            " columns\n";
                    proceed = false;
                } else {
                    // Validating length of each field
                    if (column[0].length() > campusTagNumberMaxLength) {
                        errorMsg += ", " + campusTagNumberLabel;
                    }

                    if (column[1].length() > inventoryScannedCodeMaxLength) {
                        errorMsg += ", " + inventoryScannedCodeLabel;
                    }

                    if (column[2].length() > inventoryDateMaxLength) {
                        errorMsg += ", " + inventoryDateLabel;
                    }

                    if (column[3].length() > campusCodeMaxLength) {
                        errorMsg += ", " + campusCodeLabel;
                    }
                    if (column[4].length() > buildingCodeMaxLength) {
                        errorMsg += ", " + buildingCodeLabel;
                    }
                    if (column[5].length() > buildingRoomNumberMaxLength) {
                        errorMsg += ", " + buildingRoomNumberLabel;
                    }
                    if (column[6].length() > buildingSubRoomNumberMaxLength) {
                        errorMsg += ", " + buildingSubRoomNumberLabel;
                    }
                    if (column[7].length() > conditionCodeMaxLength) {
                        errorMsg += ", " + conditionCodeLabel;
                    }

                    if (StringUtils.isNotBlank(errorMsg)) {
                        errorMsg += " " + lengthError;
                    }

                    // Validating other than the length of the fields
                    if (!column[1].equals(CamsConstants.BarCodeInventory.BCI_SCANNED_INTO_DEVICE)
                            && !column[1].equals(CamsConstants.BarCodeInventory.BCI_MANUALLY_KEYED_CODE)) {
                        errorMsg += ", " + inventoryScannedCodeLabel + " is invalid";
                    }

                    // validate date
                    if (!validateDate(column[2])) {
                        errorMsg += ", " + inventoryDateLabel + " is invalid";
                    }
                }
                if (StringUtils.isNotBlank(errorMsg)) {
                    errorMsg = "Error on record number " + recordCount + ": " + errorMsg.substring(2) + "\n";
                    GlobalVariables.getMessageMap().putError(KFSConstants.GLOBAL_ERRORS, KFSKeyConstants.ERROR_CUSTOM,
                            errorMsg);
                    errorMessage += errorMsg;
                    LOG.error(errorMsg);
                }
                if (!proceed) {
                    break;
                }
            }
            return StringUtils.isBlank(errorMessage);
        } catch (final FileNotFoundException e1) {
            LOG.error("file to parse not found {}", file::getName, () -> e1);
            throw new RuntimeException("Cannot find the file requested to be parsed " + file.getName() + " " +
                    e1.getMessage(), e1);
        } catch (final Exception e) {
            LOG.error("Error running file validation - File: {}", file::getName, () -> e);
            throw new IllegalArgumentException("Error running file validation - File: " + file.getName());
        }
    }

    @Override
    public boolean processFile(
            final File file,
            final AssetBarCodeInventoryInputFileForm form
    ) {
        LOG.debug("processFile(File file) - start");

        // Removing *.done files that are created automatically by the framework.
        removeDoneFile(file);

        String day;
        String month;
        String year;
        String hours;
        String minutes;
        String seconds;

        final SimpleDateFormat formatter = new SimpleDateFormat(CamsConstants.DateFormats.MONTH_DAY_YEAR + " " +
                CamsConstants.DateFormats.MILITARY_TIME, Locale.US);
        formatter.setLenient(false);

        BarcodeInventoryErrorDetail barcodeInventoryErrorDetail;
        final List<BarcodeInventoryErrorDetail> barcodeInventoryErrorDetails = new ArrayList<>();

        try (BufferedReader input = new BufferedReader(new FileReader(file, StandardCharsets.UTF_8))) {
            Long ln = 1L;
            String line;

            while ((line = input.readLine()) != null) {
                line = StringUtils.remove(line, "\"");
                final String[] lineStrings = line.split(",");

                // Parsing date so it can be validated.
                lineStrings[2] = StringUtils.rightPad(lineStrings[2].trim(), 14, "0");

                day = lineStrings[2].substring(0, 2);
                month = lineStrings[2].substring(2, 4);
                year = lineStrings[2].substring(4, 8);
                hours = lineStrings[2].substring(8, 10);
                minutes = lineStrings[2].substring(10, 12);
                seconds = lineStrings[2].substring(12);

                final String stringDate = month + "/" + day + "/" + year + " " + hours + ":" + minutes + ":" + seconds;
                Timestamp timestamp = null;

                // If date has invalid format set its value to null
                try {
                    timestamp = new Timestamp(formatter.parse(stringDate).getTime());
                } catch (final ParseException e) {
                    LOG.warn("processFile() - invalid date format : stringDate={} for line={}", stringDate, ln);
                }

                // Its set to null because for some reason java parses "00000000000000" as 0002-11-30
                if (lineStrings[2].equals(StringUtils.repeat("0", 14))) {
                    timestamp = null;
                }

                barcodeInventoryErrorDetail = new BarcodeInventoryErrorDetail();
                barcodeInventoryErrorDetail.setUploadRowNumber(ln);
                barcodeInventoryErrorDetail.setAssetTagNumber(lineStrings[0].trim());
                barcodeInventoryErrorDetail.setUploadScanIndicator(lineStrings[1]
                        .equals(CamsConstants.BarCodeInventory.BCI_SCANNED_INTO_DEVICE));
                barcodeInventoryErrorDetail.setUploadScanTimestamp(timestamp);
                barcodeInventoryErrorDetail.setCampusCode(lineStrings[3].trim().toUpperCase(Locale.US));
                barcodeInventoryErrorDetail.setBuildingCode(lineStrings[4].trim().toUpperCase(Locale.US));
                barcodeInventoryErrorDetail.setBuildingRoomNumber(lineStrings[5].trim().toUpperCase(Locale.US));
                barcodeInventoryErrorDetail.setBuildingSubRoomNumber(lineStrings[6].trim().toUpperCase(Locale.US));
                barcodeInventoryErrorDetail.setAssetConditionCode(lineStrings[7].trim().toUpperCase(Locale.US));
                barcodeInventoryErrorDetail.setErrorCorrectionStatusCode(
                        CamsConstants.BarCodeInventoryError.STATUS_CODE_ERROR);
                barcodeInventoryErrorDetail.setCorrectorUniversalIdentifier(
                        GlobalVariables.getUserSession().getPerson().getPrincipalId());

                barcodeInventoryErrorDetails.add(barcodeInventoryErrorDetail);
                ln++;
            }
            processBarcodeInventory(barcodeInventoryErrorDetails, form);

            return true;
        } catch (final FileNotFoundException e1) {
            LOG.error("file to parse not found {}", file::getName, () -> e1);
            throw new RuntimeException("Cannot find the file requested to be parsed " + file.getName() + " " +
                    e1.getMessage(), e1);
        } catch (final Exception ex) {
            LOG.error("Error reading file", ex);
            throw new IllegalArgumentException("Error reading file: " + ex.getMessage(), ex);
        }
    }

    /**
     * This method removes the *.done files. If not deleted, then the program will display the name of the file in a
     * select menu with a label of ready for process.
     */
    // Used in CSU Overlay
    protected static void removeDoneFile(final File file) {
        final String filePath = file.getAbsolutePath();
        final File doneFile = new File(StringUtils.substringBeforeLast(filePath, ".") + ".done");
        if (doneFile.exists()) {
            doneFile.delete();
        }
    }

    /**
     * This method invokes the rules in order to validate each records of the barcode file and invokes the method
     * that updates the asset table with the records that passes the rules validation
     */
    // Used in CSU Overlay
    protected void processBarcodeInventory(
            final List<BarcodeInventoryErrorDetail> barcodeInventoryErrorDetails,
            final AssetBarCodeInventoryInputFileForm form
    ) {
        long lineNumber = 0L;
        boolean docCreated = false;
        int errorRecCount = 0;
        int totalRecCount = 0;

        final BarcodeInventoryErrorDocument barcodeInventoryErrorDocument =
                createInvalidBarcodeInventoryDocument(barcodeInventoryErrorDetails, form.getUploadDescription());
        // apply rules for the new cash control detail
        kualiRuleService.applyRules(new ValidateBarcodeInventoryEvent("", barcodeInventoryErrorDocument, true));

        final List<BarcodeInventoryErrorDetail> tmpBarcodeInventoryErrorDetails = new ArrayList<>();

        for (final BarcodeInventoryErrorDetail barcodeInventoryErrorDetail : barcodeInventoryErrorDetails) {
            totalRecCount++;
            if (barcodeInventoryErrorDetail.getErrorCorrectionStatusCode()
                    .equals(CamsConstants.BarCodeInventoryError.STATUS_CODE_ERROR)) {
                errorRecCount++;
                lineNumber++;
                // Assigning the row number to each invalid BCIE record
                barcodeInventoryErrorDetail.setUploadRowNumber(lineNumber);

                // Storing in temp collection the invalid BCIE records.
                tmpBarcodeInventoryErrorDetails.add(barcodeInventoryErrorDetail);
            } else {
                // if no error found, then update asset table.
                updateAssetInformation(barcodeInventoryErrorDetail, true);
            }
        }
        // *********************************************************************
        // Storing the invalid barcode inventory records.
        // *********************************************************************
        String documentsCreated = "";
        if (!tmpBarcodeInventoryErrorDetails.isEmpty()) {
            documentsCreated = createBarcodeInventoryErrorDocuments(tmpBarcodeInventoryErrorDetails,
                    barcodeInventoryErrorDocument, form);
            docCreated = true;
        }

        if (docCreated) {
            // Adding the list of documents that were created in the message list
            form.getMessages().add(DOCUMENTS_MSG + ": " + documentsCreated.substring(2));
        } else {
            form.getMessages().add(MESSAGE_NO_DOCUMENT_CREATED);
        }
        form.getMessages().add(TOTAL_RECORDS_UPLOADED_MSG + ": " +
                StringUtils.rightPad(Integer.toString(totalRecCount), 5, " "));
        form.getMessages().add(TOTAL_RECORDS_IN_ERROR_MSG + ": " +
                StringUtils.rightPad(Integer.toString(errorRecCount), 5, " "));
    }

    // Used in CSU Overlay
    protected String createBarcodeInventoryErrorDocuments(
            final List<? extends BarcodeInventoryErrorDetail> bcies,
            BarcodeInventoryErrorDocument barcodeInventoryErrorDocument,
            final AssetBarCodeInventoryInputFileForm form
    ) {
        List<BarcodeInventoryErrorDetail> barcodeInventoryErrorDetails = new ArrayList<>();
        boolean isFirstDocument = true;
        int ln = 0;
        int bcieCount = 0;
        String documentsCreated = "";
        int maxNumberRecordsPerDocument = 300;

        if (parameterService.parameterExists(BarcodeInventoryErrorDocument.class,
                CamsParameterConstants.RECORDS_PER_DOCUMENT
        )) {
            maxNumberRecordsPerDocument = new Integer(parameterService
                    .getParameterValueAsString(BarcodeInventoryErrorDocument.class,
                            CamsParameterConstants.RECORDS_PER_DOCUMENT
                    ));
        }

        while (true) {
            if (ln > maxNumberRecordsPerDocument || bcieCount >= bcies.size()) {
                // This if was added in order to not waste the document already created and not create a new one.
                if (!isFirstDocument) {
                    barcodeInventoryErrorDocument = createInvalidBarcodeInventoryDocument(
                            barcodeInventoryErrorDetails, form.getUploadDescription());
                }
                documentsCreated += ", " + barcodeInventoryErrorDocument.getDocumentNumber();

                barcodeInventoryErrorDocument.setBarcodeInventoryErrorDetail(barcodeInventoryErrorDetails);
                saveInvalidBarcodeInventoryDocument(barcodeInventoryErrorDocument);

                barcodeInventoryErrorDetails = new ArrayList<>();

                if (bcieCount >= bcies.size()) {
                    break;
                }

                ln = 0;
                isFirstDocument = false;
            }

            final BarcodeInventoryErrorDetail barcodeInventoryErrorDetail = bcies.get(bcieCount);
            barcodeInventoryErrorDetail.setUploadRowNumber((long) (ln + 1));
            barcodeInventoryErrorDetails.add(barcodeInventoryErrorDetail);

            ln++;
            bcieCount++;
        }
        return documentsCreated;
    }

    /**
     * This method updates the asset information particularly the building code, building room, building subrool,
     * campus code, and condition code.
     */
    @Override
    public void updateAssetInformation(
            final BarcodeInventoryErrorDetail barcodeInventoryErrorDetail,
            final boolean updateWithDateAssetWasScanned
    ) {
        LOG.debug("updateAssetInformation() started");

        final Asset asset =
                assetService.findActiveAssetsMatchingTagNumber(barcodeInventoryErrorDetail.getAssetTagNumber()).get(0);
        asset.setInventoryScannedCode(barcodeInventoryErrorDetail.isUploadScanIndicator() ?
                CamsConstants.BarCodeInventory.BCI_SCANNED_INTO_DEVICE :
                CamsConstants.BarCodeInventory.BCI_MANUALLY_KEYED_CODE);
        asset.setBuildingCode(barcodeInventoryErrorDetail.getBuildingCode());
        asset.setBuildingRoomNumber(barcodeInventoryErrorDetail.getBuildingRoomNumber());
        asset.setBuildingSubRoomNumber(barcodeInventoryErrorDetail.getBuildingSubRoomNumber());
        asset.setCampusCode(barcodeInventoryErrorDetail.getCampusCode());
        asset.setConditionCode(barcodeInventoryErrorDetail.getAssetConditionCode());

        // set building code and room number to null if they are empty string, to avoid FK violation exception
        if (StringUtils.isEmpty(asset.getBuildingCode())) {
            asset.setBuildingCode(null);
            asset.setBuilding(null);
        }

        if (StringUtils.isEmpty(asset.getBuildingRoomNumber())) {
            asset.setBuildingRoomNumber(null);
            asset.setBuildingRoom(null);
        }

        if (updateWithDateAssetWasScanned) {
            asset.setLastInventoryDate(barcodeInventoryErrorDetail.getUploadScanTimestamp());
        } else {
            asset.setLastInventoryDate(new Timestamp(dateTimeService.getCurrentSqlDate().getTime()));
        }

        // Purposefully deleting off-campus locations when loading locations via barcode scanning.
        final List<AssetLocation> assetLocations = asset.getAssetLocations();
        for (final AssetLocation assetLocation : assetLocations) {
            if (CamsConstants.AssetLocationTypeCode.OFF_CAMPUS.equals(assetLocation.getAssetLocationTypeCode())) {
                assetLocations.remove(assetLocation);
                break;
            }
        }

        updateAssetPreSave(barcodeInventoryErrorDetail, asset);
        businessObjectService.save(asset);
    }

    @Override
    public void updateAssetPreSave(
            final BarcodeInventoryErrorDetail barcodeInventoryErrorDetail,
            final Asset asset
    ) {
        LOG.debug("updateAssetPreSave() started");
        // Not implemented by default.  This method can be overridden by implementing institutions
    }

    /**
     * This method creates a transaction document with the invalid barcode inventory records.
     */
    // Used in CSU Overlay
    protected BarcodeInventoryErrorDocument createInvalidBarcodeInventoryDocument(
            final List<BarcodeInventoryErrorDetail> barcodeInventoryErrorDetails,
            final String uploadDescription
    ) {
        final BarcodeInventoryErrorDocument document = (BarcodeInventoryErrorDocument) documentService.getNewDocument(
                BarcodeInventoryErrorDocument.class);

        document.getDocumentHeader().setExplanation(DOCUMENT_EXPLANATION);
        document.getDocumentHeader().setFinancialDocumentTotalAmount(KualiDecimal.ZERO);
        document.getDocumentHeader().setDocumentDescription(uploadDescription);
        document.setUploaderUniversalIdentifier(GlobalVariables.getUserSession().getPerson().getPrincipalId());
        document.setBarcodeInventoryErrorDetail(barcodeInventoryErrorDetails);

        return document;
    }

    private void saveInvalidBarcodeInventoryDocument(final BarcodeInventoryErrorDocument document) {
        try {
            // The errors are being deleted because, when the document services finds any error then, changes are not
            // saved.
            GlobalVariables.clear();

            // no adhoc recipient need to add when submit doc. doc will route to the doc uploader, i.e. initiator
            // automatically.
            final List<AdHocRouteRecipient> adHocRouteRecipients = new ArrayList<>();
            documentService.routeDocument(document, "Routed Update Barcode Inventory Document",
                    adHocRouteRecipients);
        } catch (final Exception e) {
            final String documentNumber = document.getDocumentHeader().getDocumentNumber();
            LOG.error("Error persisting document # {}", documentNumber, e);
            throw new RuntimeException("Error persisting document # " + documentNumber, e);
        }
    }

    private static boolean validateDate(String date) {
        // Parsing date so it can be validated.
        boolean valid = true;
        if (StringHelper.isEmpty(date)) {
            valid = false;
        } else {
            final SimpleDateFormat formatter = new SimpleDateFormat(CamsConstants.DateFormats.MONTH_DAY_YEAR +
                    " " + CamsConstants.DateFormats.STANDARD_TIME, Locale.US);
            date = StringUtils.rightPad(date.trim(), 14, "0");
            final String day = date.substring(0, 2);
            final String month = date.substring(2, 4);
            final String year = date.substring(4, 8);
            final String hours = date.substring(8, 10);
            final String minutes = date.substring(10, 12);
            final String seconds = date.substring(12);

            final String stringDate = month + "/" + day + "/" + year + " " + hours + ":" + minutes + ":" + seconds;

            // If date has invalid format set its value to null
            try {
                new Timestamp(formatter.parse(stringDate).getTime());
            } catch (final Exception e) {
                valid = false;
            }
        }

        return valid;
    }

    public void setBusinessObjectService(final BusinessObjectService businessObjectService) {
        this.businessObjectService = businessObjectService;
    }

    public void setDataDictionaryService(final DataDictionaryService dataDictionaryService) {
        this.dataDictionaryService = dataDictionaryService;
    }

    public void setKualiRuleService(final KualiRuleService kualiRuleService) {
        this.kualiRuleService = kualiRuleService;
    }

    public void setDocumentService(final DocumentService documentService) {
        this.documentService = documentService;
    }

    public void setParameterService(final ParameterService parameterService) {
        this.parameterService = parameterService;
    }

    public void setDateTimeService(final DateTimeService dateTimeService) {
        this.dateTimeService = dateTimeService;
    }

    public void setAssetService(final AssetService assetService) {
        this.assetService = assetService;
    }
}
