/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cam.document;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.integration.cam.CapitalAssetManagementModuleService;
import org.kuali.kfs.krad.document.Copyable;
import org.kuali.kfs.krad.exception.ValidationException;
import org.kuali.kfs.krad.rules.rule.event.KualiDocumentEvent;
import org.kuali.kfs.krad.rules.rule.event.SaveDocumentEvent;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.cam.CamsConstants;
import org.kuali.kfs.module.cam.CamsParameterConstants;
import org.kuali.kfs.module.cam.CamsPropertyConstants;
import org.kuali.kfs.module.cam.batch.AssetDepreciationStep;
import org.kuali.kfs.module.cam.businessobject.Asset;
import org.kuali.kfs.module.cam.businessobject.AssetObjectCode;
import org.kuali.kfs.module.cam.businessobject.AssetPayment;
import org.kuali.kfs.module.cam.businessobject.AssetPaymentAllocationType;
import org.kuali.kfs.module.cam.businessobject.AssetPaymentAssetDetail;
import org.kuali.kfs.module.cam.businessobject.AssetPaymentDetail;
import org.kuali.kfs.module.cam.businessobject.AssetPaymentInProcessPayment;
import org.kuali.kfs.module.cam.document.service.AssetObjectCodeService;
import org.kuali.kfs.module.cam.document.service.AssetPaymentService;
import org.kuali.kfs.module.cam.document.validation.event.AssetPaymentManuallyAddAccountingLineEvent;
import org.kuali.kfs.module.cam.util.distribution.AssetDistribution;
import org.kuali.kfs.module.cam.util.distribution.AssetDistributionEvenly;
import org.kuali.kfs.module.cam.util.distribution.AssetDistributionManual;
import org.kuali.kfs.module.cam.util.distribution.AssetDistributionPercent;
import org.kuali.kfs.module.cam.util.distribution.AssetPaymentDistributionByTotalCost;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.businessobject.AccountingLine;
import org.kuali.kfs.sys.businessobject.GeneralLedgerPendingEntry;
import org.kuali.kfs.sys.businessobject.GeneralLedgerPendingEntrySequenceHelper;
import org.kuali.kfs.sys.businessobject.GeneralLedgerPendingEntrySourceDetail;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.sys.document.AccountingDocumentBase;
import org.kuali.kfs.sys.document.AmountTotaling;
import org.kuali.kfs.sys.document.validation.impl.AccountingDocumentRuleBaseConstants;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.kew.api.WorkflowDocument;
import org.kuali.kfs.kew.framework.postprocessor.DocumentRouteStatusChange;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

/**
 * Capital assets document class for the asset payment document
 */
public class AssetPaymentDocument extends AccountingDocumentBase implements Copyable, AmountTotaling {

    private static final Logger LOG = LogManager.getLogger();

    protected List<AssetPaymentAssetDetail> assetPaymentAssetDetail;
    protected Long capitalAssetNumber;
    protected boolean capitalAssetBuilderOriginIndicator;
    protected AssetPaymentAllocationType assetPaymentAllocationType;
    protected String assetPaymentAllocationTypeCode;
    protected boolean allocationFromFPDocuments;
    protected List<AssetPaymentInProcessPayment> assetPaymentInProcessPayments;

    // helper transients for GLPE generation. Used to avoid having to modify the method arguments in super class
    protected transient AssetPayment apipAssetPaymentHelper;
    protected transient KualiDecimal apipAssetPaymentTotalDepreciationHelper;

    public AssetPaymentDocument() {
        super();
        setAllocationFromFPDocuments(false);
        assetPaymentAllocationTypeCode = CamsPropertyConstants.AssetPaymentAllocation.ASSET_DISTRIBUTION_DEFAULT_CODE;
        setAssetPaymentAssetDetail(new ArrayList<>());
        setAssetPaymentInProcessPayments(new ArrayList<>());
    }

    /**
     * Remove asset from collection for deletion
     */
    @Override
    public List buildListOfDeletionAwareLists() {
        final List<List> deletionAwareList = super.buildListOfDeletionAwareLists();
        deletionAwareList.add(getAssetPaymentAssetDetail());
        deletionAwareList.add(getAssetPaymentInProcessPayments());
        return deletionAwareList;
    }

    /**
     * When document save, AddAccountingLineEvent is added by the framework. Also, we need to add
     * AssetPaymentManuallyAddAccountingLineEvent manually to run all relating validations.
     */
    @Override
    public List generateSaveEvents() {
        final List subEvents = new ArrayList();
        // keep the order of events as for validation will run in the same order.
        if (!isCapitalAssetBuilderOriginIndicator()) {
            // Add AssetPaymentManuallyAddAccountingLineEvent for each manually added accounting line.
            final String errorPathPrefix = KFSConstants.DOCUMENT_PROPERTY_NAME + "." +
                                           KFSConstants.EXISTING_SOURCE_ACCT_LINE_PROPERTY_NAME + KFSConstants.ACCOUNTING_LINE_GROUP_SUFFIX;
            int index = 0;
            for (final Iterator i = getSourceAccountingLines().iterator(); i.hasNext(); index++) {
                final String indexedErrorPathPrefix = errorPathPrefix + "[" + index + "]";
                final AccountingLine currentLine = (AccountingLine) i.next();
                final AssetPaymentManuallyAddAccountingLineEvent newSubEvent =
                        new AssetPaymentManuallyAddAccountingLineEvent(indexedErrorPathPrefix, this, currentLine);
                subEvents.add(newSubEvent);
            }
        }

        subEvents.addAll(super.generateSaveEvents());

        return subEvents;
    }

    /**
     * Lock on purchase order document since post processor will update PO document by adding notes.
     */
    @Override
    public List<String> getWorkflowEngineDocumentIdsToLock() {
        List<String> documentIds = null;
        if (isCapitalAssetBuilderOriginIndicator()) {
            final String poDocId = SpringContext.getBean(CapitalAssetManagementModuleService.class)
                    .getCurrentPurchaseOrderDocumentNumber(getDocumentNumber());
            if (StringUtils.isNotBlank(poDocId)) {
                documentIds = new ArrayList<>();
                documentIds.add(poDocId);
            }
        }
        return documentIds;
    }

    /**
     * Determines if the given AccountingLine (as a GeneralLedgerPostable) is a credit or a debit, in terms of GLPE
     * generation
     */
    @Override
    public boolean isDebit(final GeneralLedgerPendingEntrySourceDetail postable) {
        return false;
    }

    public boolean isCapitalAssetBuilderOriginIndicator() {
        return capitalAssetBuilderOriginIndicator;
    }

    public void setCapitalAssetBuilderOriginIndicator(final boolean capitalAssetBuilderOriginIndicator) {
        this.capitalAssetBuilderOriginIndicator = capitalAssetBuilderOriginIndicator;
    }

    /**
     * @param assetPaymentAssetDetail
     */
    public void addAssetPaymentAssetDetail(final AssetPaymentAssetDetail assetPaymentAssetDetail) {
        getAssetPaymentAssetDetail().add(assetPaymentAssetDetail);
    }

    @Override
    public void postProcessSave(final KualiDocumentEvent event) {
        super.postProcessSave(event);

        if (!(event instanceof SaveDocumentEvent)) {
            // don't lock until they route
            final ArrayList<Long> capitalAssetNumbers = new ArrayList<>();
            for (final AssetPaymentAssetDetail assetPaymentAssetDetail : getAssetPaymentAssetDetail()) {
                if (assetPaymentAssetDetail.getCapitalAssetNumber() != null) {
                    capitalAssetNumbers.add(assetPaymentAssetDetail.getCapitalAssetNumber());
                }
            }

            String documentTypeForLocking = CamsConstants.DocumentTypeName.ASSET_PAYMENT;
            if (isCapitalAssetBuilderOriginIndicator()) {
                documentTypeForLocking = CamsConstants.DocumentTypeName.ASSET_PAYMENT_FROM_CAB;
            }

            if (!getCapitalAssetManagementModuleService().storeAssetLocks(capitalAssetNumbers,
                    getDocumentNumber(), documentTypeForLocking, null)) {
                throw new ValidationException("Asset " + capitalAssetNumbers.toString() +
                        " is being locked by other documents.");
            }
        }
    }

    protected CapitalAssetManagementModuleService getCapitalAssetManagementModuleService() {
        return SpringContext.getBean(CapitalAssetManagementModuleService.class);
    }

    @Override
    public void doRouteStatusChange(final DocumentRouteStatusChange statusChangeEvent) {
        super.doRouteStatusChange(statusChangeEvent);
        final WorkflowDocument workflowDocument = getDocumentHeader().getWorkflowDocument();

        // Update asset payment table with the approved asset detail records.
        if (workflowDocument.isProcessed()) {
            SpringContext.getBean(AssetPaymentService.class).processApprovedAssetPayment(this);
        }

        // Remove asset lock when doc status change. We don't include
        // isFinal since document always go to 'processed' first.
        if (workflowDocument.isCanceled() || workflowDocument.isDisapproved() || workflowDocument.isProcessed()) {
            getCapitalAssetManagementModuleService().deleteAssetLocks(getDocumentNumber(), null);
        }

        if (isCapitalAssetBuilderOriginIndicator()) {
            SpringContext.getBean(CapitalAssetManagementModuleService.class)
                    .notifyRouteStatusChange(getDocumentHeader());
        }
    }

    @Override
    public void prepareForSave(final KualiDocumentEvent event) {
        for (final AssetPaymentAssetDetail assetDetail : getAssetPaymentAssetDetail()) {
            assetDetail.refreshReferenceObject(CamsPropertyConstants.AssetPaymentAssetDetail.ASSET);
            if (ObjectUtils.isNotNull(assetDetail.getAsset()) && assetDetail.getAsset().getTotalCostAmount() != null) {
                assetDetail.setPreviousTotalCostAmount(assetDetail.getAsset().getTotalCostAmount());
            }

            final List<AssetPaymentDetail> apdList = assetDetail.getAssetPaymentDetails();
            for (final AssetPaymentDetail apd : apdList) {
                final String accountingPeriodCompositeString = getAccountingPeriodCompositeString();
                apd.setPostingYear(new Integer(StringUtils.right(accountingPeriodCompositeString, 4)));
                apd.setPostingPeriodCode(StringUtils.left(accountingPeriodCompositeString, 2));
            }
        }

        super.prepareForSave(event);
    }

    @Override
    public void clearAnyGeneralLedgerPendingEntries() {
        /*Reset the transfer payment indicators.*/
        final List<GeneralLedgerPendingEntrySourceDetail> pendingEntrySourceDetails =
                getGeneralLedgerPendingEntrySourceDetails();
        for (final GeneralLedgerPendingEntrySourceDetail sourceDetail : pendingEntrySourceDetails) {
            final AssetPaymentDetail detail = (AssetPaymentDetail) sourceDetail;
            detail.setTransferPaymentIndicator(false);
        }
        super.clearAnyGeneralLedgerPendingEntries();
    }

    /**
     * For source accounting lines that have asset payments selected, we set payment transfer codes to Y. Once that is
     * done, when the payments are created (AssetPaymentServiceImpl.processPayments) accumulated depreciation and
     * current period depreciation will also be set. Furthermore we prepare two variables that are needed for
     * custom GLPE generation
     */
    @Override
    public boolean generateGeneralLedgerPendingEntries(
            final GeneralLedgerPendingEntrySourceDetail glpeSourceDetail,
            final GeneralLedgerPendingEntrySequenceHelper sequenceHelper) {
        LOG.debug("processGenerateGeneralLedgerPendingEntries(AccountingDocument, AccountingLine, " +
                "GeneralLedgerPendingEntrySequenceHelper) - start");

        boolean success = true;
        final AssetPaymentDetail assetPaymentDetail = (AssetPaymentDetail) glpeSourceDetail;

        // Loop over all selected assets
        for (final AssetPaymentAssetDetail assetDetail : getAssetPaymentAssetDetail()) {
            // Reset the amount -- GLPEs are created per asset
            apipAssetPaymentTotalDepreciationHelper = new KualiDecimal(0);

            // Loop over the payment selections the user made for each asset
            for (final AssetPaymentInProcessPayment assetPaymentInProcessPayment : assetPaymentInProcessPayments) {
                if (assetPaymentInProcessPayment.getCapitalAssetNumber().equals(assetDetail.getCapitalAssetNumber())
                        && assetPaymentInProcessPayment.getSequenceNumber().equals(
                                assetPaymentDetail.getSequenceNumber())) {
                    assetPaymentInProcessPayment.refreshReferenceObject(
                            CamsPropertyConstants.AssetPaymentInProcessPayments.ASSET_PAYMENTS);
                    final AssetPayment apipAssetPayment = assetPaymentInProcessPayment.getAssetPayment();

                    if (ObjectUtils.isNotNull(apipAssetPayment.getAccumulatedPrimaryDepreciationAmount())) {
                        apipAssetPaymentTotalDepreciationHelper = apipAssetPaymentTotalDepreciationHelper
                                .add(apipAssetPayment.getAccumulatedPrimaryDepreciationAmount());
                    }

                    // This will be repeated on each iteration, that's fine -- the accounting key will be the same
                    // on all of these per the UI selection restrictions
                    apipAssetPaymentHelper = apipAssetPayment;
                    assetPaymentDetail.setTransferPaymentIndicator(true);
                }
            }

            // Make sure we don't get 0$ GLPE lines
            if (apipAssetPaymentTotalDepreciationHelper.isNonZero()) {
                success &= super.generateGeneralLedgerPendingEntries(glpeSourceDetail, sequenceHelper);
                sequenceHelper.increment();
            }
        }

        return success;
    }

    /**
     * Sets custom plant fund account, fields in support of AssetObjectCode, transaction amount, and debitCreditCode
     * per depreciation requirements
     */
    @Override
    public void customizeExplicitGeneralLedgerPendingEntry(
            final GeneralLedgerPendingEntrySourceDetail postable,
            final GeneralLedgerPendingEntry explicitEntry) {
        determinePlantFund(postable.getObjectCode().getFinancialObjectSubTypeCode(), explicitEntry);

        final AssetObjectCodeService assetObjectCodeService = SpringContext.getBean(AssetObjectCodeService.class);
        final AssetObjectCode assetObjectCode = assetObjectCodeService.findAssetObjectCode(
                apipAssetPaymentHelper.getChartOfAccountsCode(),
                apipAssetPaymentHelper.getFinancialObject().getFinancialObjectSubTypeCode());
        explicitEntry.setFinancialObjectCode(assetObjectCode.getAccumulatedDepreciationFinancialObjectCode());
        explicitEntry.setFinancialObjectTypeCode(assetObjectCode.getAccumulatedDepreciationFinancialObject()
                .getFinancialObjectTypeCode());

        explicitEntry.setTransactionLedgerEntryAmount(apipAssetPaymentTotalDepreciationHelper.abs());
        explicitEntry.setTransactionLedgerEntryDescription(
                CamsConstants.Depreciation.CORRECTION_TRANSACTION_DESCRIPTION +
                        apipAssetPaymentHelper.getCapitalAssetNumber());

        if (apipAssetPaymentTotalDepreciationHelper.isPositive()) {
            explicitEntry.setTransactionDebitCreditCode(KFSConstants.GL_DEBIT_CODE);
        } else {
            explicitEntry.setTransactionDebitCreditCode(KFSConstants.GL_CREDIT_CODE);
        }

        explicitEntry.setSubAccountNumber(
                AccountingDocumentRuleBaseConstants.GENERAL_LEDGER_PENDING_ENTRY_CODE.getBlankSubAccountNumber());
        explicitEntry.setFinancialSubObjectCode(
                AccountingDocumentRuleBaseConstants.GENERAL_LEDGER_PENDING_ENTRY_CODE.getBlankFinancialSubObjectCode());
    }

    /**
     * Sets custom plant fund account, fields in support of AssetObjectCode, transaction amount, and debitCreditCode
     * per depreciation requirements. This is for the offset entry.
     */
    @Override
    public boolean customizeOffsetGeneralLedgerPendingEntry(
            final GeneralLedgerPendingEntrySourceDetail accountingLine,
            final GeneralLedgerPendingEntry explicitEntry, final GeneralLedgerPendingEntry offsetEntry) {
        // on the explicitEntry we executed the plant fund account determination. No need to do that again
        offsetEntry.setAccountNumber(explicitEntry.getAccountNumber());
        offsetEntry.setChartOfAccountsCode(explicitEntry.getChartOfAccountsCode());

        final AssetObjectCodeService assetObjectCodeService = SpringContext.getBean(AssetObjectCodeService.class);
        final AssetObjectCode assetObjectCode = assetObjectCodeService.findAssetObjectCode(
                apipAssetPaymentHelper.getChartOfAccountsCode(),
                apipAssetPaymentHelper.getFinancialObject().getFinancialObjectSubTypeCode());
        offsetEntry.setFinancialObjectCode(assetObjectCode.getDepreciationExpenseFinancialObjectCode());
        offsetEntry.setFinancialObjectTypeCode(assetObjectCode.getDepreciationExpenseFinancialObject()
                .getFinancialObjectTypeCode());

        offsetEntry.setTransactionLedgerEntryAmount(explicitEntry.getTransactionLedgerEntryAmount());
        offsetEntry.setTransactionLedgerEntryDescription(explicitEntry.getTransactionLedgerEntryDescription());

        if (explicitEntry.getTransactionDebitCreditCode().equals(KFSConstants.GL_CREDIT_CODE)) {
            offsetEntry.setTransactionDebitCreditCode(KFSConstants.GL_DEBIT_CODE);
        } else {
            offsetEntry.setTransactionDebitCreditCode(KFSConstants.GL_CREDIT_CODE);
        }

        offsetEntry.setSubAccountNumber(
                AccountingDocumentRuleBaseConstants.GENERAL_LEDGER_PENDING_ENTRY_CODE.getBlankSubAccountNumber());
        offsetEntry.setFinancialSubObjectCode(
                AccountingDocumentRuleBaseConstants.GENERAL_LEDGER_PENDING_ENTRY_CODE.getBlankFinancialSubObjectCode());

        return true;
    }

    /**
     * Logic for determining how plant fund account should be set.
     * @param postableFinancialObjectSubTypeCode used to determined what the plant fund account should be per
     *                                           AssetObjectCode reference table
     * @param explicitEntry                      will have the accountNumber and chartOfAccountsCode set ot the plant
     *                                           fund account values
     */
    protected void determinePlantFund(
            final String postableFinancialObjectSubTypeCode,
            final GeneralLedgerPendingEntry explicitEntry) {
        String plantAccount = "";
        String plantCOA = "";

        final ParameterService parameterService = SpringContext.getBean(ParameterService.class);
        Collection<String> organizationPlantFundObjectSubType = new ArrayList<>();
        Collection<String> campusPlantFundObjectSubType = new ArrayList<>();
        if (parameterService.parameterExists(AssetDepreciationStep.class,
                CamsParameterConstants.ORGANIZATION_PLANT_FUND_SUB_TYPES)) {
            organizationPlantFundObjectSubType = new ArrayList<>(parameterService.getParameterValuesAsString(
                    AssetDepreciationStep.class,
                    CamsParameterConstants.ORGANIZATION_PLANT_FUND_SUB_TYPES));
        }
        if (parameterService.parameterExists(AssetDepreciationStep.class,
                CamsParameterConstants.CAMPUS_PLANT_FUND_SUB_TYPES)) {
            campusPlantFundObjectSubType = new ArrayList<>(parameterService.getParameterValuesAsString(
                    AssetDepreciationStep.class,
                    CamsParameterConstants.CAMPUS_PLANT_FUND_SUB_TYPES));
        }

        final Asset apipAssetPaymentAssetHelper = apipAssetPaymentHelper.getAsset();
        // getting the right Plant Fund Chart code & Plant Fund Account
        if (organizationPlantFundObjectSubType.contains(postableFinancialObjectSubTypeCode)) {
            plantAccount = apipAssetPaymentHelper.getAccount().getOrganization().getOrganizationPlantAccountNumber();
            plantCOA = apipAssetPaymentHelper.getAccount().getOrganization().getOrganizationPlantChartCode();
        } else if (campusPlantFundObjectSubType.contains(postableFinancialObjectSubTypeCode)) {
            plantAccount = apipAssetPaymentHelper.getAccount().getOrganization().getCampusPlantAccountNumber();
            plantCOA = apipAssetPaymentHelper.getAccount().getOrganization().getCampusPlantChartCode();
        }
        if (StringUtils.isBlank(plantCOA) || StringUtils.isBlank(plantAccount)) {
            final String loggablePlantCOA = plantCOA;
            final String loggablePlantAccount = plantAccount;
            LOG.error(
                    "Asset Manual Payment Plant COA is {} and plant account is {} for Financial Object SubType Code ="
                    + " {} so Asset payment will use postables account# and coaCd {} - {}",
                    () -> loggablePlantCOA,
                    () -> loggablePlantAccount,
                    apipAssetPaymentAssetHelper::getFinancialObjectSubTypeCode,
                    apipAssetPaymentAssetHelper::getCapitalAssetNumber,
                    apipAssetPaymentHelper::getPaymentSequenceNumber
            );
        } else {
            explicitEntry.setAccountNumber(plantAccount);
            explicitEntry.setChartOfAccountsCode(plantCOA);
        }
    }

    public List<AssetPaymentAssetDetail> getAssetPaymentAssetDetail() {
        return assetPaymentAssetDetail;
    }

    public void setAssetPaymentAssetDetail(final List<AssetPaymentAssetDetail> assetPaymentAssetDetail) {
        this.assetPaymentAssetDetail = assetPaymentAssetDetail;
    }

    public Long getCapitalAssetNumber() {
        return capitalAssetNumber;
    }

    public void setCapitalAssetNumber(final Long capitalAssetNumber) {
        this.capitalAssetNumber = capitalAssetNumber;
    }

    /**
     * calculates the total previous cost amount of all the assets in the document
     *
     * @return KualiDecimal
     */
    public KualiDecimal getAssetsTotalHistoricalCost() {
        KualiDecimal total = new KualiDecimal(0);
        if (getAssetPaymentAssetDetail().isEmpty()) {
            return new KualiDecimal(0);
        }

        for (final AssetPaymentAssetDetail detail : getAssetPaymentAssetDetail()) {
            final KualiDecimal amount = detail.getPreviousTotalCostAmount() == null ? new KualiDecimal(0) :
                    detail.getPreviousTotalCostAmount();
            total = total.add(amount);
        }
        return total;
    }

    /**
     * Get the asset payment distributor built by AssetPaymentDetails, AssetPaymentAssetDetail and totalHistoricalCost
     *
     * @return AssetPaymentDistributor
     */
    public AssetDistribution getAssetPaymentDistributor() {
        if (CamsPropertyConstants.AssetPaymentAllocation.ASSET_DISTRIBUTION_BY_AMOUNT_CODE.equals(
                getAssetPaymentAllocationTypeCode())) {
            return new AssetDistributionManual(this);
        } else if (CamsPropertyConstants.AssetPaymentAllocation.ASSET_DISTRIBUTION_BY_PERCENTAGE_CODE.equals(
                getAssetPaymentAllocationTypeCode())) {
            return new AssetDistributionPercent(this);
        } else if (CamsPropertyConstants.AssetPaymentAllocation.ASSET_DISTRIBUTION_BY_TOTAL_COST_CODE.equals(
                getAssetPaymentAllocationTypeCode())) {
            return new AssetPaymentDistributionByTotalCost(this);
        }
        return new AssetDistributionEvenly(this);

    }

    public String getAssetPaymentAllocationTypeCode() {
        return assetPaymentAllocationTypeCode;
    }

    public void setAssetPaymentAllocationTypeCode(final String code) {
        assetPaymentAllocationTypeCode = code;
        refreshReferenceObject("assetPaymentAllocationType");
    }

    public void setAssetPaymentAllocationType(final AssetPaymentAllocationType assetPaymentAllocationType) {
        this.assetPaymentAllocationType = assetPaymentAllocationType;
    }

    public AssetPaymentAllocationType getAssetPaymentAllocationType() {
        return assetPaymentAllocationType;
    }

    public boolean isAllocationFromFPDocuments() {
        return allocationFromFPDocuments;
    }

    public void setAllocationFromFPDocuments(final boolean allocationFromFPDocuments) {
        this.allocationFromFPDocuments = allocationFromFPDocuments;
    }

    public List<AssetPaymentInProcessPayment> getAssetPaymentInProcessPayments() {
        return assetPaymentInProcessPayments;
    }

    public void setAssetPaymentInProcessPayments(final List<AssetPaymentInProcessPayment> assetPaymentInProcessPayments) {
        this.assetPaymentInProcessPayments = assetPaymentInProcessPayments;
    }
}
