/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cam.document.service.impl;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.krad.exception.ValidationException;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.cam.CamsConstants;
import org.kuali.kfs.module.cam.CamsParameterConstants;
import org.kuali.kfs.module.cam.CamsPropertyConstants;
import org.kuali.kfs.module.cam.businessobject.Asset;
import org.kuali.kfs.module.cam.businessobject.AssetDepreciationConvention;
import org.kuali.kfs.module.cam.businessobject.AssetGlobal;
import org.kuali.kfs.module.cam.businessobject.AssetType;
import org.kuali.kfs.module.cam.document.service.AssetDateService;
import org.kuali.kfs.module.cam.document.service.AssetService;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.businessobject.UniversityDate;
import org.kuali.kfs.sys.service.UniversityDateService;
import org.kuali.kfs.core.api.datetime.DateTimeService;

import java.sql.Date;
import java.sql.Timestamp;
import java.time.LocalDate;
import java.util.HashMap;
import java.util.Map;

public class AssetDateServiceImpl implements AssetDateService {

    private static final Logger LOG = LogManager.getLogger();

    private AssetService assetService;
    private UniversityDateService universityDateService;
    private DateTimeService dateTimeService;
    private BusinessObjectService businessObjectService;
    private ParameterService parameterService;

    @Override
    public void checkAndUpdateFiscalYearAndPeriod(final Asset oldAsset, final Asset newAsset) {
        if (assetService.isInServiceDateChanged(oldAsset, newAsset)) {
            Integer newPostingYear = null;
            String newPostingPeriodCode = null;

            if (ObjectUtils.isNotNull(newAsset.getCapitalAssetInServiceDate())) {
                final Map<String, Object> primaryKeys = new HashMap<>();
                primaryKeys.put(KFSPropertyConstants.UNIVERSITY_DATE, newAsset.getCapitalAssetInServiceDate());
                final UniversityDate inServiceDate =
                        businessObjectService.findByPrimaryKey(UniversityDate.class, primaryKeys);

                if (ObjectUtils.isNotNull(inServiceDate)) {
                    newPostingYear = inServiceDate.getUniversityFiscalYear();
                    newPostingPeriodCode = inServiceDate.getUniversityFiscalAccountingPeriod();
                }
            }
            // if the user blank in-service date, posting year and period code will be blank also.
            newAsset.setFinancialDocumentPostingYear(newPostingYear);
            newAsset.setFinancialDocumentPostingPeriodCode(newPostingPeriodCode);
        }
    }

    @Override
    public void checkAndUpdateLastInventoryDate(final Asset oldAsset, final Asset newAsset) {
        if (assetService.hasAssetLocationChanged(oldAsset, newAsset)) {
            final Timestamp timestamp = new Timestamp(dateTimeService.getCurrentDate().getTime());
            newAsset.setLastInventoryDate(timestamp);
        }
    }

    @Override
    public void checkAndUpdateDepreciationDate(final Asset oldAsset, final Asset newAsset) {
        if (assetService.isAssetTypeCodeChanged(oldAsset, newAsset) && assetService.isAssetDepreciableLifeLimitZero(newAsset)) {
            // If Asset Type changed to Depreciable Life Limit to "0", set both In-service Date and Depreciation Date to NULL.
            newAsset.setDepreciationDate(null);
            newAsset.setCapitalAssetInServiceDate(null);
        } else if (assetService.isInServiceDateChanged(oldAsset, newAsset)
                || assetService.isFinancialObjectSubTypeCodeChanged(oldAsset, newAsset)) {
            final Map<String, String> primaryKeys = new HashMap<>();
            primaryKeys.put(CamsPropertyConstants.AssetDepreciationConvention.FINANCIAL_OBJECT_SUB_TYPE_CODE, newAsset.getFinancialObjectSubTypeCode());
            final AssetDepreciationConvention depreciationConvention = businessObjectService.findByPrimaryKey(AssetDepreciationConvention.class, primaryKeys);
            newAsset.setDepreciationDate(computeDepreciationDate(newAsset.getCapitalAssetType(), depreciationConvention, newAsset.getCapitalAssetInServiceDate()));
        }
    }

    @Override
    public Date computeDepreciationDate(
            final AssetType assetType, final AssetDepreciationConvention depreciationConvention,
            final Date inServiceDate) {
        Date depreciationDate = null;
        if (assetType.getDepreciableLifeLimit() != null && assetType.getDepreciableLifeLimit() != 0) {
            if (depreciationConvention == null
                    || !depreciationConvention.isActive()
                    || CamsConstants.DepreciationConvention.CREATE_DATE.equalsIgnoreCase(
                            depreciationConvention.getDepreciationConventionCode())) {
                depreciationDate = inServiceDate;
            } else {
                final Integer fiscalYear = universityDateService.getFiscalYear(inServiceDate);
                if (fiscalYear == null) {
                    throw new ValidationException("University Fiscal year is not defined for date - " + inServiceDate);
                }

                final String newInServiceAssetDepreciationStartMMDD = parameterService.getParameterValueAsString(
                        AssetGlobal.class, CamsParameterConstants.NEW_IN_SERVICE_ASSET_DEPRECIATION_START_DATE);
                final Date newInServiceFiscalYearStartDate;
                try {
                    newInServiceFiscalYearStartDate = dateTimeService
                            .convertToSqlDate(newInServiceAssetDepreciationStartMMDD + "/" + fiscalYear);
                } catch (final Exception e) {
                    //issue to construct the new in service fiscal year start date
                    final String error = "Unable to construct new in service fiscal year start date - param value: [ " +
                                         newInServiceAssetDepreciationStartMMDD + "]";
                    LOG.error(error, e);
                    throw new IllegalArgumentException(error, e);
                }
                final String conventionCode = depreciationConvention.getDepreciationConventionCode();

                // when depreciationConventionCode = FY then NEW_IN_SERVICE_ASSET_DEPRECIATION_START_DATE + fiscal
                // year and then subtracting 1 from the year to set the depreciation date
                if (CamsConstants.DepreciationConvention.FULL_YEAR.equalsIgnoreCase(conventionCode)) {
                    final LocalDate localDate =
                            dateTimeService.getLocalDate(newInServiceFiscalYearStartDate).minusYears(1L);
                    depreciationDate = Date.valueOf(localDate);
                } else if (CamsConstants.DepreciationConvention.HALF_YEAR.equalsIgnoreCase(conventionCode)) {
                    //when depreciationConventionCode = HY
                    // Half year depreciation convention
                    //start the month and date as "01/01" and concat the fiscal year...
                    final Date newInServiceFiscalYearStartDateForHalfYear;
                    try {
                        newInServiceFiscalYearStartDateForHalfYear = dateTimeService.convertToSqlDate("01/01/"
                                + fiscalYear);
                    } catch (final Exception e) {
                        //issue to construct the new in service fiscal year start date
                        final String error = "Unable to construct new in service fiscal year start date";
                        LOG.error(error, e);
                        throw new IllegalArgumentException(error, e);
                    }

                    depreciationDate = newInServiceFiscalYearStartDateForHalfYear;
                }
            }
        }
        return depreciationDate;
    }

    public AssetService getAssetService() {
        return assetService;
    }

    public void setAssetService(final AssetService assetService) {
        this.assetService = assetService;
    }

    public UniversityDateService getUniversityDateService() {
        return universityDateService;
    }

    public void setUniversityDateService(final UniversityDateService universityDateService) {
        this.universityDateService = universityDateService;
    }

    public DateTimeService getDateTimeService() {
        return dateTimeService;
    }

    public void setDateTimeService(final DateTimeService dateTimeService) {
        this.dateTimeService = dateTimeService;
    }

    public BusinessObjectService getBusinessObjectService() {
        return businessObjectService;
    }

    public void setBusinessObjectService(final BusinessObjectService businessObjectService) {
        this.businessObjectService = businessObjectService;
    }

    @Override
    public Date computeDepreciationDateForPeriod13(
            final String depreciationConventionCode,
            final Date inServiceDate) {
        Date depreciationDate = null;
        final Integer fiscalYear = getUniversityDateService().getFiscalYear(inServiceDate);
        final Date fiscalYearStartDate =
                new Date(getUniversityDateService().getFirstDateOfFiscalYear(fiscalYear).getTime());

        if (CamsConstants.DepreciationConvention.CREATE_DATE.equals(depreciationConventionCode)) {
            depreciationDate = inServiceDate;
        } else if (CamsConstants.DepreciationConvention.FULL_YEAR.equals(depreciationConventionCode)) {
            depreciationDate = fiscalYearStartDate;
        } else if (CamsConstants.DepreciationConvention.HALF_YEAR.equals(depreciationConventionCode)) {
            // Half year depreciation convention
            final LocalDate localDate = dateTimeService.getLocalDate(fiscalYearStartDate).plusMonths(6L);
            depreciationDate = Date.valueOf(localDate);
        }
        return depreciationDate;
    }

    public void setParameterService(final ParameterService parameterService) {
        this.parameterService = parameterService;
    }
}
