/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cam.document.validation.impl;

import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.module.cam.CamsKeyConstants;
import org.kuali.kfs.module.cam.CamsPropertyConstants;
import org.kuali.kfs.module.cam.businessobject.AssetPaymentDetail;
import org.kuali.kfs.module.cam.businessobject.AssetPaymentInProcessPayment;
import org.kuali.kfs.module.cam.document.AssetPaymentDocument;
import org.kuali.kfs.sys.document.validation.GenericValidation;
import org.kuali.kfs.sys.document.validation.event.AttributedDocumentEvent;
import org.kuali.kfs.core.api.util.type.KualiDecimal;

import java.util.List;

public class AssetPaymentInProcessPaymentValidation extends GenericValidation {

    @Override
    public boolean validate(final AttributedDocumentEvent event) {
        boolean valid = true;

        final AssetPaymentDocument assetPaymentDocument = (AssetPaymentDocument) event.getDocument();

        if (!assetPaymentDocument.getAssetPaymentInProcessPayments().isEmpty()) {
            valid = validateAccountingLinesToInProcessPaymentsSumToZero(assetPaymentDocument.getSourceAccountingLines(),
                    assetPaymentDocument.getAssetPaymentInProcessPayments());
        }

        return valid;
    }

    /**
     * Validates that the sum of selected payments on an In Process payment added to the corresponding source accounting
     * line equals zero.
     *
     * @param sourceAccountingLines         corresponding to each assetPaymentInProcessPayments by document number and
     *                                      sequence number
     * @param assetPaymentInProcessPayments the in process payments for the source accounting lines. Each has a selected
     *                                      asset payment associated with it
     * @return true if all in validations passed. False if at least one failed
     */
    protected boolean validateAccountingLinesToInProcessPaymentsSumToZero(
            final List<AssetPaymentDetail> sourceAccountingLines,
            final List<AssetPaymentInProcessPayment> assetPaymentInProcessPayments) {
        boolean valid = true;

        for (final AssetPaymentDetail sourceAccountingLine : sourceAccountingLines) {
            final Integer sequenceNumber = sourceAccountingLine.getSequenceNumber();
            Long capitalAssetNumber = 0L;

            KualiDecimal totalInProcessPayment = new KualiDecimal(0);
            for (final AssetPaymentInProcessPayment assetPaymentInProcessPayment : assetPaymentInProcessPayments) {
                if (sequenceNumber.equals(assetPaymentInProcessPayment.getSequenceNumber())) {
                    assetPaymentInProcessPayment.refreshReferenceObject(
                            CamsPropertyConstants.AssetPaymentInProcessPayments.ASSET_PAYMENTS);
                    totalInProcessPayment = totalInProcessPayment.add(
                            assetPaymentInProcessPayment.getAssetPayment().getAccountChargeAmount());
                    capitalAssetNumber = assetPaymentInProcessPayment.getCapitalAssetNumber();
                }
            }

            if (totalInProcessPayment.isNonZero()) {
                if (totalInProcessPayment.add(sourceAccountingLine.getAmount()).isNonZero()) {
                    GlobalVariables.getMessageMap().putErrorForSectionId(
                            CamsPropertyConstants.ASSET_IN_PROCESS_PAYMENT_ERROR_SECTION_ID +
                                    capitalAssetNumber + "." + sequenceNumber,
                            CamsKeyConstants.Payment.ERROR_IN_PROCESS_PAYMENT_NOT_BALANCE);
                    valid = false;
                }
            }
        }

        return valid;
    }

}
