/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cam.report;

import com.lowagie.text.Cell;
import com.lowagie.text.Chunk;
import com.lowagie.text.Document;
import com.lowagie.text.DocumentException;
import com.lowagie.text.Element;
import com.lowagie.text.ExceptionConverter;
import com.lowagie.text.Font;
import com.lowagie.text.FontFactory;
import com.lowagie.text.Paragraph;
import com.lowagie.text.Phrase;
import com.lowagie.text.Rectangle;
import com.lowagie.text.Table;
import com.lowagie.text.pdf.PdfPCell;
import com.lowagie.text.pdf.PdfPTable;
import com.lowagie.text.pdf.PdfPageEventHelper;
import com.lowagie.text.pdf.PdfWriter;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.module.cam.CamsConstants;
import org.kuali.kfs.module.cam.CamsKeyConstants;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.core.api.config.property.ConfigurationService;
import org.kuali.kfs.core.api.datetime.DateTimeService;

import java.io.File;
import java.io.FileOutputStream;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Locale;

public class DepreciationReport {


    private static final Logger LOG = LogManager.getLogger();
    private int pageNumber;
    private int line;
    private final int linesPerPage = 28;
    private Document document;
    private PdfWriter writer;

    /**
     * This method creates the report file and invokes the methods that write the data
     *
     * @param reportLog
     * @param errorMsg
     */
    public void generateReport(final List<String[]> reportLog, final String errorMsg, final String sDepreciationDate) {
        try {
            final DateTimeService dateTimeService = SpringContext.getBean(DateTimeService.class);
            LOG.debug("createReport() started");
            document = new Document();

            final String destinationDirectory = SpringContext.getBean(ConfigurationService.class).getPropertyValueAsString(
                    KFSConstants.REPORTS_DIRECTORY_KEY);

            final SimpleDateFormat sdf = new SimpleDateFormat(CamsConstants.DateFormats.YEAR_MONTH_DAY_NO_DELIMITER +
                                                              "_" + CamsConstants.DateFormats.MILITARY_TIME_NO_DELIMITER, Locale.US);

            final String filename = destinationDirectory + File.separator + "cam" + File.separator +
                                    CamsConstants.Report.FILE_PREFIX + "_" + CamsConstants.Depreciation.REPORT_FILE_NAME + "_" +
                                    sdf.format(dateTimeService.getCurrentDate()) + "." + CamsConstants.Report.REPORT_EXTENSION;

            final Font headerFont = FontFactory.getFont(FontFactory.HELVETICA, 12, Font.NORMAL);

            final PageHelper helper = new PageHelper();
            helper.runDate = dateTimeService.getCurrentDate();
            helper.headerFont = headerFont;
            helper.title = CamsConstants.Depreciation.DEPRECIATION_REPORT_TITLE;

            writer = PdfWriter.getInstance(document, new FileOutputStream(filename));
            writer.setPageEvent(helper);

            document.open();

            // Generate body of document.
            generateReportLogBody(reportLog);
            generateReportErrorLog(errorMsg);

        } catch (final Exception e) {
            throw new RuntimeException("DepreciationReport.generateReport(List<String[]> reportLog, " +
                    "List<String> errorLog) - Error on report generation: " + e.getMessage());
        } finally {
            if (document != null && document.isOpen()) {
                document.close();
            }
        }
    }

    /**
     * This method adds the log lines into the report
     *
     * @param reportLog
     */
    private void generateReportLogBody(final List<String[]> reportLog) {
        try {
            final Font font = FontFactory.getFont(FontFactory.HELVETICA, 9, Font.NORMAL);
            final int[] columnWidths = new int[]{40, 15};

            Table aTable = new Table(2, linesPerPage);
            int rowsWritten = 0;
            for (final String[] columns : reportLog) {
                if (pageNumber == 0 || line >= linesPerPage) {
                    if (pageNumber > 0) {
                        document.add(aTable);
                    }
                    final int elementsLeft = reportLog.size() - rowsWritten;
                    final int rowsNeeded = elementsLeft >= linesPerPage ? linesPerPage : elementsLeft;
                    document.newPage();

                    generateColumnHeaders();

                    aTable = new Table(2, rowsNeeded);

                    aTable.setAutoFillEmptyCells(true);
                    aTable.setPadding(3);
                    aTable.setWidths(columnWidths);
                    aTable.setWidth(100);
                    aTable.setBorder(Rectangle.NO_BORDER);

                    line = 0;
                    pageNumber++;
                }
                rowsWritten++;

                Cell cell;
                cell = new Cell(new Phrase(columns[0], font));
                cell.setHorizontalAlignment(Element.ALIGN_LEFT);
                cell.setVerticalAlignment(Element.ALIGN_MIDDLE);
                aTable.addCell(cell);

                cell = new Cell(new Phrase(columns[1], font));
                cell.setHorizontalAlignment(Element.ALIGN_RIGHT);
                cell.setVerticalAlignment(Element.ALIGN_MIDDLE);
                aTable.addCell(cell);
                line++;
            }
            document.add(aTable);
        } catch (final DocumentException de) {
            throw new RuntimeException("DepreciationReport.generateReportLogBody(List<String[]> reportLog) - error: " +
                    de.getMessage());
        }
    }

    /**
     * This method adds any error to the report
     *
     * @param errorMsg
     */
    private void generateReportErrorLog(final String errorMsg) {
        try {
            final Font font = FontFactory.getFont(FontFactory.HELVETICA, 9, Font.NORMAL);

            if (StringUtils.isNotEmpty(errorMsg)) {
                generateErrorColumnHeaders();

                final Paragraph p1 = new Paragraph(new Chunk(errorMsg, font));
                document.add(p1);
                line++;
            }
        } catch (final Exception de) {
            throw new RuntimeException("DepreciationReport.generateReportErrorLog(List<String> reportLog) - " +
                    "Report Generation Failed: " + de.getMessage());
        }
    }

    /**
     * This method creates a report group for the error message on the report
     */
    private void generateErrorColumnHeaders() {
        try {
            final int[] headerWidths = {60};

            final Table aTable = new Table(1, 1);

            aTable.setAutoFillEmptyCells(true);
            aTable.setPadding(3);
            aTable.setWidths(headerWidths);
            aTable.setWidth(100);

            final Cell cell;

            final Font font = FontFactory.getFont(FontFactory.HELVETICA, 9, Font.NORMAL);

            cell = new Cell(new Phrase("Error(s)", font));
            cell.setHorizontalAlignment(Element.ALIGN_CENTER);
            cell.setVerticalAlignment(Element.ALIGN_MIDDLE);
            cell.setGrayFill(0.9f);
            aTable.addCell(cell);

            document.add(aTable);

        } catch (final Exception e) {
            throw new RuntimeException("DepreciationReport.generateErrorColumnHeaders() - Error: " + e.getMessage());
        }
    }

    /**
     * This method creates the headers for the report statistics
     */
    private void generateColumnHeaders() {
        try {
            final int[] headerWidths = {40, 15};

            final Table aTable = new Table(2, 1);

            aTable.setAutoFillEmptyCells(true);
            aTable.setPadding(3);
            aTable.setWidths(headerWidths);
            aTable.setWidth(100);

            Cell cell;

            final Font font = FontFactory.getFont(FontFactory.HELVETICA, 9, Font.NORMAL);

            cell = new Cell(new Phrase(SpringContext.getBean(ConfigurationService.class)
                    .getPropertyValueAsString(CamsKeyConstants.Depreciation.MSG_REPORT_DEPRECIATION_HEADING1), font));
            cell.setHorizontalAlignment(Element.ALIGN_CENTER);
            cell.setVerticalAlignment(Element.ALIGN_MIDDLE);
            cell.setGrayFill(0.9f);
            aTable.addCell(cell);

            cell = new Cell(new Phrase(SpringContext.getBean(ConfigurationService.class)
                    .getPropertyValueAsString(CamsKeyConstants.Depreciation.MSG_REPORT_DEPRECIATION_HEADING2), font));
            cell.setHorizontalAlignment(Element.ALIGN_CENTER);
            cell.setVerticalAlignment(Element.ALIGN_MIDDLE);
            cell.setGrayFill(0.9f);
            aTable.addCell(cell);
            document.add(aTable);

        } catch (final Exception e) {
            throw new RuntimeException("DepreciationReport.generateColumnHeaders() - Error: " + e.getMessage());
        }
    }

    /**
     * An inner class to help set up the PDF that is written
     */
    class PageHelper extends PdfPageEventHelper {
        public Date runDate;
        public Font headerFont;
        public String title;

        /**
         * Writes the footer on the last page
         */
        @Override
        public void onEndPage(final PdfWriter writer, final Document document) {
            try {
                final Font titleFont = FontFactory.getFont(FontFactory.HELVETICA, 12, Font.NORMAL);
                final Font font = FontFactory.getFont(FontFactory.HELVETICA, 8, Font.NORMAL);

                final Rectangle page = document.getPageSize();
                final PdfPTable head = new PdfPTable(3);

                final int[] widths = {15, 70, 15};
                head.setWidths(widths);

                final SimpleDateFormat sdf = new SimpleDateFormat(CamsConstants.DateFormats.MONTH_DAY_YEAR + " " +
                                                                  CamsConstants.DateFormats.MILITARY_TIME, Locale.US);

                PdfPCell cell = new PdfPCell(new Phrase(sdf.format(runDate), font));
                cell.setBorder(Rectangle.NO_BORDER);
                head.addCell(cell);

                cell = new PdfPCell(new Phrase(title, titleFont));
                cell.setBorder(Rectangle.NO_BORDER);
                cell.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
                head.addCell(cell);

                cell = new PdfPCell(new Phrase("Page: " + writer.getPageNumber(), font));
                cell.setBorder(Rectangle.NO_BORDER);
                cell.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
                head.addCell(cell);

                head.setTotalWidth(page.getWidth() - document.leftMargin() - document.rightMargin());
                head.writeSelectedRows(0, -1, document.leftMargin(),
                        page.getHeight() - document.topMargin() + head.getTotalHeight(), writer.getDirectContent());
            } catch (final Exception e) {
                throw new ExceptionConverter(e);
            }
        }
    }
}
