/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2021 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cam.document;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.coa.businessobject.Account;
import org.kuali.kfs.coa.businessobject.Chart;
import org.kuali.kfs.coa.service.AccountService;
import org.kuali.kfs.fp.document.TransferOfFundsDocument;
import org.kuali.kfs.integration.cam.CapitalAssetManagementModuleService;
import org.kuali.kfs.krad.exception.ValidationException;
import org.kuali.kfs.krad.rules.rule.event.KualiDocumentEvent;
import org.kuali.kfs.krad.rules.rule.event.SaveDocumentEvent;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.cam.CamsConstants;
import org.kuali.kfs.module.cam.businessobject.Asset;
import org.kuali.kfs.module.cam.businessobject.AssetGlpeSourceDetail;
import org.kuali.kfs.module.cam.document.service.AssetTransferService;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.businessobject.Building;
import org.kuali.kfs.sys.businessobject.Campus;
import org.kuali.kfs.sys.businessobject.Country;
import org.kuali.kfs.sys.businessobject.GeneralLedgerPendingEntry;
import org.kuali.kfs.sys.businessobject.GeneralLedgerPendingEntrySequenceHelper;
import org.kuali.kfs.sys.businessobject.GeneralLedgerPendingEntrySourceDetail;
import org.kuali.kfs.sys.businessobject.PostalCode;
import org.kuali.kfs.sys.businessobject.Room;
import org.kuali.kfs.sys.businessobject.State;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.sys.document.GeneralLedgerPendingEntrySource;
import org.kuali.kfs.sys.document.GeneralLedgerPostingDocumentBase;
import org.kuali.kfs.sys.service.GeneralLedgerPendingEntryService;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.kew.api.WorkflowDocument;
import org.kuali.kfs.kew.framework.postprocessor.DocumentRouteStatusChange;
import org.kuali.kfs.kim.api.identity.Person;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class AssetTransferDocument extends GeneralLedgerPostingDocumentBase implements GeneralLedgerPendingEntrySource {

    protected String hiddenFieldForError;
    protected String representativeUniversalIdentifier;
    protected String campusCode;
    protected String buildingCode;
    protected String buildingRoomNumber;
    protected String buildingSubRoomNumber;
    protected String organizationTagNumber;
    protected String organizationOwnerChartOfAccountsCode;
    protected String organizationOwnerAccountNumber;
    protected String organizationText;
    protected String organizationInventoryName;
    protected String transferOfFundsFinancialDocumentNumber;
    protected String offCampusAddress;
    protected String offCampusCityName;
    protected String offCampusStateCode;
    protected String offCampusZipCode;
    protected String oldOrganizationOwnerChartOfAccountsCode;
    protected String oldOrganizationOwnerAccountNumber;
    protected String offCampusName;
    protected String offCampusCountryCode;
    protected boolean interdepartmentalSalesIndicator;
    protected Long capitalAssetNumber;
    protected Person assetRepresentative;
    protected Campus campus;
    protected Account organizationOwnerAccount;
    protected Account oldOrganizationOwnerAccount;
    protected Chart organizationOwnerChartOfAccounts;
    protected Country offCampusCountry;
    protected State offCampusState;
    protected Building building;
    protected Room buildingRoom;
    protected transient List<AssetGlpeSourceDetail> sourceAssetGlpeSourceDetails;
    protected transient List<AssetGlpeSourceDetail> targetAssetGlpeSourceDetails;
    protected Asset asset;
    protected PostalCode postalZipCode;

    public AssetTransferDocument() {
        super();
        this.sourceAssetGlpeSourceDetails = new ArrayList<>();
        this.targetAssetGlpeSourceDetails = new ArrayList<>();
    }

    public void customizeExplicitGeneralLedgerPendingEntry(GeneralLedgerPendingEntrySourceDetail postable,
            GeneralLedgerPendingEntry explicitEntry) {
    }

    public boolean customizeOffsetGeneralLedgerPendingEntry(GeneralLedgerPendingEntrySourceDetail accountingLine,
            GeneralLedgerPendingEntry explicitEntry, GeneralLedgerPendingEntry offsetEntry) {
        return false;
    }

    public boolean generateDocumentGeneralLedgerPendingEntries(GeneralLedgerPendingEntrySequenceHelper sequenceHelper) {
        return true;
    }

    public Asset getAsset() {
        return asset;
    }

    public Person getAssetRepresentative() {
        assetRepresentative = SpringContext.getBean(org.kuali.kfs.kim.api.identity.PersonService.class)
                .updatePersonIfNecessary(representativeUniversalIdentifier, assetRepresentative);
        return assetRepresentative;
    }

    public void setAssetRepresentative(Person assetRepresentative) {
        this.assetRepresentative = assetRepresentative;
    }

    public Building getBuilding() {
        return building;
    }

    public String getBuildingCode() {
        return buildingCode;
    }

    public Room getBuildingRoom() {
        return buildingRoom;
    }

    public String getBuildingRoomNumber() {
        return buildingRoomNumber;
    }

    public String getBuildingSubRoomNumber() {
        return buildingSubRoomNumber;
    }

    public Campus getCampus() {
        return campus;
    }

    public String getCampusCode() {
        return campusCode;
    }

    public String getDocumentNumber() {
        return documentNumber;
    }

    public KualiDecimal getGeneralLedgerPendingEntryAmountForDetail(GeneralLedgerPendingEntrySourceDetail postable) {
        return postable.getAmount().abs();
    }

    public List<AssetGlpeSourceDetail> getSourceAssetGlpeSourceDetails() {
        if (this.sourceAssetGlpeSourceDetails == null) {
            this.sourceAssetGlpeSourceDetails = new ArrayList<>();
        }
        return this.sourceAssetGlpeSourceDetails;
    }

    public boolean generateGeneralLedgerPendingEntries(GeneralLedgerPendingEntrySourceDetail postable,
            GeneralLedgerPendingEntrySequenceHelper sequenceHelper) {
        GeneralLedgerPendingEntry explicitEntry = new GeneralLedgerPendingEntry();
        SpringContext.getBean(GeneralLedgerPendingEntryService.class).populateExplicitGeneralLedgerPendingEntry(
                this, postable, sequenceHelper, explicitEntry);
        customizeExplicitGeneralLedgerPendingEntry(postable, explicitEntry);
        explicitEntry.refreshNonUpdateableReferences();
        addPendingEntry(explicitEntry);
        return true;
    }

    public String getOffCampusAddress() {
        return offCampusAddress;
    }

    public String getOffCampusCityName() {
        return offCampusCityName;
    }

    public State getOffCampusState() {
        return offCampusState;
    }

    public String getOffCampusStateCode() {
        return offCampusStateCode;
    }

    public String getOffCampusZipCode() {
        return offCampusZipCode;
    }

    public PostalCode getPostalZipCode() {
        return postalZipCode;
    }

    public Country getOffCampusCountry() {
        return offCampusCountry;
    }

    public String getOrganizationInventoryName() {
        return organizationInventoryName;
    }

    public Account getOrganizationOwnerAccount() {
        return organizationOwnerAccount;
    }

    public Account getOldOrganizationOwnerAccount() {
        return oldOrganizationOwnerAccount;
    }

    public String getOrganizationOwnerAccountNumber() {
        return organizationOwnerAccountNumber;
    }

    public Chart getOrganizationOwnerChartOfAccounts() {
        return organizationOwnerChartOfAccounts;
    }

    public String getOrganizationOwnerChartOfAccountsCode() {
        return organizationOwnerChartOfAccountsCode;
    }

    public String getOrganizationTagNumber() {
        return organizationTagNumber;
    }

    public String getOrganizationText() {
        return organizationText;
    }

    public String getRepresentativeUniversalIdentifier() {
        return representativeUniversalIdentifier;
    }

    public TransferOfFundsDocument getTransferOfFundsFinancialDocument() {
        if (StringUtils.isNotBlank(getTransferOfFundsFinancialDocumentNumber())) {
            Map<String, String> primaryKeys = new HashMap<>();
            primaryKeys.put(KFSPropertyConstants.DOCUMENT_NUMBER, getTransferOfFundsFinancialDocumentNumber());
            TransferOfFundsDocument obj = SpringContext.getBean(BusinessObjectService.class).findByPrimaryKey(
                    TransferOfFundsDocument.class, primaryKeys);
            if (ObjectUtils.isNotNull(obj)) {
                return obj;
            }
        }
        return null;
    }

    public String getTransferOfFundsFinancialDocumentNumber() {
        return transferOfFundsFinancialDocumentNumber;
    }

    public Long getCapitalAssetNumber() {
        return capitalAssetNumber;
    }

    public void setCapitalAssetNumber(Long capitalAssetNumber) {
        this.capitalAssetNumber = capitalAssetNumber;
    }

    public void postProcessSave(KualiDocumentEvent event) {
        super.postProcessSave(event);

        // don't lock until they route
        if (!(event instanceof SaveDocumentEvent)) {
            List<Long> capitalAssetNumbers = new ArrayList<>();
            if (this.getCapitalAssetNumber() != null) {
                capitalAssetNumbers.add(this.getCapitalAssetNumber());
            }

            if (!this.getCapitalAssetManagementModuleService().storeAssetLocks(capitalAssetNumbers,
                    this.getDocumentNumber(), CamsConstants.DocumentTypeName.ASSET_TRANSFER, null)) {
                throw new ValidationException("Asset " + capitalAssetNumbers.toString() +
                        " is being locked by other documents.");
            }
        }
    }

    protected CapitalAssetManagementModuleService getCapitalAssetManagementModuleService() {
        return SpringContext.getBean(CapitalAssetManagementModuleService.class);
    }

    @Override
    public void doRouteStatusChange(DocumentRouteStatusChange statusChangeEvent) {
        super.doRouteStatusChange(statusChangeEvent);

        WorkflowDocument workflowDocument = getDocumentHeader().getWorkflowDocument();
        if (workflowDocument.isProcessed()) {
            SpringContext.getBean(AssetTransferService.class).saveApprovedChanges(this);
        }

        // Remove asset lock when doc status change. We don't include isFinal since document always go to 'processed'
        // first.
        if (workflowDocument.isCanceled() || workflowDocument.isDisapproved() || workflowDocument.isProcessed()) {
            getCapitalAssetManagementModuleService().deleteAssetLocks(this.getDocumentNumber(), null);
        }
    }

    public boolean isDebit(GeneralLedgerPendingEntrySourceDetail postable) {
        AssetGlpeSourceDetail srcDetail = (AssetGlpeSourceDetail) postable;
        boolean isDebit = false;
        // For source account, debit line when capitalization amount is negative or accumulated depreciation is
        // positive or offset amount is positive
        if (srcDetail.isSource()) {
            if (srcDetail.isCapitalization() && srcDetail.getAmount().isNegative()
                || srcDetail.isAccumulatedDepreciation() && srcDetail.getAmount().isPositive()
                || srcDetail.isCapitalizationOffset() && srcDetail.getAmount().isPositive()) {
                isDebit = true;
            }
        }
        // For target account, debit line when capitalization is positive or accumulated depreciation is negative or
        // offset amount is negative
        if (!srcDetail.isSource()) {
            if (srcDetail.isCapitalization() && srcDetail.getAmount().isPositive()
                || srcDetail.isAccumulatedDepreciation() && srcDetail.getAmount().isNegative()
                || srcDetail.isCapitalizationOffset() && srcDetail.getAmount().isNegative()) {
                isDebit = true;
            }
        }
        return isDebit;

    }

    public boolean isInterdepartmentalSalesIndicator() {
        return interdepartmentalSalesIndicator;
    }

    public void setAsset(Asset asset) {
        this.asset = asset;
    }

    /**
     * @deprecated
     */
    public void setBuilding(Building building) {
        this.building = building;
    }

    public void setBuildingCode(String buildingCode) {
        this.buildingCode = buildingCode;
    }

    /**
     * @deprecated
     */
    public void setBuildingRoom(Room buildingRoom) {
        this.buildingRoom = buildingRoom;
    }

    public void setBuildingRoomNumber(String buildingRoomNumber) {
        this.buildingRoomNumber = buildingRoomNumber;
    }

    public void setBuildingSubRoomNumber(String buildingSubRoomNumber) {
        this.buildingSubRoomNumber = buildingSubRoomNumber;
    }

    /**
     * @deprecated
     */
    public void setCampus(Campus campus) {
        this.campus = campus;
    }

    public void setCampusCode(String campusCode) {
        this.campusCode = campusCode;
    }

    public void setDocumentNumber(String documentNumber) {
        this.documentNumber = documentNumber;
    }

    public void setGeneralLedgerPostables(List<AssetGlpeSourceDetail> generalLedgerPostables) {
        this.sourceAssetGlpeSourceDetails = generalLedgerPostables;
    }

    public void setInterdepartmentalSalesIndicator(boolean interdepartmentalSalesIndicator) {
        this.interdepartmentalSalesIndicator = interdepartmentalSalesIndicator;
    }

    public void setOffCampusAddress(String offCampusAddress) {
        this.offCampusAddress = offCampusAddress;
    }

    public void setOffCampusCityName(String offCampusCityName) {
        this.offCampusCityName = offCampusCityName;
    }

    /**
     * @deprecated
     */
    public void setOffCampusState(State offCampusState) {
        this.offCampusState = offCampusState;
    }

    public void setOffCampusStateCode(String offCampusStateCode) {
        this.offCampusStateCode = offCampusStateCode;
    }

    public void setOffCampusZipCode(String offCampusZipCode) {
        this.offCampusZipCode = offCampusZipCode;
    }

    public void setPostalZipCode(PostalCode postalZipCode) {
        this.postalZipCode = postalZipCode;
    }

    /**
     * @deprecated
     */
    public void setOffCampusCountry(Country offCampusCountry) {
        this.offCampusCountry = offCampusCountry;
    }

    public void setOrganizationInventoryName(String organizationInventoryName) {
        this.organizationInventoryName = organizationInventoryName;
    }

    /**
     * @deprecated
     */
    public void setOrganizationOwnerAccount(Account organizationOwnerAccount) {
        this.organizationOwnerAccount = organizationOwnerAccount;
    }

    /**
     * @deprecated
     */
    public void setOldOrganizationOwnerAccount(Account oldOrganizationOwnerAccount) {
        this.oldOrganizationOwnerAccount = oldOrganizationOwnerAccount;
    }

    public void setOrganizationOwnerAccountNumber(String organizationOwnerAccountNumber) {
        this.organizationOwnerAccountNumber = organizationOwnerAccountNumber;

        // if accounts can't cross charts, set chart code whenever account number is set
        AccountService accountService = SpringContext.getBean(AccountService.class);
        if (!accountService.accountsCanCrossCharts()) {
            Account account = accountService.getUniqueAccountForAccountNumber(organizationOwnerAccountNumber);
            if (ObjectUtils.isNotNull(account)) {
                setOrganizationOwnerChartOfAccountsCode(account.getChartOfAccountsCode());
            }
        }
    }

    /**
     * @deprecated
     */
    public void setOrganizationOwnerChartOfAccounts(Chart organizationOwnerChartOfAccounts) {
        this.organizationOwnerChartOfAccounts = organizationOwnerChartOfAccounts;
    }

    public void setOrganizationOwnerChartOfAccountsCode(String organizationOwnerChartOfAccountsCode) {
        this.organizationOwnerChartOfAccountsCode = organizationOwnerChartOfAccountsCode;
    }

    public void setOrganizationTagNumber(String organizationTagNumber) {
        this.organizationTagNumber = organizationTagNumber;
    }

    public void setOrganizationText(String organizationText) {
        this.organizationText = organizationText;
    }

    public void setRepresentativeUniversalIdentifier(String representativeUniversalIdentifier) {
        this.representativeUniversalIdentifier = representativeUniversalIdentifier;
    }

    public void setTransferOfFundsFinancialDocumentNumber(String transferOfFundsFinancialDocumentNumber) {
        this.transferOfFundsFinancialDocumentNumber = transferOfFundsFinancialDocumentNumber;
    }

    public List<GeneralLedgerPendingEntrySourceDetail> getGeneralLedgerPendingEntrySourceDetails() {
        List<GeneralLedgerPendingEntrySourceDetail> generalLedgerPostables = new ArrayList<>();
        generalLedgerPostables.addAll(this.sourceAssetGlpeSourceDetails);
        generalLedgerPostables.addAll(this.targetAssetGlpeSourceDetails);
        return generalLedgerPostables;
    }

    public List<AssetGlpeSourceDetail> getTargetAssetGlpeSourceDetails() {
        if (this.targetAssetGlpeSourceDetails == null) {
            this.targetAssetGlpeSourceDetails = new ArrayList<>();
        }
        return this.targetAssetGlpeSourceDetails;
    }

    public void setTargetAssetGlpeSourceDetails(List<AssetGlpeSourceDetail> targetAssetGlpeSourceDetails) {
        this.targetAssetGlpeSourceDetails = targetAssetGlpeSourceDetails;
    }

    public void setSourceAssetGlpeSourceDetails(List<AssetGlpeSourceDetail> sourceAssetGlpeSourceDetails) {
        this.sourceAssetGlpeSourceDetails = sourceAssetGlpeSourceDetails;
    }

    public List<GeneralLedgerPendingEntrySourceDetail> getGeneralLedgerPostables() {
        List<GeneralLedgerPendingEntrySourceDetail> generalLedgerPostables = new ArrayList<>();
        generalLedgerPostables.addAll(this.sourceAssetGlpeSourceDetails);
        generalLedgerPostables.addAll(this.targetAssetGlpeSourceDetails);
        return generalLedgerPostables;
    }

    public String getOffCampusCountryCode() {
        return offCampusCountryCode;
    }

    public void setOffCampusCountryCode(String offCampusCountryCode) {
        this.offCampusCountryCode = offCampusCountryCode;
    }

    public String getOffCampusName() {
        return offCampusName;
    }

    public void setOffCampusName(String offCampusName) {
        this.offCampusName = offCampusName;
    }

    public String getOldOrganizationOwnerAccountNumber() {
        return oldOrganizationOwnerAccountNumber;
    }

    public void setOldOrganizationOwnerAccountNumber(String oldOrganizationOwnerAccountNumber) {
        this.oldOrganizationOwnerAccountNumber = oldOrganizationOwnerAccountNumber;
    }

    public String getOldOrganizationOwnerChartOfAccountsCode() {
        return oldOrganizationOwnerChartOfAccountsCode;
    }

    public void setOldOrganizationOwnerChartOfAccountsCode(String oldOrganizationOwnerChartOfAccountsCode) {
        this.oldOrganizationOwnerChartOfAccountsCode = oldOrganizationOwnerChartOfAccountsCode;
    }

    public void clearGlPostables() {
        getGeneralLedgerPendingEntries().clear();
        getSourceAssetGlpeSourceDetails().clear();
        getTargetAssetGlpeSourceDetails().clear();
    }

    public String getHiddenFieldForError() {
        return hiddenFieldForError;
    }

    public void setHiddenFieldForError(String hiddenFieldForError) {
        this.hiddenFieldForError = hiddenFieldForError;
    }

    @Override
    public void prepareForSave(KualiDocumentEvent event) {
        super.prepareForSave(event);
        String accountingPeriodCompositeString = getAccountingPeriodCompositeString();
        setPostingYear(new Integer(StringUtils.right(accountingPeriodCompositeString, 4)));
        setPostingPeriodCode(StringUtils.left(accountingPeriodCompositeString, 2));
    }

    @Override
    public boolean answerSplitNodeQuestion(String nodeName) throws UnsupportedOperationException {
        if (CamsConstants.RouteLevelNames.ORGANIZATION_INACTIVE.equals(nodeName)) {
            return isRequiresOrganizationInactiveRouteNode(asset);
        }

        return super.answerSplitNodeQuestion(nodeName);
    }

    /**
     * @return if the organization inactive route node needs to be stopped at
     */
    protected static boolean isRequiresOrganizationInactiveRouteNode(Asset asset) {
        return !asset.getOrganizationOwnerAccount().getOrganization().isActive();
    }

}
