/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2021 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cam.document.validation.impl;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.datadictionary.legacy.BusinessObjectDictionaryService;
import org.kuali.kfs.kew.api.WorkflowDocument;
import org.kuali.kfs.kim.api.identity.Person;
import org.kuali.kfs.kim.api.identity.PersonService;
import org.kuali.kfs.kns.document.MaintenanceDocument;
import org.kuali.kfs.kns.maintenance.rules.MaintenanceDocumentRuleBase;
import org.kuali.kfs.krad.bo.PersistableBusinessObject;
import org.kuali.kfs.krad.maintenance.MaintenanceDocumentAuthorizer;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.module.cam.CamsKeyConstants;
import org.kuali.kfs.module.cam.CamsPropertyConstants;
import org.kuali.kfs.module.cam.businessobject.Asset;
import org.kuali.kfs.module.cam.businessobject.Pretag;
import org.kuali.kfs.module.cam.businessobject.PretagDetail;
import org.kuali.kfs.module.cam.document.service.PurApInfoService;
import org.kuali.kfs.module.purap.businessobject.PurApItem;
import org.kuali.kfs.module.purap.document.PurchaseOrderDocument;
import org.kuali.kfs.sys.KFSKeyConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.businessobject.Building;
import org.kuali.kfs.sys.businessobject.Room;
import org.kuali.kfs.sys.context.SpringContext;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * This class represents the business rules for the maintenance of
 * {@link org.kuali.kfs.coa.businessobject.AccountGlobal} business objects
 */
public class PretagRule extends MaintenanceDocumentRuleBase {

    protected PersistableBusinessObject bo;
    protected Pretag newPretag;

    public PretagRule() {
        super();
    }

    /**
     * This method sets the convenience objects like newPretag and oldPretag, so you have short and easy handles to
     * the new and old objects contained in the maintenance document. It also calls the BusinessObjectBase.refresh(),
     * which will attempt to load all sub-objects from the DB by their primary keys, if available.
     */
    @Override
    public void setupConvenienceObjects() {
        // setup Pretag convenience objects, make sure all possible detail lines are populated
        newPretag = (Pretag) super.getNewBo();
        for (PretagDetail dtl : newPretag.getPretagDetails()) {
            dtl.refreshNonUpdateableReferences();
        }
    }

    /**
     * Does not fail on rules failure
     */
    @Override
    protected boolean processCustomSaveDocumentBusinessRules(MaintenanceDocument document) {
        boolean success = processPretagValidation();
        return success & super.processCustomSaveDocumentBusinessRules(document);
    }

    @Override
    protected boolean processCustomRouteDocumentBusinessRules(MaintenanceDocument document) {
        MaintenanceDocumentAuthorizer documentAuthorizer =
                (MaintenanceDocumentAuthorizer) getDocumentHelperService().getDocumentAuthorizer(document);

        WorkflowDocument workflowDocument = document.getDocumentHeader().getWorkflowDocument();
        boolean success = true;
        if (workflowDocument.isInitiated() || workflowDocument.isSaved()) {
            success = documentAuthorizer.canCreateOrMaintain(document, GlobalVariables.getUserSession().getPerson());
            if (!success) {
                putFieldError(CamsPropertyConstants.Pretag.CHART_OF_ACCOUNTS_CODE,
                        CamsKeyConstants.CHART_ORG_DISALLOWED_BY_CURRENT_USER);
            }
        }

        return success & super.processCustomRouteDocumentBusinessRules(document);
    }

    /**
     * Validates Pretag and its PretagDetail.
     *
     * @return boolean false or true
     */
    public boolean processPretagValidation() {
        setupConvenienceObjects();
        boolean success = checkPurchaseOrderItemExists();
        success &= checkAssetRepresentativePrincipalNameExists();

        if (newPretag.isActive()) {
            success &= checkTotalDetailCount(newPretag, false);
            success &= isAllCampusBuildingRoomValid(newPretag.getPretagDetails());
        } else {
            deactivePretagDetails(newPretag);
        }

        return success;
    }

    /**
     * validate the asset representative principal name.
     *
     * @return boolean false or true
     */
    protected boolean checkAssetRepresentativePrincipalNameExists() {
        boolean valid = true;
        if (StringUtils.isNotBlank(newPretag.getPersonUniversal().getPrincipalName())) {
            PersonService personService = SpringContext.getBean(PersonService.class);
            Person person = personService.getPersonByPrincipalName(newPretag.getPersonUniversal().getPrincipalName());
            if (person != null) {
                newPretag.setPersonUniversal(person);
                newPretag.setRepresentativeUniversalIdentifier(person.getPrincipalId());
            } else {
                putFieldError(CamsPropertyConstants.Pretag.REPRESENTATIVE_ID,
                        CamsKeyConstants.PreTag.ERROR_PRE_TAG_INVALID_REPRESENTATIVE_ID,
                        newPretag.getPersonUniversal().getPrincipalName());
                newPretag.setPersonUniversal(null);
                newPretag.setRepresentativeUniversalIdentifier(null);
                valid = false;
            }
        }
        return valid;
    }

    /**
     * validate the purchase order item existence in PurAp.
     *
     * @return boolean false or true
     */
    protected boolean checkPurchaseOrderItemExists() {
        boolean valid = true;
        if (StringUtils.isNotBlank(newPretag.getPurchaseOrderNumber()) && newPretag.getItemLineNumber() != null) {
            PurchaseOrderDocument purchaseOrderDoc = getPurApInfoService()
                    .getCurrentDocumentForPurchaseOrderIdentifier(
                            Integer.valueOf(newPretag.getPurchaseOrderNumber()));
            if (purchaseOrderDoc == null) {
                String label = SpringContext.getBean(BusinessObjectDictionaryService.class)
                        .getBusinessObjectEntry(Pretag.class.getName()).getAttributeDefinition(
                                CamsPropertyConstants.Pretag.PURCHASE_ORDER_NUMBER).getLabel();
                putFieldError(CamsPropertyConstants.Pretag.PURCHASE_ORDER_NUMBER, KFSKeyConstants.ERROR_EXISTENCE,
                        label);
                valid = false;
            } else if (getItemByLineNumber(purchaseOrderDoc, newPretag.getItemLineNumber()) == null) {
                String label = SpringContext.getBean(BusinessObjectDictionaryService.class)
                        .getBusinessObjectEntry(Pretag.class.getName()).getAttributeDefinition(
                                CamsPropertyConstants.Pretag.ITEM_LINE_NUMBER).getLabel();
                putFieldError(CamsPropertyConstants.Pretag.ITEM_LINE_NUMBER, KFSKeyConstants.ERROR_EXISTENCE, label);
                valid = false;
            }
        }
        return valid;
    }

    /**
     * Get PurchaseOrderItem by given item line number
     *
     * @param purchaseOrderDocument
     * @param lineNumber
     * @return
     */
    protected PurApItem getItemByLineNumber(PurchaseOrderDocument purchaseOrderDocument, int lineNumber) {
        List items = purchaseOrderDocument.getItems();
        for (Object entry : items) {
            PurApItem item = (PurApItem) entry;
            if (item.getItemLineNumber() != null && item.getItemLineNumber() == lineNumber) {
                return item;
            }
        }
        return null;
    }

    /**
     * This method loops through the list of {@link PretagDetail}s and passes them off to
     * isAllCampusBuildingRoomValid for further rule analysis
     *
     * @param details
     * @return true if the collection of {@link PretagDetail}s passes the sub-rules
     */
    public boolean isAllCampusBuildingRoomValid(List<PretagDetail> details) {
        boolean success = true;

        // check if there are any pretag details
        if (details.size() != 0) {

            // check each CampusBuildingRoom
            int index = 0;
            for (PretagDetail dtl : details) {
                String errorPath = MAINTAINABLE_ERROR_PREFIX + "pretagDetails[" + index + "]";
                GlobalVariables.getMessageMap().addToErrorPath(errorPath);
                success &= isCampusBuildingRoomValid(dtl);
                GlobalVariables.getMessageMap().removeFromErrorPath(errorPath);
                index++;
            }
        }

        return success;
    }

    /**
     * This method calls isCampusTagNumberValid whenever a new {@link PretagDetail} is added to Pretag
     */
    @Override
    public boolean processCustomAddCollectionLineBusinessRules(MaintenanceDocument document, String collectionName,
            PersistableBusinessObject bo) {
        setupConvenienceObjects();

        Pretag pretag = (Pretag) document.getNewMaintainableObject().getBusinessObject();
        PretagDetail detail = (PretagDetail) bo;

        boolean success = true;
        if (detail.isActive()) {

            detail.setPurchaseOrderNumber(pretag.getPurchaseOrderNumber());
            detail.setItemLineNumber(pretag.getItemLineNumber());

            success = checkDuplicateTagNumber(pretag, detail.getCampusTagNumber());
            success &= checkTotalDetailCount(pretag, true);
            success &= isCampusTagNumberValid(detail);
            success &= isCampusBuildingRoomValid(detail);
        }

        return success;
    }

    /**
     * This method check to see if duplicate tag exists
     *
     * @return boolean indicating if validation succeeded
     */
    protected boolean checkDuplicateTagNumber(Pretag pretag, String tagNumber) {
        boolean success = true;

        for (PretagDetail dtl : pretag.getPretagDetails()) {
            if (dtl.getCampusTagNumber().equals(tagNumber) && dtl.isActive()) {
                GlobalVariables.getMessageMap().putError(CamsPropertyConstants.Pretag.CAMPUS_TAG_NUMBER,
                        CamsKeyConstants.ERROR_TAG_NUMBER_DUPLICATE, tagNumber);
                success = false;
            }
        }

        return success;
    }

    /**
     * This method ensures that total {@link PretagDetail} tag details does not excees in quantity invoiced
     *
     * @param pretag
     * @param addLine
     * @return true if the detail tag doesn't exist in Asset
     */
    public boolean checkTotalDetailCount(Pretag pretag, boolean addLine) {
        boolean success = true;
        if (pretag.getQuantityInvoiced() != null) {
            int totalActiveDetails = getActiveDetailsCount(pretag, addLine);
            KualiDecimal totalNumerOfDetails = new KualiDecimal(totalActiveDetails);

            if (pretag.getQuantityInvoiced().compareTo(totalNumerOfDetails) < 0) {
                GlobalVariables.getMessageMap().putError(CamsPropertyConstants.Pretag.CAMPUS_TAG_NUMBER,
                        CamsKeyConstants.PreTag.ERROR_PRE_TAG_DETAIL_EXCESS,
                        pretag.getQuantityInvoiced().toString() + "" +
                                " Total number of detail lines " + totalNumerOfDetails.toString());
                success = false;
            }
        }

        return success;
    }

    /**
     * This method reply that total active detail in {@link Pretag}
     *
     * @param pretag and newDetailLine
     * @return total number of active pretagDetails
     */
    public int getActiveDetailsCount(Pretag pretag, boolean newDetailLine) {
        Collection<PretagDetail> pretagDetails = pretag.getPretagDetails();
        if (newDetailLine) {
            return countActive(pretagDetails) + 1;
        } else {
            return countActive(pretagDetails);
        }
    }

    /**
     * This method ensures that each {@link PretagDetail} tag number does not exist in Asset table
     *
     * @param dtl
     * @return true if the detail tag doesn't exist in Asset
     */
    public boolean isCampusTagNumberValid(PretagDetail dtl) {
        boolean success = true;

        if (dtl.getCampusTagNumber() != null && dtl.isActive()
            && !dtl.getCampusTagNumber().equalsIgnoreCase("N")) {
            Map<String, String> tagMap = new HashMap<>();
            tagMap.put(CamsPropertyConstants.Pretag.CAMPUS_TAG_NUMBER, dtl.getCampusTagNumber());
            int matchDetailCount = getMatchDetailCount(tagMap);
            if (getBoService().countMatching(Asset.class, tagMap) != 0 || matchDetailCount > 0) {
                GlobalVariables.getMessageMap().putError(CamsPropertyConstants.Pretag.CAMPUS_TAG_NUMBER,
                        CamsKeyConstants.PreTag.ERROR_PRE_TAG_NUMBER, dtl.getCampusTagNumber());
                success = false;
            }
        }

        return success;
    }

    /**
     * This method ensures that each {@link PretagDetail} buildingCode and buildingRoomNumber does exist in building
     * and room tables
     *
     * @param dtl
     * @return true if the detail buildingCode and buildingRoomNumber does exist in building and room
     */
    public boolean isCampusBuildingRoomValid(PretagDetail dtl) {
        int originalErrorCount = GlobalVariables.getMessageMap().getErrorCount();
        getDictionaryValidationService().validateBusinessObject(dtl);

        if (StringUtils.isNotBlank(dtl.getCampusCode()) && StringUtils.isNotBlank(dtl.getBuildingCode())) {
            Map<String, String> preTagMap = new HashMap<>();
            preTagMap.put(KFSPropertyConstants.CAMPUS_CODE, dtl.getCampusCode());
            preTagMap.put(KFSPropertyConstants.BUILDING_CODE, dtl.getBuildingCode());

            bo = SpringContext.getBean(BusinessObjectService.class).findByPrimaryKey(Building.class, preTagMap);
            if (bo == null) {
                GlobalVariables.getMessageMap().putError(KFSPropertyConstants.BUILDING_CODE,
                        CamsKeyConstants.ERROR_INVALID_BUILDING_CODE, dtl.getCampusCode(), dtl.getBuildingCode());
            }

            if (StringUtils.isNotBlank(dtl.getBuildingRoomNumber())) {
                preTagMap.put(KFSPropertyConstants.BUILDING_ROOM_NUMBER, dtl.getBuildingRoomNumber());
                bo = SpringContext.getBean(BusinessObjectService.class).findByPrimaryKey(Room.class, preTagMap);
                if (bo == null) {
                    GlobalVariables.getMessageMap().putError(KFSPropertyConstants.BUILDING_ROOM_NUMBER,
                            CamsKeyConstants.ERROR_INVALID_ROOM_NUMBER, dtl.getCampusCode(), dtl.getBuildingCode(),
                            dtl.getBuildingRoomNumber());
                }
            }
        }
        return GlobalVariables.getMessageMap().getErrorCount() == originalErrorCount;
    }

    /**
     * @param tagMap
     * @return number of active pretagDetail with same campusTagNumber
     */
    public int getMatchDetailCount(Map<String, String> tagMap) {
        Collection<PretagDetail> pretagDetails = SpringContext.getBean(BusinessObjectService.class).findMatching(
                PretagDetail.class, tagMap);
        return countActive(pretagDetails);
    }

    /**
     * This method ensures that count {@link PretagDetail} active detail lines
     *
     * @param pretagDetails
     * @return active pretagDetail count
     */
    public int countActive(Collection<PretagDetail> pretagDetails) {
        int activeCount = 0;

        for (PretagDetail dtl : pretagDetails) {
            if (dtl.isActive()) {
                activeCount++;
            }
        }

        return activeCount;
    }

    /**
     * This method ensures that all {@link Pretag} detail lines deactivated
     *
     * @param pretag
     */
    public void deactivePretagDetails(Pretag pretag) {
        for (PretagDetail dtl : pretag.getPretagDetails()) {
            dtl.setActive(false);
        }
    }

    protected PurApInfoService getPurApInfoService() {
        return SpringContext.getBean(PurApInfoService.class);
    }
}
