/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cg.businessobject.inquiry;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.integration.ar.AccountsReceivableMilestoneSchedule;
import org.kuali.kfs.integration.ar.AccountsReceivableModuleBillingService;
import org.kuali.kfs.integration.ar.AccountsReceivablePredeterminedBillingSchedule;
import org.kuali.kfs.integration.cg.ContractsAndGrantsBillingAwardAccount;
import org.kuali.kfs.kns.datadictionary.InquirySectionDefinition;
import org.kuali.kfs.kns.inquiry.InquiryRestrictions;
import org.kuali.kfs.kns.inquiry.KualiInquirableImpl;
import org.kuali.kfs.kns.lookup.HtmlData;
import org.kuali.kfs.kns.lookup.HtmlData.AnchorHtmlData;
import org.kuali.kfs.kns.service.KNSServiceLocator;
import org.kuali.kfs.kns.util.FieldUtils;
import org.kuali.kfs.kns.web.ui.Field;
import org.kuali.kfs.kns.web.ui.FieldBridge;
import org.kuali.kfs.kns.web.ui.Row;
import org.kuali.kfs.kns.web.ui.Section;
import org.kuali.kfs.kns.web.ui.SectionBridge;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.KRADConstants;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.krad.util.UrlFactory;
import org.kuali.kfs.module.cg.CGConstants;
import org.kuali.kfs.module.cg.CGPropertyConstants;
import org.kuali.kfs.module.cg.businessobject.Award;
import org.kuali.kfs.module.cg.service.ContractsAndGrantsBillingService;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.krad.bo.BusinessObject;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * Used for wiring up {@link Award} for inquiries.
 */
public class AwardInquirableImpl extends KualiInquirableImpl {
    private static final Logger LOG = LogManager.getLogger();

    /**
     * Helper method to build an inquiry URLs for MilestoneSchedule or PredeterminedBillingSchedule links.
     */
    @Override
    public HtmlData getInquiryUrl(BusinessObject businessObject, String attributeName, boolean forceInquiry) {

        if (StringUtils.equals(CGPropertyConstants.AwardFields.MILESTONE_SCHEDULE_INQUIRY_TITLE, attributeName)) {
            return buildInquiryUrl(businessObject, AccountsReceivableMilestoneSchedule.class);
        } else if (StringUtils.equals(CGPropertyConstants.AwardFields.PREDETERMINED_BILLING_SCHEDULE_INQUIRY_TITLE, attributeName)) {
            return buildInquiryUrl(businessObject, AccountsReceivablePredeterminedBillingSchedule.class);
        }

        return super.getInquiryUrl(businessObject, attributeName, forceInquiry);
    }

    /**
     * Build the inquiry URL for the business object (Milestone Schedule or Predetermined Billing Schedule) passed in
     * as a parameter. In this case this is a link to the lookup instead of an inquiry.
     *
     * @param businessObject Schedule to build inquiry URL for
     * @param businessObjectClass Class of the schedule to build inquiry URL for
     * @return inquiry href
     */
    private AnchorHtmlData buildInquiryUrl(BusinessObject businessObject, Class businessObjectClass) {
        AnchorHtmlData inquiryHref = new AnchorHtmlData(KFSConstants.EMPTY_STRING, KFSConstants.EMPTY_STRING);

        String baseUrl = KRADConstants.LOOKUP_ACTION;
        Map<String, String> parameters = new HashMap<>();
        parameters.put(KFSConstants.DISPATCH_REQUEST_PARAMETER, KFSConstants.SEARCH_METHOD);
        // Without this, the mode is set to standalone because this is an inquiry link without a corresponding modal,
        // and as a result the dashboard is hidden.
        parameters.put(KFSConstants.MODE, KRADConstants.PARAM_MAINTENANCE_VIEW_MODE_LOOKUP);
        parameters.put(KFSConstants.BUSINESS_OBJECT_CLASS_ATTRIBUTE, businessObjectClass.getName());
        parameters.put(KFSPropertyConstants.PROPOSAL_NUMBER, ObjectUtils.getPropertyValue(businessObject,
            KFSPropertyConstants.PROPOSAL_NUMBER).toString());

        inquiryHref.setHref(UrlFactory.parameterizeUrl(baseUrl, parameters));

        return inquiryHref;
    }

    /**
     * Only show the Schedule link if CGB is enabled and for the appropriate Billing Frequency
     * (Milestone Schedule for Milestone billing, Predetermined Billing Schedule for PDBS billing, or none).
     *
     * <p>
     * KRAD Conversion: Inquirable performs conditional display/hiding of the sections on the inquiry
     * But all field/section definitions are in data dictionary for bo Asset.
     */
    @Override
    public List<Section> getSections(BusinessObject businessObject) {
        List<Section> sections = new ArrayList<>();
        if (getBusinessObjectClass() == null) {
            LOG.error("Business object class not set in inquirable.");
            throw new RuntimeException("Business object class not set in inquirable.");
        }

        InquiryRestrictions inquiryRestrictions = KNSServiceLocator.getBusinessObjectAuthorizationService()
            .getInquiryRestrictions(businessObject, GlobalVariables.getUserSession().getPerson());

        Collection<InquirySectionDefinition> inquirySections = getBusinessObjectDictionaryService().getInquirySections(
            getBusinessObjectClass());
        Collection<?> sectionIdsToIgnore = getSectionIdsToIgnore();
        for (InquirySectionDefinition inquirySection : inquirySections) {
            String sectionId = inquirySection.getId();
            if (!inquiryRestrictions.isHiddenSectionId(sectionId) && !sectionIdsToIgnore.contains(sectionId)) {
                Section section = SectionBridge.toSection(this, inquirySection, businessObject,
                    inquiryRestrictions);
                if (StringUtils.equals(sectionId, CGPropertyConstants.SectionId.AWARD_INVOICING_SECTION_ID)) {
                    for (Row row : section.getRows()) {
                        List<Field> updatedFields = row.getFields().stream()
                            .map(field -> determineScheduleInquiryField(field, (Award) businessObject))
                            .collect(Collectors.toList());
                        row.setFields(updatedFields);
                    }
                }
                sections.add(section);
            }
        }

        return sections;
    }

    private Field determineScheduleInquiryField(Field field, Award award) {
        if (field.getPropertyName().equalsIgnoreCase(
            CGPropertyConstants.AwardFields.SCHEDULE_INQUIRY_TITLE)) {

            for (ContractsAndGrantsBillingAwardAccount awardAccount : award.getActiveAwardAccounts()) {
                if (StringUtils.equals(award.getBillingFrequencyCode(),
                    CGConstants.MILESTONE_BILLING_SCHEDULE_CODE) &&
                    SpringContext.getBean(AccountsReceivableModuleBillingService.class)
                        .hasMilestoneSchedule(award.getProposalNumber(), awardAccount.getChartOfAccountsCode(),
                            awardAccount.getAccountNumber())) {

                    Field newField = FieldUtils.getPropertyField(Award.class,
                        CGPropertyConstants.AwardFields.MILESTONE_SCHEDULE_INQUIRY_TITLE, false);
                    FieldBridge.populateFieldFromBusinessObject(newField, award);

                    return newField;
                } else if (StringUtils.equals(award.getBillingFrequencyCode(),
                    CGConstants.PREDETERMINED_BILLING_SCHEDULE_CODE) &&
                    SpringContext.getBean(AccountsReceivableModuleBillingService.class)
                        .hasPredeterminedBillingSchedule(award.getProposalNumber(),
                            awardAccount.getChartOfAccountsCode(), awardAccount.getAccountNumber())) {

                    Field newField = FieldUtils.getPropertyField(Award.class,
                        CGPropertyConstants.AwardFields.PREDETERMINED_BILLING_SCHEDULE_INQUIRY_TITLE, false);
                    FieldBridge.populateFieldFromBusinessObject(newField, award);

                    return newField;
                }
            }

            return null;
        }

        return field;
    }

    /**
     * If the Contracts & Grants Billing (CGB) enhancement is disabled, we don't want to
     * process sections only related to CGB.
     *
     * @return Collection of section ids to ignore
     */
    private Collection<?> getSectionIdsToIgnore() {
        if (!SpringContext.getBean(AccountsReceivableModuleBillingService.class).isContractsGrantsBillingEnhancementActive()) {
            return SpringContext.getBean(ContractsAndGrantsBillingService.class).getAwardContractsGrantsBillingSectionIds();
        } else {
            return CollectionUtils.EMPTY_COLLECTION;
        }
    }

}

