/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2021 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cg.service.impl;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.coa.businessobject.Account;
import org.kuali.kfs.coa.COAConstants;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.integration.cg.ContractsAndGrantsModuleService;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.cg.CGConstants;
import org.kuali.kfs.module.cg.businessobject.AwardAccount;
import org.kuali.kfs.module.cg.service.AgencyService;
import org.kuali.kfs.module.cg.service.CfdaService;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.kim.api.identity.Person;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * This Class provides implementation to the services required for inter module communication.
 */
public class ContractsAndGrantsModuleServiceImpl implements ContractsAndGrantsModuleService {

    protected ParameterService parameterService;
    protected AgencyService agencyService;
    protected CfdaService cfdaService;
    protected BusinessObjectService businessObjectService;

    @Override
    public Person getProjectDirectorForAccount(String chartOfAccountsCode, String accountNumber) {
        Map<String, Object> awardAccountMap = new HashMap<>();
        awardAccountMap.put(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, chartOfAccountsCode);
        awardAccountMap.put(KFSPropertyConstants.ACCOUNT_NUMBER, accountNumber);

        Collection<AwardAccount> proposals = getBusinessObjectService().findMatchingOrderBy(AwardAccount.class,
                awardAccountMap, KFSPropertyConstants.PROPOSAL_NUMBER, false);
        if (proposals != null && !proposals.isEmpty()) {
            AwardAccount proposalWithMaxProposalNumber = proposals.iterator().next();

            return proposalWithMaxProposalNumber.getProjectDirector();
        }

        return null;
    }

    @Override
    public Person getProjectDirectorForAccount(Account account) {
        if (ObjectUtils.isNotNull(account)) {
            account.refreshNonUpdateableReferences();
            String chartOfAccountsCode = account.getChartOfAccountsCode();
            String accountNumber = account.getAccountNumber();
            return this.getProjectDirectorForAccount(chartOfAccountsCode, accountNumber);
        }
        return null;
    }

    @Override
    public boolean isAwardedByFederalAgency(String chartOfAccountsCode, String accountNumber,
            Collection<String> federalAgencyTypeCodes) {
        AwardAccount primaryAward = getPrimaryAwardAccount(chartOfAccountsCode, accountNumber);
        if (primaryAward == null) {
            return false;
        }

        String agencyTypeCode = primaryAward.getAward().getAgency().getAgencyTypeCode();
        return federalAgencyTypeCodes.contains(agencyTypeCode)
                || primaryAward.getAward().getFederalPassThroughIndicator();

    }

    /**
     * get the primary award account for the given account
     *
     * @param chartOfAccountsCode the chart of accounts code for the given account
     * @return the primary award account for the given account
     */
    protected AwardAccount getPrimaryAwardAccount(String chartOfAccountsCode, String accountNumber) {
        AwardAccount primaryAwardAccount = null;
        long highestProposalNumber = 0;

        Map<String, Object> accountKeyValues = new HashMap<>();
        accountKeyValues.put(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, chartOfAccountsCode);
        accountKeyValues.put(KFSPropertyConstants.ACCOUNT_NUMBER, accountNumber);

        for (Object awardAccountAsObject : getBusinessObjectService().findMatching(AwardAccount.class,
                accountKeyValues)) {
            AwardAccount awardAccount = (AwardAccount) awardAccountAsObject;
            Long proposalNumber = Long.parseLong(awardAccount.getProposalNumber());

            if (proposalNumber >= highestProposalNumber) {
                highestProposalNumber = proposalNumber;
                primaryAwardAccount = awardAccount;
            }
        }

        return primaryAwardAccount;
    }

    @Override
    public List<Integer> getAllAccountReponsiblityIds() {
        int maxResponsibilityId = this.getMaxiumAccountResponsibilityId();

        List<Integer> contractsAndGrantsResponsiblityIds = new ArrayList<>();
        for (int id = 1; id <= maxResponsibilityId; id++) {
            contractsAndGrantsResponsiblityIds.add(id);
        }

        return contractsAndGrantsResponsiblityIds;
    }

    @Override
    public boolean hasValidAccountReponsiblityIdIfNotNull(Account account) {
        Integer accountResponsibilityId = account.getContractsAndGrantsAccountResponsibilityId();

        if (accountResponsibilityId == null) {
            return true;
        }

        return accountResponsibilityId >= 1 && accountResponsibilityId <= this.getMaxiumAccountResponsibilityId();
    }

    protected int getMaxiumAccountResponsibilityId() {
        String maxResponsibilityId = getParameterService().getParameterValueAsString(
                COAConstants.COA_NAMESPACE_CODE,
                CGConstants.Components.ACCOUNT_CMPNT,
                CGConstants.CG_RESPONSIBILITY_ID);
        return Integer.parseInt(maxResponsibilityId);
    }

    public ParameterService getParameterService() {
        return parameterService;
    }

    public AgencyService getAgencyService() {
        return agencyService;
    }

    public CfdaService getCfdaService() {
        return cfdaService;
    }

    public BusinessObjectService getBusinessObjectService() {
        return businessObjectService;
    }

    @Override
    public String getProposalNumberForAccountAndProjectDirector(String chartOfAccountsCode, String accountNumber,
            String projectDirectorId) {
        String proposalNumber = null;

        Map<String, Object> awardAccountMap = new HashMap<>();
        awardAccountMap.put(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, chartOfAccountsCode);
        awardAccountMap.put(KFSPropertyConstants.ACCOUNT_NUMBER, accountNumber);

        Collection<AwardAccount> proposals = getBusinessObjectService().findMatchingOrderBy(AwardAccount.class,
                awardAccountMap, KFSPropertyConstants.PROPOSAL_NUMBER, false);
        if (proposals != null && !proposals.isEmpty()) {
            AwardAccount proposalWithMaxProposalNumber = proposals.iterator().next();

            if (StringUtils.equalsIgnoreCase(proposalWithMaxProposalNumber.getProjectDirector().getPrincipalId(),
                    projectDirectorId)) {
                proposalNumber = proposalWithMaxProposalNumber.getProposalNumber();
            }
        }

        return proposalNumber;
    }

    public void setParameterService(ParameterService parameterService) {
        this.parameterService = parameterService;
    }

    public void setAgencyService(AgencyService agencyService) {
        this.agencyService = agencyService;
    }

    public void setCfdaService(CfdaService cfdaService) {
        this.cfdaService = cfdaService;
    }

    public void setBusinessObjectService(BusinessObjectService businessObjectService) {
        this.businessObjectService = businessObjectService;
    }
}
