/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2020 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.sys.datatools.liquirelational;

import liquibase.Liquibase;
import liquibase.database.Database;
import liquibase.database.DatabaseFactory;
import liquibase.database.jvm.JdbcConnection;
import liquibase.exception.DatabaseException;
import liquibase.exception.LiquibaseException;
import liquibase.resource.ClassLoaderResourceAccessor;
import liquibase.resource.ResourceAccessor;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.core.config.Configurator;
import org.apache.logging.log4j.core.config.DefaultConfiguration;
import org.springframework.context.support.ClassPathXmlApplicationContext;
import org.springframework.core.env.PropertySource;

import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class LiquiRelational {

    private static final Logger LOG = LogManager.getLogger();

    private static final Map<Integer, String> KFS_LIQUIBASE_MASTER_FILES = Stream.of(
            new AbstractMap.SimpleEntry<>(1, "org/kuali/kfs/db/liquibase-phase1-master.xml"),
            new AbstractMap.SimpleEntry<>(2, "org/kuali/kfs/db/liquibase-phase2-master.xml"),
            new AbstractMap.SimpleEntry<>(3, "org/kuali/kfs/db/liquibase-phase3-master.xml"),
            new AbstractMap.SimpleEntry<>(4, "org/kuali/kfs/db/liquibase-phase4-master.xml"),
            new AbstractMap.SimpleEntry<>(5, "org/kuali/kfs/db/liquibase-phase5-master.xml"))
            .collect(Collectors.toMap(AbstractMap.SimpleEntry::getKey, AbstractMap.SimpleEntry::getValue));

    private static final Map<Integer, String> RICE_LIQUIBASE_MASTER_FILES = Stream.of(
            new AbstractMap.SimpleEntry<>(1, "org/kuali/rice/db/liquibase-phase1-master.xml"),
            new AbstractMap.SimpleEntry<>(2, "org/kuali/rice/db/liquibase-phase2-master.xml"),
            new AbstractMap.SimpleEntry<>(3, "org/kuali/rice/db/liquibase-phase3-master.xml"),
            new AbstractMap.SimpleEntry<>(4, "org/kuali/rice/db/liquibase-phase4-master.xml"),
            new AbstractMap.SimpleEntry<>(5, "org/kuali/rice/db/liquibase-phase5-master.xml"))
            .collect(Collectors.toMap(AbstractMap.SimpleEntry::getKey, AbstractMap.SimpleEntry::getValue));

    private static final Map<String, Map<Integer, String>> LIQUIBASE_MASTER_FILES = Stream.of(
            new AbstractMap.SimpleEntry<>("dataSource", KFS_LIQUIBASE_MASTER_FILES),
            new AbstractMap.SimpleEntry<>("riceDataSource", RICE_LIQUIBASE_MASTER_FILES))
            .collect(Collectors.toMap(AbstractMap.SimpleEntry::getKey, AbstractMap.SimpleEntry::getValue));

    private static final String ADDITIONAL_PHASE5_LIQUIBASE_MASTER_FILE = "additional.phase5.liquibase.master.file";
    private static final String UPDATE_DATABASE_CONTEXT = "updateDatabaseContext";
    private static final String UPDATE_DATABASE_FULL_REBUILD = "updateDatabaseFullRebuild";

    private ClassPathXmlApplicationContext applicationContext;
    protected Properties properties = null;

    private LiquiRelational() {
    }

    public LiquiRelational(Properties properties) {
        this.properties = properties;
    }

    public static void main(String[] args) {
        Configurator.initialize(new DefaultConfiguration());

        LiquiRelational liquiRelational = new LiquiRelational();
        liquiRelational.updateDatabase();
        System.exit(0);
    }

    public void updateDatabase() {
        initializeContext();
        applyUpdates();
        applicationContext.close();
    }

    private void updateDatabase(DataSource dataSource, String dataSourceName) {
        Connection connection = null;
        try {
            connection = dataSource.getConnection();
            Database database = DatabaseFactory.getInstance().findCorrectDatabaseImplementation(new JdbcConnection(connection));

            ResourceAccessor resourceAccessor = new ClassLoaderResourceAccessor(applicationContext.getClassLoader());
            String liquibaseContext = getBaseProperty(UPDATE_DATABASE_CONTEXT);

            runUpdatesPhase(database, resourceAccessor, liquibaseContext, dataSourceName);
        } catch (SQLException | DatabaseException e) {
            LOG.error("Failed to get datasource.", e);
            throw new RuntimeException(e);
        } finally {
            if (connection != null) {
                try {
                    connection.close();
                } catch (SQLException e) {
                    LOG.warn("Failed to close connection.", e);
                    connection = null;
                    throw new RuntimeException(e);
                }
            }
        }
    }

    private void initializeContext() {
        long startInit = System.currentTimeMillis();
        if (LOG.isInfoEnabled()) {
            LOG.info("Initializing LiquiRelational Context...");
        }

        applicationContext = new ClassPathXmlApplicationContext(
                "org/kuali/kfs/sys/datatools/liquirelational/kfs-liqui-relational-bootstrap.xml");
        if (this.properties != null) {
            applicationContext.getEnvironment().getPropertySources().addFirst(new PropertiesSource("properties",
                    this.properties));
        } else {
            this.properties = applicationContext.getBean("properties", Properties.class);
        }
        applicationContext.start();

        long endInit = System.currentTimeMillis();
        if (LOG.isInfoEnabled()) {
            LOG.info("...LiquiRelational Context successfully initialized, startup took " + (endInit - startInit) +
                    " ms.");
        }
    }

    private void applyUpdates() {
        applyDatabaseUpdates("riceDataSource");
        applyDatabaseUpdates("dataSource");
    }

    private void applyDatabaseUpdates(String dataSourceName) {
        DataSource dataSource = applicationContext.getBean(dataSourceName, DataSource.class);
        updateDatabase(dataSource, dataSourceName);
    }

    private List<String> getPhasesToRun(String dataSourceName) {
        String updateDatabaseFullRebuild = getBaseProperty(UPDATE_DATABASE_FULL_REBUILD);

        List<String> phaseFilenames = new ArrayList<>();
        if (Boolean.parseBoolean(updateDatabaseFullRebuild)) {
            LOG.info("getAutoPhasesToRun() Running all phases");
            for (int i = 1; i < 5; i++) {
                phaseFilenames.add(LIQUIBASE_MASTER_FILES.get(dataSourceName).get(i));
            }
        } else {
            LOG.info("getAutoPhasesToRun() Running phase 5 only");
        }

        String additionalPhase5LiquibaseMasterFile = getBaseProperty(dataSourceName + "." +
                ADDITIONAL_PHASE5_LIQUIBASE_MASTER_FILE);
        if (StringUtils.isNotBlank(additionalPhase5LiquibaseMasterFile)) {
            phaseFilenames.add(additionalPhase5LiquibaseMasterFile);
        }

        phaseFilenames.add(LIQUIBASE_MASTER_FILES.get(dataSourceName).get(5));
        return phaseFilenames;
    }

    private void runUpdatesPhase(Database database, ResourceAccessor resourceAccessor, String liquibaseContext,
            String dataSourceName) {
        List<String> phaseFilenames = getPhasesToRun(dataSourceName);

        for (String filename : phaseFilenames) {
            try {
                LOG.info("Processing " + filename);
                Liquibase liquibase = new Liquibase(filename, resourceAccessor, database);
                liquibase.update(liquibaseContext);
            } catch (LiquibaseException e) {
                throw new RuntimeException("Failed to create Liquibase for " + filename, e);
            }
        }
    }

    private String getBaseProperty(String propertyName) {
        return properties.getProperty(propertyName);
    }

    public static class PropertiesSource extends PropertySource<String> {
        protected Properties properties;

        PropertiesSource(String name, Properties properties) {
            super(name);
            this.properties = properties;
        }

        @Override
        public String getProperty(String s) {
            if (properties != null) {
                return String.valueOf(properties.get(s));
            } else {
                return null;
            }
        }
    }
}
