/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.sys.datatools.liquirelational;

import liquibase.Liquibase;
import liquibase.database.Database;
import liquibase.database.DatabaseFactory;
import liquibase.database.jvm.JdbcConnection;
import liquibase.exception.DatabaseException;
import liquibase.exception.LiquibaseException;
import liquibase.resource.ClassLoaderResourceAccessor;
import liquibase.resource.ResourceAccessor;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.core.LoggerContext;
import org.apache.logging.log4j.core.config.Configurator;
import org.apache.logging.log4j.core.config.DefaultConfiguration;
import org.springframework.context.support.ClassPathXmlApplicationContext;
import org.springframework.core.env.PropertySource;

import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import static java.util.Map.entry;

public class LiquiRelational {

    private static final Logger LOG = LogManager.getLogger();

    private static final Map<Integer, String> LIQUIBASE_MASTER_FILES = Map.ofEntries(
            entry(1, "org/kuali/kfs/db/liquibase-phase1-master.xml"),
            entry(2, "org/kuali/kfs/db/liquibase-phase2-master.xml"),
            entry(3, "org/kuali/kfs/db/liquibase-phase3-master.xml"),
            entry(4, "org/kuali/kfs/db/liquibase-phase4-master.xml"),
            entry(5, "org/kuali/kfs/db/liquibase-phase5-master.xml")
    );

    private static final String ADDITIONAL_PHASE5_LIQUIBASE_MASTER_FILE = "additional.phase5.liquibase.master.file";
    private static final String UPDATE_DATABASE_CONTEXT = "updateDatabaseContext";
    private static final String UPDATE_DATABASE_FULL_REBUILD = "updateDatabaseFullRebuild";

    private ClassPathXmlApplicationContext applicationContext;
    protected Properties properties;

    private LiquiRelational() {
    }

    public LiquiRelational(final Properties properties) {
        this.properties = properties;
    }

    public static void main(final String[] args) {
        try (LoggerContext ignored = Configurator.initialize(new DefaultConfiguration())) {
            final LiquiRelational liquiRelational = new LiquiRelational();
            liquiRelational.updateDatabase();
        }

        System.exit(0);
    }

    public void updateDatabase() {
        initializeContext();
        applyUpdates();
        applicationContext.close();
    }

    private void initializeContext() {
        final long startInit = System.currentTimeMillis();
        LOG.info("Initializing LiquiRelational Context...");

        applicationContext = new ClassPathXmlApplicationContext(
                "org/kuali/kfs/sys/datatools/liquirelational/kfs-liqui-relational-bootstrap.xml");
        if (properties != null) {
            applicationContext.getEnvironment().getPropertySources().addFirst(new PropertiesSource("properties",
                    properties));
        } else {
            properties = applicationContext.getBean("properties", Properties.class);
        }
        applicationContext.start();

        final long endInit = System.currentTimeMillis();
        LOG.info("...LiquiRelational Context successfully initialized, startup took {} ms.", endInit - startInit);
    }

    private void applyUpdates() {
        final DataSource dataSource = applicationContext.getBean("dataSource", DataSource.class);
        try (Connection connection = dataSource.getConnection()) {
            final Database database =
                    DatabaseFactory.getInstance().findCorrectDatabaseImplementation(new JdbcConnection(connection));

            final ResourceAccessor resourceAccessor =
                    new ClassLoaderResourceAccessor(applicationContext.getClassLoader());
            final String liquibaseContext = getBaseProperty(UPDATE_DATABASE_CONTEXT);

            runUpdatesPhase(database, resourceAccessor, liquibaseContext);
        } catch (SQLException | DatabaseException e) {
            LOG.error("Failed to get datasource.", e);
            throw new RuntimeException(e);
        }
    }

    private List<String> getPhasesToRun() {
        final String updateDatabaseFullRebuild = getBaseProperty(UPDATE_DATABASE_FULL_REBUILD);

        final List<String> phaseFilenames = new ArrayList<>();
        if (Boolean.parseBoolean(updateDatabaseFullRebuild)) {
            LOG.info("getAutoPhasesToRun() Running all phases");
            for (int i = 1; i < 5; i++) {
                phaseFilenames.add(LIQUIBASE_MASTER_FILES.get(i));
            }
        } else {
            LOG.info("getAutoPhasesToRun() Running phase 5 only");
        }

        final String additionalPhase5LiquibaseMasterFile = getBaseProperty(ADDITIONAL_PHASE5_LIQUIBASE_MASTER_FILE);
        if (StringUtils.isNotBlank(additionalPhase5LiquibaseMasterFile)) {
            phaseFilenames.add(additionalPhase5LiquibaseMasterFile);
        }

        phaseFilenames.add(LIQUIBASE_MASTER_FILES.get(5));
        return phaseFilenames;
    }

    private void runUpdatesPhase(
            final Database database,
            final ResourceAccessor resourceAccessor,
            final String liquibaseContext
    ) {
        final List<String> phaseFilenames = getPhasesToRun();

        for (final String filename : phaseFilenames) {
            try {
                LOG.info("Processing {}", filename);
                // This will trigger IDEA's "AutoCloseable used without 'try-with-resources'" Inspection. I don't see a
                // way around it, but it's not a problem. Liquibase.close() just calls database.close(), which just
                // call connection.close(). The connection is created in a try-with-resources block in the calling
                // method, so it will be closed. Our non-standard use of Liquibase is the real culprit here.
                final Liquibase liquibase = new Liquibase(filename, resourceAccessor, database);
                liquibase.update(liquibaseContext);
            } catch (LiquibaseException e) {
                throw new RuntimeException("Failed to create Liquibase for " + filename, e);
            }
        }
    }

    private String getBaseProperty(final String propertyName) {
        return properties.getProperty(propertyName);
    }

    public static class PropertiesSource extends PropertySource<String> {
        protected Properties properties;

        PropertiesSource(final String name, final Properties properties) {
            super(name);
            this.properties = properties;
        }

        @Override
        public String getProperty(final String s) {
            if (properties != null) {
                return String.valueOf(properties.get(s));
            } else {
                return null;
            }
        }
    }
}
