/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.sys.datatools.liquirelational;

import liquibase.Liquibase;
import liquibase.database.Database;
import liquibase.database.DatabaseFactory;
import liquibase.database.jvm.JdbcConnection;
import liquibase.exception.DatabaseException;
import liquibase.exception.LiquibaseException;
import liquibase.resource.ClassLoaderResourceAccessor;
import liquibase.resource.ResourceAccessor;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.core.LoggerContext;
import org.apache.logging.log4j.core.config.Configurator;
import org.apache.logging.log4j.core.config.DefaultConfiguration;
import org.springframework.context.support.ClassPathXmlApplicationContext;

import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import static java.util.Map.entry;

public class LiquiRelational {

    private static final Logger LOG = LogManager.getLogger();

    private static final Map<Integer, String> LIQUIBASE_MASTER_FILES = Map.ofEntries(
            entry(1, "org/kuali/kfs/db/liquibase-phase1-master.xml"),
            entry(2, "org/kuali/kfs/db/liquibase-phase2-master.xml"),
            entry(3, "org/kuali/kfs/db/liquibase-phase3-master.xml"),
            entry(4, "org/kuali/kfs/db/liquibase-phase4-master.xml"),
            entry(5, "org/kuali/kfs/db/liquibase-phase5-master.xml")
    );

    private static final String ADDITIONAL_PHASE5_LIQUIBASE_MASTER_FILE = "additional.phase5.liquibase.master.file";
    private static final String UPDATE_DATABASE_CONTEXT = "updateDatabaseContext";
    private static final String UPDATE_DATABASE_FULL_REBUILD = "updateDatabaseFullRebuild";

    private final String liquibaseContext;
    private final boolean fullRebuild;
    private final String additionalPhase5MasterFile;
    private final DataSource dataSource;
    private final ResourceAccessor resourceAccessor;

    public LiquiRelational(
            final String liquibaseContext,
            final boolean fullRebuild,
            final String additionalPhase5MasterFile,
            final DataSource dataSource,
            final ResourceAccessor resourceAccessor
    ) {
        this.liquibaseContext = liquibaseContext;
        this.fullRebuild = fullRebuild;
        this.additionalPhase5MasterFile = additionalPhase5MasterFile;
        this.dataSource = dataSource;
        this.resourceAccessor = resourceAccessor;
    }

    public static void main(final String[] args) {
        try (LoggerContext ignored = Configurator.initialize(new DefaultConfiguration())) {
            final ClassPathXmlApplicationContext applicationContext = initializeContext();

            final Properties properties = applicationContext.getBean("properties", Properties.class);
            final String liquibaseContext = properties.getProperty(UPDATE_DATABASE_CONTEXT);
            final boolean fullRebuild = Boolean.parseBoolean(properties.getProperty(UPDATE_DATABASE_FULL_REBUILD));
            final String additionalPhase5MasterFile = properties.getProperty(ADDITIONAL_PHASE5_LIQUIBASE_MASTER_FILE);

            final DataSource dataSource = applicationContext.getBean("dataSource", DataSource.class);
            final ResourceAccessor resourceAccessor = new ClassLoaderResourceAccessor(applicationContext.getClassLoader());

            new LiquiRelational(liquibaseContext, fullRebuild, additionalPhase5MasterFile, dataSource, resourceAccessor)
                    .applyUpdates();

            applicationContext.close();
        }
        System.exit(0);
    }

    private static ClassPathXmlApplicationContext initializeContext() {
        LOG.info("Initializing LiquiRelational Context...");
        final ClassPathXmlApplicationContext applicationContext =
                new ClassPathXmlApplicationContext("org/kuali/kfs/sys/datatools/liquirelational/kfs-liqui-relational-bootstrap.xml");
        applicationContext.start();
        LOG.info("...LiquiRelational Context successfully initialized");
        return applicationContext;
    }

    public void applyUpdates() {
        try (Connection connection = dataSource.getConnection()) {
            final Database database =
                    DatabaseFactory.getInstance().findCorrectDatabaseImplementation(new JdbcConnection(connection));

            final List<String> phaseFilenames = getPhasesToRun();

            runUpdatesPhase(database, phaseFilenames);
        } catch (SQLException | DatabaseException e) {
            LOG.error("Failed to get datasource.", e);
            throw new RuntimeException(e);
        }
    }

    private List<String> getPhasesToRun() {
        final List<String> phaseFilenames = new ArrayList<>();
        if (fullRebuild) {
            LOG.info("getAutoPhasesToRun() Running all phases");
            for (int i = 1; i < 5; i++) {
                phaseFilenames.add(LIQUIBASE_MASTER_FILES.get(i));
            }
        } else {
            LOG.info("getAutoPhasesToRun() Running phase 5 only");
        }

        if (StringUtils.isNotBlank(additionalPhase5MasterFile)) {
            phaseFilenames.add(additionalPhase5MasterFile);
        }

        phaseFilenames.add(LIQUIBASE_MASTER_FILES.get(5));
        return phaseFilenames;
    }

    private void runUpdatesPhase(
            final Database database,
            final List<String> phaseFilenames
    ) {
        for (final String filename : phaseFilenames) {
            try {
                LOG.info("Processing {}", filename);
                // This will trigger IDEA's "AutoCloseable used without 'try-with-resources'" Inspection. I don't see a
                // way around it, but it's not a problem. Liquibase.close() just calls database.close(), which just
                // call connection.close(). The connection is created in a try-with-resources block in the calling
                // method, so it will be closed. Our non-standard use of Liquibase is the real culprit here.
                final Liquibase liquibase = new Liquibase(filename, resourceAccessor, database);
                liquibase.update(liquibaseContext);
            } catch (final LiquibaseException e) {
                throw new RuntimeException("Failed to create Liquibase for " + filename, e);
            }
        }
    }

}
