/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2020 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ec.businessobject;

import org.kuali.kfs.coa.businessobject.Account;
import org.kuali.kfs.module.ec.document.EffortCertificationDocument;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.sys.service.FinancialSystemUserService;
import org.kuali.rice.kew.api.KewApiConstants;
import org.kuali.rice.kew.api.KewApiServiceLocator;
import org.kuali.rice.kew.api.action.ActionItem;
import org.kuali.rice.kim.api.identity.Person;
import org.kuali.rice.kim.api.identity.PersonService;
import org.kuali.rice.kim.api.services.KimApiServiceLocator;

import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * This guy exists entirely for lookups. Ideally we wouldn't need a separate BO for this. Even though a doc is BO,
 * the doc entry and bo entry types are siblings not parent/child. If I try to create both a bo entry and a doc entry
 * backed by the document class, that creates a whole host of problems. So we are doing this.
 */
public class EffortCertificationBO extends EffortCertificationDocument {

    /**
     * Queries workflow to get users who have an approval request for this effort certification.
     *
     * @return names of users (separated by comma) who have an approval request
     */
    public String getNextApprovers() {
        List<ActionItem> actionList = KewApiServiceLocator.getActionListService().getAllActionItems(getDocumentHeader()
                .getDocumentNumber());
        PersonService personService = KimApiServiceLocator.getPersonService();

        return actionList.stream()
                .filter(actionItem -> actionItem.getActionRequestCd().equals(KewApiConstants.ACTION_REQUEST_APPROVE_REQ))
                .map(ActionItem::getPrincipalId)
                .filter(Objects::nonNull)
                .map(personService::getPerson)
                .map(Person::getName)
                .collect(Collectors.joining("; "));
    }

    public String getOrganizationCodesForLookup() {
        Set<String> codes = getEffortCertificationDetailLines().stream()
                .map(EffortCertificationDetail::getAccount)
                .map(Account::getOrganizationCode)
                .filter(Objects::nonNull)
                .collect(Collectors.toSet());

        // trim the brackets
        return codes.toString().substring(1, codes.toString().length() - 1);
    }

    public String getChartCodesForLookup() {
        Set<String> codes = getEffortCertificationDetailLines().stream()
                .map(EffortCertificationDetail::getChartOfAccountsCode)
                .filter(Objects::nonNull)
                .collect(Collectors.toSet());

        // trim the brackets
        return codes.toString().substring(1, codes.toString().length() - 1);
    }

    public String getEmployeeName() {
        return SpringContext.getBean(FinancialSystemUserService.class).getPersonNameByEmployeeId(emplid);
    }
}
