/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ec.businessobject;

import org.kuali.kfs.coa.businessobject.AccountingPeriod;
import org.kuali.kfs.integration.ec.EffortCertificationReport;
import org.kuali.kfs.krad.bo.PersistableBusinessObjectBase;
import org.kuali.kfs.module.ec.EffortPropertyConstants;
import org.kuali.kfs.module.ec.util.AccountingPeriodMonth;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.businessobject.FiscalYearBasedBusinessObject;
import org.kuali.kfs.sys.businessobject.SystemOptions;
import org.kuali.kfs.core.api.mo.common.active.MutableInactivatable;

import java.sql.Date;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

/**
 * Business Object for the Effort Certification Report Definition Table.
 */
public class EffortCertificationReportDefinition extends PersistableBusinessObjectBase implements
        EffortCertificationReport, MutableInactivatable, FiscalYearBasedBusinessObject {

    private Integer universityFiscalYear;
    private String effortCertificationReportNumber;
    private String effortCertificationReportPeriodTitle;
    private String effortCertificationReportPeriodStatusCode;
    private Integer expenseTransferFiscalYear;
    private String expenseTransferFiscalPeriodCode;
    private String effortCertificationReportTypeCode;
    private Date effortCertificationReportReturnDate;
    private Integer effortCertificationReportBeginFiscalYear;
    private String effortCertificationReportBeginPeriodCode;
    private Integer effortCertificationReportEndFiscalYear;
    private String effortCertificationReportEndPeriodCode;
    private boolean active;

    private SystemOptions options;
    private SystemOptions reportBeginFiscalYear;
    private SystemOptions reportEndFiscalYear;
    private AccountingPeriod reportBeginPeriod;
    private AccountingPeriod reportEndPeriod;
    private SystemOptions expenseTransferYear;
    private AccountingPeriod expenseTransferFiscalPeriod;
    private EffortCertificationPeriodStatusCode effortCertificationPeriodStatusCode;
    private EffortCertificationReportType effortCertificationReportType;
    private Collection<EffortCertificationReportPosition> effortCertificationReportPositions;

    private Map<Integer, Set<String>> reportPeriods;

    public EffortCertificationReportDefinition() {
        super();
        effortCertificationReportPositions = new ArrayList<>();
    }

    @Override
    public Integer getUniversityFiscalYear() {
        return universityFiscalYear;
    }

    @Override
    public void setUniversityFiscalYear(final Integer universityFiscalYear) {
        this.universityFiscalYear = universityFiscalYear;
    }

    @Override
    public String getEffortCertificationReportNumber() {
        return effortCertificationReportNumber;
    }

    public void setEffortCertificationReportNumber(final String effortCertificationReportNumber) {
        this.effortCertificationReportNumber = effortCertificationReportNumber;
    }

    @Override
    public String getEffortCertificationReportPeriodTitle() {
        return effortCertificationReportPeriodTitle;
    }

    public void setEffortCertificationReportPeriodTitle(final String effortCertificationReportPeriodTitle) {
        this.effortCertificationReportPeriodTitle = effortCertificationReportPeriodTitle;
    }

    @Override
    public String getEffortCertificationReportPeriodStatusCode() {
        return effortCertificationReportPeriodStatusCode;
    }

    public void setEffortCertificationReportPeriodStatusCode(final String effortCertificationReportPeriodStatusCode) {
        this.effortCertificationReportPeriodStatusCode = effortCertificationReportPeriodStatusCode;
    }

    @Override
    public Integer getExpenseTransferFiscalYear() {
        return expenseTransferFiscalYear;
    }

    public void setExpenseTransferFiscalYear(final Integer expenseTransferFiscalYear) {
        this.expenseTransferFiscalYear = expenseTransferFiscalYear;
    }

    @Override
    public String getExpenseTransferFiscalPeriodCode() {
        return expenseTransferFiscalPeriodCode;
    }

    public void setExpenseTransferFiscalPeriodCode(final String expenseTransferFiscalPeriodCode) {
        this.expenseTransferFiscalPeriodCode = expenseTransferFiscalPeriodCode;
    }

    @Override
    public String getEffortCertificationReportTypeCode() {
        return effortCertificationReportTypeCode;
    }

    public void setEffortCertificationReportTypeCode(final String effortCertificationReportTypeCode) {
        this.effortCertificationReportTypeCode = effortCertificationReportTypeCode;
    }

    @Override
    public Date getEffortCertificationReportReturnDate() {
        return effortCertificationReportReturnDate;
    }

    public void setEffortCertificationReportReturnDate(final Date effortCertificationReportReturnDate) {
        this.effortCertificationReportReturnDate = effortCertificationReportReturnDate;
    }

    @Override
    public Integer getEffortCertificationReportBeginFiscalYear() {
        return effortCertificationReportBeginFiscalYear;
    }

    public void setEffortCertificationReportBeginFiscalYear(final Integer effortCertificationReportBeginFiscalYear) {
        this.effortCertificationReportBeginFiscalYear = effortCertificationReportBeginFiscalYear;
    }

    @Override
    public String getEffortCertificationReportBeginPeriodCode() {
        return effortCertificationReportBeginPeriodCode;
    }

    public void setEffortCertificationReportBeginPeriodCode(final String effortCertificationReportBeginPeriodCode) {
        this.effortCertificationReportBeginPeriodCode = effortCertificationReportBeginPeriodCode;
    }

    @Override
    public Integer getEffortCertificationReportEndFiscalYear() {
        return effortCertificationReportEndFiscalYear;
    }

    public void setEffortCertificationReportEndFiscalYear(final Integer effortCertificationReportEndFiscalYear) {
        this.effortCertificationReportEndFiscalYear = effortCertificationReportEndFiscalYear;
    }

    @Override
    public String getEffortCertificationReportEndPeriodCode() {
        return effortCertificationReportEndPeriodCode;
    }

    public void setEffortCertificationReportEndPeriodCode(final String effortCertificationReportEndPeriodCode) {
        this.effortCertificationReportEndPeriodCode = effortCertificationReportEndPeriodCode;
    }

    public AccountingPeriod getExpenseTransferFiscalPeriod() {
        return expenseTransferFiscalPeriod;
    }

    @Deprecated
    public void setExpenseTransferFiscalPeriod(final AccountingPeriod expenseTransferFiscalPeriod) {
        this.expenseTransferFiscalPeriod = expenseTransferFiscalPeriod;
    }

    public SystemOptions getExpenseTransferYear() {
        return expenseTransferYear;
    }

    @Deprecated
    public void setExpenseTransferYear(final SystemOptions expenseTransferYear) {
        this.expenseTransferYear = expenseTransferYear;
    }

    public SystemOptions getReportBeginFiscalYear() {
        return reportBeginFiscalYear;
    }

    @Deprecated
    public void setReportBeginFiscalYear(final SystemOptions reportBeginFiscalYear) {
        this.reportBeginFiscalYear = reportBeginFiscalYear;
    }

    public SystemOptions getReportEndFiscalYear() {
        return reportEndFiscalYear;
    }

    @Deprecated
    public void setReportEndFiscalYear(final SystemOptions reportEndFiscalYear) {
        this.reportEndFiscalYear = reportEndFiscalYear;
    }

    public AccountingPeriod getReportBeginPeriod() {
        return reportBeginPeriod;
    }

    @Deprecated
    public void setReportBeginPeriod(final AccountingPeriod reportBeginPeriod) {
        this.reportBeginPeriod = reportBeginPeriod;
    }

    public AccountingPeriod getReportEndPeriod() {
        return reportEndPeriod;
    }

    @Deprecated
    public void setReportEndPeriod(final AccountingPeriod reportEndPeriod) {
        this.reportEndPeriod = reportEndPeriod;
    }

    public EffortCertificationPeriodStatusCode getEffortCertificationPeriodStatusCode() {
        return effortCertificationPeriodStatusCode;
    }

    @Deprecated
    public void setEffortCertificationPeriodStatusCode(
            final EffortCertificationPeriodStatusCode effortCertificationPeriodStatusCode) {
        this.effortCertificationPeriodStatusCode = effortCertificationPeriodStatusCode;
    }

    public EffortCertificationReportType getEffortCertificationReportType() {
        return effortCertificationReportType;
    }

    @Deprecated
    public void setEffortCertificationReportType(final EffortCertificationReportType effortCertificationReportType) {
        this.effortCertificationReportType = effortCertificationReportType;
    }

    public Collection<EffortCertificationReportPosition> getEffortCertificationReportPositions() {
        return effortCertificationReportPositions;
    }

    public void setEffortCertificationReportPositions(
            final Collection<EffortCertificationReportPosition> effortCertificationReportPositions) {
        this.effortCertificationReportPositions = effortCertificationReportPositions;
    }

    @Override
    public boolean isActive() {
        return active;
    }

    @Override
    public void setActive(final boolean active) {
        this.active = active;
    }

    public SystemOptions getOptions() {
        return options;
    }

    @Deprecated
    public void setOptions(final SystemOptions options) {
        this.options = options;
    }

    /**
     * build a primary key field map for the current report definition
     *
     * @return a primary key field map for the current report definition
     */
    public Map<String, String> buildKeyMapForCurrentReportDefinition() {
        return buildKeyMap(getUniversityFiscalYear(), getEffortCertificationReportNumber());
    }

    /**
     * build a primary key field map for a report definition from the given values
     *
     * @param universityFiscalYear the given fiscal year
     * @param reportNumber         the given report number
     * @return a primary key field map for a report definition
     */
    public static Map<String, String> buildKeyMap(final Integer universityFiscalYear, final String reportNumber) {
        final Map<String, String> primaryKeyMap = new HashMap<>();

        final String stringFiscalYear = universityFiscalYear == null ? "" : universityFiscalYear.toString();
        primaryKeyMap.put(KFSPropertyConstants.UNIVERSITY_FISCAL_YEAR, stringFiscalYear);
        primaryKeyMap.put(EffortPropertyConstants.EFFORT_CERTIFICATION_REPORT_NUMBER, reportNumber);

        return primaryKeyMap;
    }

    /**
     * build all report periods map covered by the specified report definition
     */
    public void buildReportPeriods() {
        final Integer beginYear = getEffortCertificationReportBeginFiscalYear();
        final String beginPeriodCode = getEffortCertificationReportBeginPeriodCode();
        final Integer endYear = getEffortCertificationReportEndFiscalYear();
        final String endPeriodCode = getEffortCertificationReportEndPeriodCode();

        setReportPeriods(AccountingPeriodMonth.findAccountingPeriodsBetween(beginYear, beginPeriodCode, endYear,
                endPeriodCode));
    }

    public Map<Integer, Set<String>> getReportPeriods() {
        if (reportPeriods == null) {
            buildReportPeriods();
        }
        return reportPeriods;
    }

    public void setReportPeriods(final Map<Integer, Set<String>> reportPeriods) {
        this.reportPeriods = reportPeriods;
    }
}
