/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ec.document;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.coa.businessobject.Organization;
import org.kuali.kfs.kim.impl.identity.Person;
import org.kuali.kfs.sys.service.impl.KfsParameterConstants.COMPONENT;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.krad.UserSession;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ec.businessobject.EffortCertificationDetail;
import org.kuali.kfs.module.ec.businessobject.EffortCertificationReportDefinition;
import org.kuali.kfs.module.ec.service.EffortCertificationDocumentService;
import org.kuali.kfs.module.ec.util.EffortCertificationParameterFinder;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.businessobject.SystemOptions;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.sys.document.FinancialSystemTransactionalDocumentBase;
import org.kuali.kfs.sys.service.impl.KfsParameterConstants;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.kew.api.WorkflowDocument;
import org.kuali.kfs.kew.framework.postprocessor.DocumentRouteStatusChange;
import org.kuali.kfs.kim.api.identity.PersonService;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

@COMPONENT(component = "EffortCertification")
public class EffortCertificationDocument extends FinancialSystemTransactionalDocumentBase {

    private static final Logger LOG = LogManager.getLogger();

    protected static final String DO_AWARD_SPLIT = "DoAwardSplit";
    protected static final String DO_RECREATE_SPLIT = "DoRecreateSplit";

    protected String effortCertificationReportNumber;
    protected boolean effortCertificationDocumentCode;
    protected Integer universityFiscalYear;
    protected String emplid;
    protected String organizationCode;
    protected KualiDecimal financialDocumentTotalAmount;

    protected Integer totalEffortPercent;
    protected Integer totalOriginalEffortPercent;
    protected KualiDecimal totalPayrollAmount;
    protected KualiDecimal totalOriginalPayrollAmount;

    protected EffortCertificationReportDefinition effortCertificationReportDefinition;
    protected Person employee;
    protected Organization organization;
    protected SystemOptions options;

    protected List<EffortCertificationDetail> effortCertificationDetailLines;
    protected List<EffortCertificationDetail> summarizedDetailLines;

    protected Person ledgerPerson;

    public EffortCertificationDocument() {
        super();

        effortCertificationDetailLines = new ArrayList<>();
        summarizedDetailLines = new ArrayList<>();
    }

    public String getEffortCertificationReportNumber() {
        return effortCertificationReportNumber;
    }

    public void setEffortCertificationReportNumber(final String effortCertificationReportNumber) {
        this.effortCertificationReportNumber = effortCertificationReportNumber;
    }

    public boolean getEffortCertificationDocumentCode() {
        return effortCertificationDocumentCode;
    }

    public void setEffortCertificationDocumentCode(final boolean effortCertificationDocumentCode) {
        this.effortCertificationDocumentCode = effortCertificationDocumentCode;
    }

    public Integer getUniversityFiscalYear() {
        return universityFiscalYear;
    }

    public void setUniversityFiscalYear(final Integer universityFiscalYear) {
        this.universityFiscalYear = universityFiscalYear;
    }

    public String getOrganizationCode() {
        return organizationCode;
    }

    public void setOrganizationCode(final String organizationCode) {
        this.organizationCode = organizationCode;
    }

    public Organization getOrganization() {
        return organization;
    }

    public void setOrganization(final Organization organization) {
        this.organization = organization;
    }

    public String getEmplid() {
        return emplid;
    }

    public void setEmplid(final String emplid) {
        this.emplid = emplid;
    }

    public EffortCertificationReportDefinition getEffortCertificationReportDefinition() {
        return effortCertificationReportDefinition;
    }

    @Deprecated
    public void setEffortCertificationReportDefinition(
            final EffortCertificationReportDefinition effortCertificationReportDefinition) {
        this.effortCertificationReportDefinition = effortCertificationReportDefinition;
    }

    public Person getEmployee() {
        if (StringUtils.isNotBlank(getEmplid())) {
            return SpringContext.getBean(PersonService.class).getPersonByEmployeeId(getEmplid());
        } else {
            return null;
        }
    }

    public void setEmployee(final Person employee) {
        this.employee = employee;
    }

    public Person getLedgerPerson() {
        if ((ledgerPerson == null || !StringUtils.equals(ledgerPerson.getEmployeeId(), emplid))
                && StringUtils.isNotBlank(emplid)) {
            ledgerPerson = SpringContext.getBean(PersonService.class).getPersonByEmployeeId(emplid);
        }

        return ledgerPerson;
    }

    public void setLedgerPerson(final Person ledgerPerson) {
        this.ledgerPerson = ledgerPerson;
    }

    public SystemOptions getOptions() {
        return options;
    }

    public void setOptions(final SystemOptions options) {
        this.options = options;
    }

    public List<EffortCertificationDetail> getEffortCertificationDetailLines() {
        return effortCertificationDetailLines;
    }

    @Deprecated
    public void setEffortCertificationDetailLines(final List<EffortCertificationDetail> effortCertificationDetailLines) {
        this.effortCertificationDetailLines = effortCertificationDetailLines;
    }

    /**
     * @param effortCertificationDocument the given effort certification document
     * @return the total amount of the given effort certification document
     */
    public static KualiDecimal getDocumentTotalAmount(final EffortCertificationDocument effortCertificationDocument) {
        final List<EffortCertificationDetail> detailLines = effortCertificationDocument.getEffortCertificationDetailLines();
        return EffortCertificationDetail.getTotalPayrollAmount(detailLines);
    }

    @Override
    public void doRouteStatusChange(final DocumentRouteStatusChange statusChangeEvent) {
        LOG.debug("doRouteStatusChange() start...");

        super.doRouteStatusChange(statusChangeEvent);
        final WorkflowDocument workflowDocument = getDocumentHeader().getWorkflowDocument();
        if (workflowDocument.isFinal()) {
            GlobalVariables.setUserSession(new UserSession(KFSConstants.SYSTEM_USER));
            SpringContext.getBean(EffortCertificationDocumentService.class)
                    .generateSalaryExpenseTransferDocument(this);
        }
    }

    public Integer getTotalEffortPercent() {
        return EffortCertificationDetail.getTotalEffortPercent(getEffortCertificationDetailLines());
    }

    public Integer getTotalOriginalEffortPercent() {
        return EffortCertificationDetail.getTotalOriginalEffortPercent(getEffortCertificationDetailLines());
    }

    public KualiDecimal getTotalPayrollAmount() {
        return EffortCertificationDetail.getTotalPayrollAmount(getEffortCertificationDetailLines());
    }

    public KualiDecimal getTotalOriginalPayrollAmount() {
        return EffortCertificationDetail.getTotalOriginalPayrollAmount(getEffortCertificationDetailLines());
    }

    /**
     * @return the detail lines that have max payroll amount
     */
    public List<EffortCertificationDetail> getEffortCertificationDetailWithMaxPayrollAmount() {
        final List<EffortCertificationDetail> detailLines = new ArrayList<>();

        KualiDecimal maxAmount = null;
        for (final EffortCertificationDetail line : getEffortCertificationDetailLines()) {
            final KualiDecimal currentAmount = line.getEffortCertificationPayrollAmount();

            if (maxAmount == null) {
                maxAmount = currentAmount;
                detailLines.add(line);
                continue;
            }

            if (maxAmount.isLessThan(currentAmount)) {
                detailLines.removeAll(detailLines);
                maxAmount = currentAmount;
                detailLines.add(line);
            } else if (maxAmount.equals(currentAmount)) {
                detailLines.add(line);
            }
        }

        return detailLines;
    }

    /**
     * @return the total updated effort for all federal detail lines
     */
    public Integer getFederalTotalEffortPercent() {
        Integer effortFederalTotal = 0;
        final List<EffortCertificationDetail> detailLineList = getEffortCertificationDetailLines();

        for (final EffortCertificationDetail detailLine : detailLineList) {
            if (detailLine.isFederalOrFederalPassThroughIndicator()) {
                effortFederalTotal += detailLine.getEffortCertificationUpdatedOverallPercent();
            }
        }

        return effortFederalTotal;
    }

    /**
     * @return the total original effort for all federal detail lines
     */
    public Integer getFederalTotalOriginalEffortPercent() {
        Integer effortOrigFederalTotal = 0;
        final List<EffortCertificationDetail> detailLineList = getEffortCertificationDetailLines();

        for (final EffortCertificationDetail detailLine : detailLineList) {
            if (detailLine.isFederalOrFederalPassThroughIndicator()) {
                effortOrigFederalTotal += detailLine.getEffortCertificationCalculatedOverallPercent();
            }
        }

        return effortOrigFederalTotal;
    }

    /**
     * @return the total original fringe benefit amount for federal pass through detail lines
     */
    public KualiDecimal getFederalTotalOriginalFringeBenefit() {
        KualiDecimal totalBenAmount = KualiDecimal.ZERO;
        final List<EffortCertificationDetail> detailLineList = getEffortCertificationDetailLines();

        for (final EffortCertificationDetail detailLine : detailLineList) {
            if (detailLine.isFederalOrFederalPassThroughIndicator()) {
                totalBenAmount = totalBenAmount.add(detailLine.getOriginalFringeBenefitAmount());
            }
        }

        return totalBenAmount;
    }

    /**
     * @return the total original fringe benefit amount for non federal pass through detail lines
     */
    public KualiDecimal getOtherTotalOriginalFringeBenefit() {
        KualiDecimal totalBenAmount = KualiDecimal.ZERO;
        final List<EffortCertificationDetail> detailLineList = getEffortCertificationDetailLines();

        for (final EffortCertificationDetail detailLine : detailLineList) {
            if (!detailLine.isFederalOrFederalPassThroughIndicator()) {
                totalBenAmount = totalBenAmount.add(detailLine.getOriginalFringeBenefitAmount());
            }
        }

        return totalBenAmount;
    }

    /**
     * @return total federal benefit amount for federal pass through detail lines
     */
    public KualiDecimal getFederalTotalFringeBenefit() {
        KualiDecimal totalBenAmount = KualiDecimal.ZERO;
        final List<EffortCertificationDetail> detailLineList = getEffortCertificationDetailLines();

        for (final EffortCertificationDetail detailLine : detailLineList) {
            if (detailLine.isFederalOrFederalPassThroughIndicator()) {
                totalBenAmount = totalBenAmount.add(detailLine.getFringeBenefitAmount());
            }
        }

        return totalBenAmount;
    }

    /**
     * @return the total fringe benefit amount for non federal pass through detail lines
     */
    public KualiDecimal getOtherTotalFringeBenefit() {
        KualiDecimal totalBenAmount = KualiDecimal.ZERO;
        final List<EffortCertificationDetail> detailLineList = getEffortCertificationDetailLines();

        for (final EffortCertificationDetail detailLine : detailLineList) {
            if (!detailLine.isFederalOrFederalPassThroughIndicator()) {
                totalBenAmount = totalBenAmount.add(detailLine.getFringeBenefitAmount());
            }
        }

        return totalBenAmount;
    }

    /**
     * @return the total original effort for non federal pass through detail lines
     */
    public Integer getOtherTotalOriginalEffortPercent() {
        Integer effortOrigOtherTotal = 0;
        final List<EffortCertificationDetail> detailLineList = getEffortCertificationDetailLines();

        for (final EffortCertificationDetail detailLine : detailLineList) {
            if (!detailLine.isFederalOrFederalPassThroughIndicator()) {
                effortOrigOtherTotal += detailLine.getEffortCertificationCalculatedOverallPercent();
            }
        }

        return effortOrigOtherTotal;
    }

    /**
     * @return the total updated effort for non federal pass through detail lines
     */
    public Integer getOtherTotalEffortPercent() {
        Integer effortOtherTotal = 0;
        final List<EffortCertificationDetail> detailLineList = getEffortCertificationDetailLines();

        for (final EffortCertificationDetail detailLine : detailLineList) {
            if (!detailLine.isFederalOrFederalPassThroughIndicator()) {
                effortOtherTotal += detailLine.getEffortCertificationUpdatedOverallPercent();
            }
        }

        return effortOtherTotal;
    }

    /**
     * @return total salary for federal detail lines
     */
    public KualiDecimal getFederalTotalPayrollAmount() {
        KualiDecimal salaryFederalTotal = KualiDecimal.ZERO;
        final List<EffortCertificationDetail> detailLineList = getEffortCertificationDetailLines();

        for (final EffortCertificationDetail detailLine : detailLineList) {
            if (detailLine.isFederalOrFederalPassThroughIndicator()) {
                salaryFederalTotal = salaryFederalTotal.add(detailLine.getEffortCertificationPayrollAmount());
            }
        }

        return salaryFederalTotal;
    }

    /**
     * @return the total original salary for federal pass through detail lines
     */
    public KualiDecimal getFederalTotalOriginalPayrollAmount() {
        KualiDecimal salaryOrigFederalTotal = KualiDecimal.ZERO;
        final List<EffortCertificationDetail> detailLineList = getEffortCertificationDetailLines();

        for (final EffortCertificationDetail detailLine : detailLineList) {
            if (detailLine.isFederalOrFederalPassThroughIndicator()) {
                salaryOrigFederalTotal = salaryOrigFederalTotal.add(detailLine
                        .getEffortCertificationOriginalPayrollAmount());
            }
        }

        return salaryOrigFederalTotal;
    }

    /**
     * @return the total original salary for non federal pass through detail lines
     */
    public KualiDecimal getOtherTotalOriginalPayrollAmount() {
        KualiDecimal salaryOrigOtherTotal = KualiDecimal.ZERO;
        final List<EffortCertificationDetail> detailLineList = getEffortCertificationDetailLines();

        for (final EffortCertificationDetail detailLine : detailLineList) {
            if (!detailLine.isFederalOrFederalPassThroughIndicator()) {
                salaryOrigOtherTotal = salaryOrigOtherTotal.add(detailLine
                        .getEffortCertificationOriginalPayrollAmount());
            }
        }

        return salaryOrigOtherTotal;
    }

    /**
     * @return total updated salary for non federal pass through detail lines
     */
    public KualiDecimal getOtherTotalPayrollAmount() {
        KualiDecimal salaryOtherTotal = KualiDecimal.ZERO;
        final List<EffortCertificationDetail> detailLineList = getEffortCertificationDetailLines();

        for (final EffortCertificationDetail detailLine : detailLineList) {
            if (!detailLine.isFederalOrFederalPassThroughIndicator()) {
                salaryOtherTotal = salaryOtherTotal.add(detailLine.getEffortCertificationPayrollAmount());
            }
        }

        return salaryOtherTotal;
    }

    public KualiDecimal getTotalFringeBenefit() {
        return EffortCertificationDetail.getTotalFringeBenefit(effortCertificationDetailLines);
    }

    public KualiDecimal getTotalOriginalFringeBenefit() {
        return EffortCertificationDetail.getTotalOriginalFringeBenefit(effortCertificationDetailLines);
    }

    @Override
    public void processAfterRetrieve() {
        super.processAfterRetrieve();

        // capture each line's salary amount before route level modification for later rule validation
        for (final EffortCertificationDetail detailLine : getEffortCertificationDetailLines()) {
            detailLine.setPersistedPayrollAmount(new KualiDecimal(detailLine.getEffortCertificationPayrollAmount()
                    .bigDecimalValue()));

            final int effortPercent = detailLine.getEffortCertificationUpdatedOverallPercent();
            detailLine.setPersistedEffortPercent(effortPercent);
        }

        // calculate original fringe benefits for each line
        for (final EffortCertificationDetail detailLine : getEffortCertificationDetailLines()) {
            detailLine.recalculateOriginalFringeBenefit();
        }
    }

    /**
     * @return true if effort has changed for any of the detail lines, false otherwise
     */
    public boolean isEffortDistributionChanged() {
        for (final EffortCertificationDetail detail : getEffortCertificationDetailLines()) {
            if (!detail.getEffortCertificationCalculatedOverallPercent()
                    .equals(detail.getEffortCertificationUpdatedOverallPercent())) {
                return true;
            }
        }

        return false;
    }

    /**
     * @return default position number for display based on the detail line with the maximum effort
     */
    public String getDefaultPositionNumber() {
        return getMaxEffortLine().getPositionNumber();
    }

    /**
     * @return default object code for display based on the value for the detail line with the maximum effort
     */
    public String getDefaultObjectCode() {
        return getMaxEffortLine().getFinancialObjectCode();
    }

    /**
     * @return the detail line with the maximum effort
     */
    protected EffortCertificationDetail getMaxEffortLine() {
        Integer maxEffort = 0;
        EffortCertificationDetail maxLine = null;
        final List<EffortCertificationDetail> detailLines = getEffortCertificationDetailLines();
        for (final EffortCertificationDetail detailLine : detailLines) {
            if (detailLine.getEffortCertificationUpdatedOverallPercent() > maxEffort) {
                maxEffort = detailLine.getEffortCertificationUpdatedOverallPercent();
                maxLine = detailLine;
            }
        }
        return maxLine;
    }

    @Override
    public void populateDocumentForRouting() {
        if (ObjectUtils.isNotNull(getTotalPayrollAmount())) {
            getDocumentHeader().setFinancialDocumentTotalAmount(getTotalPayrollAmount());
        } else {
            getDocumentHeader().setFinancialDocumentTotalAmount(new KualiDecimal(0));
        }
        super.populateDocumentForRouting();
    }

    /**
     * @return list of unique object codes contained in this document
     */
    public List<String> getObjectCodeList() {
        final List<String> uniqueObjectCodeList = new ArrayList<>();
        final List<EffortCertificationDetail> allObjectCodesList = getEffortCertificationDetailLines();
        for (final EffortCertificationDetail detail : allObjectCodesList) {
            if (!uniqueObjectCodeList.contains(detail.getFinancialObjectCode())) {
                uniqueObjectCodeList.add(detail.getFinancialObjectCode());
            }
        }

        return uniqueObjectCodeList;
    }

    /**
     * @return list of unique position numbers for this document
     */
    public List<String> getPositionList() {
        final List<String> uniquePositionList = new ArrayList<>();
        final List<EffortCertificationDetail> allPositionsList = getEffortCertificationDetailLines();
        for (final EffortCertificationDetail detail : allPositionsList) {
            if (!uniquePositionList.contains(detail.getPositionNumber())) {
                uniquePositionList.add(detail.getPositionNumber());
            }
        }

        return uniquePositionList;
    }

    /**
     * This is a marker setter method and does nothing.
     */
    public void setTotalOriginalPayrollAmount(final KualiDecimal totalOriginalPayrollAmount) {
    }

    public List<EffortCertificationDetail> getSummarizedDetailLines() {
        return summarizedDetailLines;
    }

    public void setSummarizedDetailLines(final List<EffortCertificationDetail> summarizedDetailLines) {
        this.summarizedDetailLines = summarizedDetailLines;
    }

    /**
     * Provides answers to the following splits:
     * Do Award Split
     * Do Recreate Split
     */
    @Override
    public boolean answerSplitNodeQuestion(final String nodeName) throws UnsupportedOperationException {
        if (nodeName.equals(EffortCertificationDocument.DO_AWARD_SPLIT)) {
            return isDoAwardSplit();
        }
        if (nodeName.equals(EffortCertificationDocument.DO_RECREATE_SPLIT)) {
            return isDoRecreateSplit();
        }
        if (nodeName.equals(KFSConstants.REQUIRES_WORK_STUDY_REVIEW)) {
            return checkObjectCodeForWorkstudy();
        }
        throw new UnsupportedOperationException("Cannot answer split question for this node you call \"" + nodeName +
                "\"");
    }

    /**
     * @return boolean
     */
    protected boolean checkObjectCodeForWorkstudy() {
        final Collection<String> workStudyRouteObjectCodes = SpringContext.getBean(ParameterService.class)
                .getParameterValuesAsString(KfsParameterConstants.FINANCIAL_SYSTEM_DOCUMENT.class,
                        KFSConstants.WORK_STUDY_ROUTE_OBJECT_CODES_PARAM_NM);

        final List<EffortCertificationDetail> effortCertificationDetails = getEffortCertificationDetailLines();

        // check object code in accounting lines
        for (final EffortCertificationDetail effortCertificationDetail : effortCertificationDetails) {
            if (workStudyRouteObjectCodes.contains(effortCertificationDetail.getFinancialObjectCode())) {
                return true;
            }
        }

        return false;
    }

    /**
     * Checks system parameter that indicates whether routing to project directors should occur only on lines with
     * federal accounts or all lines.
     *
     * @return detail lines with federal accounts if parameter is true, otherwise all detail lines
     */
    public List<EffortCertificationDetail> getDetailLinesForPDRouting() {
        final boolean federalOnlyRouting = EffortCertificationParameterFinder.getFederalOnlyRouteIndicator();
        if (!federalOnlyRouting) {
            return effortCertificationDetailLines;
        }

        final List<EffortCertificationDetail> federalDetailLines = new ArrayList<>();
        for (final EffortCertificationDetail detail : effortCertificationDetailLines) {
            if (detail.isFederalOrFederalPassThroughIndicator()) {
                federalDetailLines.add(detail);
            }
        }

        return federalDetailLines;
    }

    protected boolean isDoAwardSplit() {
        return isEffortDistributionChanged();
    }

    protected boolean isDoRecreateSplit() {
        return getEffortCertificationDocumentCode();
    }
}
