/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ec.document.authorization;

import org.kuali.kfs.kew.actiontaken.ActionTaken;
import org.kuali.kfs.kim.impl.identity.Person;
import org.kuali.kfs.krad.UserSession;
import org.kuali.kfs.krad.document.Document;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.KRADConstants;
import org.kuali.kfs.sys.document.authorization.FinancialSystemTransactionalDocumentAuthorizerBase;
import org.kuali.kfs.kew.api.KewApiConstants;
import org.kuali.kfs.kew.api.WorkflowDocument;
import org.kuali.kfs.kim.api.identity.PersonService;
import org.kuali.kfs.kim.api.services.KimApiServiceLocator;
import org.kuali.kfs.krad.bo.BusinessObject;

import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Document Authorizer for the Effort Certification document.
 */
public class EffortCertificationDocumentAuthorizer extends FinancialSystemTransactionalDocumentAuthorizerBase {

    /**
     * Overridden to check if document error correction can be allowed here.
     */
    @Override
    public Set<String> getDocumentActions(
            final Document document, final Person user,
            final Set<String> documentActionsFromPresentationController) {
        final Set<String> documentActionsToReturn = super.getDocumentActions(document, user,
                documentActionsFromPresentationController);
        final UserSession userSession = GlobalVariables.getUserSession();
        final String principalId = userSession.getPrincipalId();
        if (document.getDocumentHeader().getWorkflowDocument().isEnroute()) {
            final Set<Person> priorApprovers = getPriorApprovers(document.getDocumentHeader().getWorkflowDocument());
            for (final Person priorApprover : priorApprovers) {
                if (principalId.equals(priorApprover.getPrincipalId())) {
                    documentActionsToReturn.add(KRADConstants.KUALI_ACTION_CAN_EDIT);
                    documentActionsToReturn.add(KRADConstants.KUALI_ACTION_CAN_SAVE);
                }
            }
        }

        return documentActionsToReturn;
    }

    protected Set<Person> getPriorApprovers(final WorkflowDocument workflowDocument) {
        final PersonService personService = KimApiServiceLocator.getPersonService();
        final List<ActionTaken> actionsTaken = workflowDocument.getActionsTaken();
        final Set<String> principalIds = new HashSet<>();
        final Set<Person> persons = new HashSet<>();

        for (final ActionTaken actionTaken : actionsTaken) {
            if (KewApiConstants.ACTION_TAKEN_APPROVED_CD.equals(actionTaken.getActionTaken())) {
                final String principalId = actionTaken.getPrincipalId();
                if (!principalIds.contains(principalId)) {
                    principalIds.add(principalId);
                    persons.add(personService.getPerson(principalId));
                }
            }
        }
        return persons;
    }

    public boolean doPermissionExistsByTemplate(
        final BusinessObject businessObject, final String namespaceCode,
        final String permissionTemplateName, final Map<String, String> permissionDetails) {
        return permissionExistsByTemplate(businessObject, namespaceCode, permissionTemplateName, permissionDetails);
    }

}
