/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ec.document.service.impl;

import org.kuali.kfs.module.ec.businessobject.EffortCertificationReportDefinition;
import org.kuali.kfs.module.ec.dataaccess.EffortCertificationReportDefinitionDao;
import org.kuali.kfs.module.ec.document.service.EffortCertificationAutomaticReportPeriodUpdateService;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;

@Transactional
public class EffortCertificationAutomaticReportPeriodUpdateServiceImpl implements
        EffortCertificationAutomaticReportPeriodUpdateService {

    private EffortCertificationReportDefinitionDao reportDefinitionDao;

    @Override
    public List<EffortCertificationReportDefinition> getAllReportDefinitions() {
        return reportDefinitionDao.getAll();
    }

    @Override
    public boolean isAnOverlappingReportDefinition(final EffortCertificationReportDefinition reportDefinition) {
        final List<EffortCertificationReportDefinition> potentialOverlappingRecords =
                reportDefinitionDao.getAllOtherActiveByType(reportDefinition);

        for (final EffortCertificationReportDefinition potentialOverlappingRecord : potentialOverlappingRecords) {
            if (isOverlapping(potentialOverlappingRecord, reportDefinition)) {
                return true;
            }
        }

        return false;
    }

    public EffortCertificationReportDefinitionDao getEffortCertificationReportDefinitionDao() {
        return reportDefinitionDao;
    }

    @Override
    public void setEffortCertificationReportDefinitionDao(
            final EffortCertificationReportDefinitionDao effortCertificationReportDefinitionDao) {
        reportDefinitionDao = effortCertificationReportDefinitionDao;
    }

    /**
     * Compares existingRecord and newRecord to see if they are overlapping (years and periods).
     *
     * @param existingReportDefinition
     * @param newReportDefinition
     * @return boolean representing whether or not the two report definitions overlap.
     */
    protected boolean isOverlapping(
            final EffortCertificationReportDefinition existingReportDefinition,
            final EffortCertificationReportDefinition newReportDefinition) {
        final Integer existingStartYear = existingReportDefinition.getEffortCertificationReportBeginFiscalYear();
        final String existingStartPeriod = existingReportDefinition.getEffortCertificationReportBeginPeriodCode();

        final Integer existingEndYear = existingReportDefinition.getEffortCertificationReportEndFiscalYear();
        final String existingEndPeriod = existingReportDefinition.getEffortCertificationReportEndPeriodCode();

        if (existingStartYear == null || existingStartPeriod == null || existingEndYear == null
                || existingEndPeriod == null) {
            return false;
        }

        final Integer newStartYear = newReportDefinition.getEffortCertificationReportBeginFiscalYear();
        final String newStartPeriod = newReportDefinition.getEffortCertificationReportBeginPeriodCode();

        final Integer newEndYear = newReportDefinition.getEffortCertificationReportEndFiscalYear();
        final String newEndPeriod = newReportDefinition.getEffortCertificationReportEndPeriodCode();

        final boolean isNewStartPeriodWithin = isPeriodWithinRange(existingStartYear, existingStartPeriod, existingEndYear,
                existingEndPeriod, newStartYear, newStartPeriod);
        if (isNewStartPeriodWithin) {
            return true;
        }

        final boolean isNewEndPeriodWithin = isPeriodWithinRange(existingStartYear, existingStartPeriod, existingEndYear,
                existingEndPeriod, newEndYear, newEndPeriod);
        if (isNewEndPeriodWithin) {
            return true;
        }

        final boolean isExistingStartPeriodWithin = isPeriodWithinRange(newStartYear, newStartPeriod, newEndYear,
                newEndPeriod, existingStartYear, existingStartPeriod);
        if (isExistingStartPeriodWithin) {
            return true;
        }

        return isPeriodWithinRange(newStartYear, newStartPeriod, newEndYear,
                newEndPeriod, existingEndYear, existingEndPeriod);

    }

    protected boolean isPeriodWithinRange(
            final Integer startYear, final String startPeriod, final Integer endYear, final String endPeriod,
            final Integer year, final String period) {
        return comparePeriod(startYear, startPeriod, year, period) <= 0
                && comparePeriod(endYear, endPeriod, year, period) >= 0;
    }

    protected int comparePeriod(final Integer year, final String periodCode, final Integer anotherYear, final String anotherPeriodCode) {
        final String period = year + periodCode;
        final String anotherPeriod = anotherYear + anotherPeriodCode;

        return period.compareTo(anotherPeriod);
    }
}
