/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ec.document.web.struts;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.struts.action.ActionForm;
import org.apache.struts.action.ActionForward;
import org.apache.struts.action.ActionMapping;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.integration.cg.ContractsAndGrantsModuleService;
import org.kuali.kfs.kim.api.KimConstants;
import org.kuali.kfs.kns.service.DocumentHelperService;
import org.kuali.kfs.kns.web.struts.form.KualiDocumentFormBase;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.KRADConstants;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ec.EffortConstants;
import org.kuali.kfs.module.ec.EffortConstants.EffortCertificationEditMode;
import org.kuali.kfs.module.ec.EffortPropertyConstants;
import org.kuali.kfs.module.ec.businessobject.EffortCertificationDetail;
import org.kuali.kfs.module.ec.businessobject.EffortCertificationDetailLineOverride;
import org.kuali.kfs.module.ec.document.EffortCertificationDocument;
import org.kuali.kfs.module.ec.document.authorization.EffortCertificationDocumentAuthorizer;
import org.kuali.kfs.module.ec.document.validation.event.AddDetailLineEvent;
import org.kuali.kfs.module.ec.document.validation.event.CheckDetailLineAmountEvent;
import org.kuali.kfs.module.ec.document.validation.impl.EffortCertificationDocumentRuleUtil;
import org.kuali.kfs.module.ec.service.EffortCertificationDocumentService;
import org.kuali.kfs.module.ec.util.DetailLineGroup;
import org.kuali.kfs.sys.DynamicCollectionComparator;
import org.kuali.kfs.sys.DynamicCollectionComparator.SortOrder;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.ObjectUtil;
import org.kuali.kfs.sys.context.SpringContext;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * This class handles Actions for EffortCertification document approval.
 */
public class CertificationReportAction extends EffortCertificationAction {

    private static final Logger LOG = LogManager.getLogger();

    /**
     * recalculate the detail line
     */
    public ActionForward recalculate(
            final ActionMapping mapping, final ActionForm form, final HttpServletRequest request,
            final HttpServletResponse response) throws Exception {
        final int lineToRecalculateIndex = getLineToDelete(request);

        final EffortCertificationForm effortForm = (EffortCertificationForm) form;
        final EffortCertificationDocument effortDocument = (EffortCertificationDocument) effortForm.getDocument();
        final List<EffortCertificationDetail> detailLines = effortDocument.getEffortCertificationDetailLines();
        final EffortCertificationDetail lineToRecalculate = detailLines.get(lineToRecalculateIndex);
        EffortCertificationDocumentRuleUtil.applyDefaultValues(lineToRecalculate);

        final String errorPathPrefix = KFSPropertyConstants.DOCUMENT + "." +
                                       EffortPropertyConstants.EFFORT_CERTIFICATION_DETAIL_LINES;
        final boolean isValid = invokeRules(new CheckDetailLineAmountEvent("", errorPathPrefix,
                effortDocument, lineToRecalculate));
        if (isValid) {
            final KualiDecimal totalPayrollAmount = effortDocument.getTotalOriginalPayrollAmount();
            lineToRecalculate.recalculatePayrollAmount(totalPayrollAmount);
        }

        return mapping.findForward(KFSConstants.MAPPING_BASIC);
    }

    /**
     * add New Effort Certification Detail Lines
     */
    public ActionForward add(
            final ActionMapping mapping, final ActionForm form, final HttpServletRequest request,
            final HttpServletResponse response) throws Exception {
        final CertificationReportForm certificationReportForm = (CertificationReportForm) form;
        final EffortCertificationDocument effortDocument = (EffortCertificationDocument) certificationReportForm.getDocument();
        final List<EffortCertificationDetail> detailLines = effortDocument.getEffortCertificationDetailLines();
        final EffortCertificationDetail newDetailLine = certificationReportForm.getNewDetailLine();

        // KFSMI-798 - refreshNonUpdatableReferences() used instead of refresh(),
        // EffortCertificationDetail does not have any updatable references
        newDetailLine.refreshNonUpdateableReferences();
        newDetailLine.setPositionNumber(effortDocument.getDefaultPositionNumber());
        newDetailLine.setFinancialObjectCode(effortDocument.getDefaultObjectCode());
        newDetailLine.setNewLineIndicator(true);

        final EffortCertificationDetail workingDetailLine = new EffortCertificationDetail();
        ObjectUtil.buildObject(workingDetailLine, newDetailLine);

        EffortCertificationDocumentRuleUtil.applyDefaultValues(workingDetailLine);

        if (newDetailLine.isAccountExpiredOverride()) {
            updateDetailLineOverrideCode(workingDetailLine);
            updateDetailLineOverrideCode(newDetailLine);
        }

        // check business rules
        final boolean isValid = invokeRules(new AddDetailLineEvent("",
                EffortPropertyConstants.EFFORT_CERTIFICATION_DOCUMENT_NEW_LINE, effortDocument, workingDetailLine));
        if (isValid) {
            if (EffortCertificationDocumentRuleUtil.hasA21SubAccount(workingDetailLine)) {
                EffortCertificationDocumentRuleUtil.updateSourceAccountInformation(workingDetailLine);
            }

            resetPersistedFields(workingDetailLine);
            detailLines.add(workingDetailLine);
            certificationReportForm.setNewDetailLine(certificationReportForm.createNewDetailLine());
        } else {
            EffortCertificationDetailLineOverride.processForOutput(newDetailLine);
        }

        processDetailLineOverrides(detailLines);

        return mapping.findForward(KFSConstants.MAPPING_BASIC);
    }

    /**
     * delete detail line
     */
    public ActionForward delete(
            final ActionMapping mapping, final ActionForm form, final HttpServletRequest request,
            final HttpServletResponse response) throws Exception {
        final int lineToDeleteIndex = getLineToDelete(request);

        final EffortCertificationForm effortForm = (EffortCertificationForm) form;
        final EffortCertificationDocument effortDocument = (EffortCertificationDocument) effortForm.getDocument();
        final List<EffortCertificationDetail> detailLines = effortDocument.getEffortCertificationDetailLines();

        detailLines.remove(lineToDeleteIndex);

        return mapping.findForward(KFSConstants.MAPPING_BASIC);
    }

    /**
     * revert the detail line to the original values
     */
    public ActionForward revert(
            final ActionMapping mapping, final ActionForm form, final HttpServletRequest request,
            final HttpServletResponse response) throws Exception {
        final int lineToRevertIndex = getLineToDelete(request);

        final EffortCertificationForm effortForm = (EffortCertificationForm) form;
        final EffortCertificationDocument effortDocument = (EffortCertificationDocument) effortForm.getDocument();
        final List<EffortCertificationDetail> detailLines = effortDocument.getEffortCertificationDetailLines();

        revertDetailLine(detailLines, lineToRevertIndex);

        return mapping.findForward(KFSConstants.MAPPING_BASIC);
    }

    @Override
    protected void loadDocument(final KualiDocumentFormBase kualiDocumentFormBase) {
        super.loadDocument(kualiDocumentFormBase);

        final CertificationReportForm certificationReportForm = (CertificationReportForm) kualiDocumentFormBase;
        if (isSummarizeDetailLinesRendered(certificationReportForm)) {
            refreshDetailLineGroupMap(certificationReportForm);
        }
    }

    @Override
    public ActionForward refresh(
            final ActionMapping mapping, final ActionForm form, final HttpServletRequest request,
            final HttpServletResponse response) throws Exception {
        final ActionForward actionForward = super.refresh(mapping, form, request, response);

        final CertificationReportForm certificationReportForm = (CertificationReportForm) form;
        for (final EffortCertificationDetail detailLine : certificationReportForm.getDetailLines()) {
            detailLine.refreshNonUpdateableReferences();

            EffortCertificationDocumentRuleUtil.applyDefaultValues(detailLine);
        }

        return actionForward;
    }

    @Override
    public ActionForward execute(
            final ActionMapping mapping, final ActionForm form, final HttpServletRequest request,
            final HttpServletResponse response) throws Exception {
        final CertificationReportForm certificationReportForm = (CertificationReportForm) form;
        if (isSummarizeDetailLinesRendered(certificationReportForm)) {
            updateDetailLinesFromSummaryLines(certificationReportForm);
        }

        final ActionForward actionForward = super.execute(mapping, form, request, response);
        refresh(mapping, form, request, response);

        return actionForward;
    }

    /**
     * sort the detail lines by column
     */
    public ActionForward sortDetailLineByColumn(
            final ActionMapping mapping, final ActionForm form, final HttpServletRequest request,
            final HttpServletResponse response) throws Exception {
        final CertificationReportForm certificationReportForm = (CertificationReportForm) form;

        final String methodToCallAttribute = (String) request.getAttribute(KRADConstants.METHOD_TO_CALL_ATTRIBUTE);
        final String sortMethodName = EffortConstants.SORT_DETAIL_LINE_BY_COLUMN_METHOD_NAME + ".";
        final String sortColumn = StringUtils.substringBetween(methodToCallAttribute, sortMethodName, ".");

        toggleSortOrder(certificationReportForm);
        sortDetailLine(certificationReportForm, certificationReportForm.getDetailLines(), sortColumn);

        if (isSummarizeDetailLinesRendered(certificationReportForm)) {
            sortDetailLine(certificationReportForm, certificationReportForm.getSummarizedDetailLines(), sortColumn);
        }

        return mapping.findForward(KFSConstants.MAPPING_BASIC);
    }

    /**
     * recalculate a detail line in summarized detail lines and make a corresponding update on the underlying detail
     * lines
     */
    public ActionForward recalculateSummarizedDetailLine(
            final ActionMapping mapping, final ActionForm form,
            final HttpServletRequest request, final HttpServletResponse response) throws Exception {
        final CertificationReportForm certificationReportForm = (CertificationReportForm) form;
        final EffortCertificationDocument effortDocument = (EffortCertificationDocument) certificationReportForm.getDocument();
        final KualiDecimal totalPayrollAmount = effortDocument.getTotalOriginalPayrollAmount();

        // recalculate the selected summary line
        final List<EffortCertificationDetail> summarizedDetailLines = certificationReportForm.getSummarizedDetailLines();
        final int lineToRecalculateIndex = getSelectedLine(request);
        final EffortCertificationDetail lineToRecalculate = summarizedDetailLines.get(lineToRecalculateIndex);
        EffortCertificationDocumentRuleUtil.applyDefaultValues(lineToRecalculate);

        final String errorPathPrefix = KFSPropertyConstants.DOCUMENT + "." + EffortPropertyConstants.SUMMARIZED_DETAIL_LINES +
                                       "[" + lineToRecalculateIndex + "]";
        final boolean isValid = invokeRules(new CheckDetailLineAmountEvent("", errorPathPrefix,
                effortDocument, lineToRecalculate));
        if (isValid) {
            lineToRecalculate.recalculatePayrollAmount(totalPayrollAmount);

            final String groupId = lineToRecalculate.getGroupId();
            final List<EffortCertificationDetail> detailLines = certificationReportForm.getDetailLines();
            final List<EffortCertificationDetail> detailLinesToRecalculate = findDetailLinesInGroup(detailLines,
                    groupId);
            for (final EffortCertificationDetail line : detailLinesToRecalculate) {
                ObjectUtil.buildObject(line, lineToRecalculate, EffortConstants.DETAIL_LINES_GROUPING_FIELDS);
            }

            // rebuild the detail line groups from the detail lines of the current document
            final Map<String, DetailLineGroup> detailLineGroupMap =
                    DetailLineGroup.groupDetailLines(certificationReportForm.getDetailLines());
            final DetailLineGroup detailLineGroup = detailLineGroupMap.get(DetailLineGroup.getKeysAsString(lineToRecalculate));
            updateDetailLineGroup(detailLineGroup, lineToRecalculate, totalPayrollAmount);
        }

        return mapping.findForward(KFSConstants.MAPPING_BASIC);
    }

    /**
     * add a new detail line and make a corresponding update on the underlying detail lines
     */
    public ActionForward addSummarizedDetailLine(
            final ActionMapping mapping, final ActionForm form, final HttpServletRequest request,
            final HttpServletResponse response) throws Exception {
        final ActionForward actionForward = add(mapping, form, request, response);

        if (GlobalVariables.getMessageMap().getErrorCount() >= 0) {
            final CertificationReportForm certificationReportForm = (CertificationReportForm) form;
            refreshDetailLineGroupMap(certificationReportForm);
        }

        return actionForward;
    }

    /**
     * delete a detail line from summarized detail lines and make a corresponding update on the underlying detail lines
     */
    public ActionForward deleteSummarizedDetailLine(
            final ActionMapping mapping, final ActionForm form, final HttpServletRequest request,
            final HttpServletResponse response) throws Exception {
        final CertificationReportForm certificationReportForm = (CertificationReportForm) form;

        // remove the selected summary line
        final List<EffortCertificationDetail> summarizedDetailLines = certificationReportForm.getSummarizedDetailLines();
        final int lineToRecalculateIndex = getSelectedLine(request);
        final EffortCertificationDetail lineToDelete = summarizedDetailLines.get(lineToRecalculateIndex);
        summarizedDetailLines.remove(lineToDelete);

        final String groupId = lineToDelete.getGroupId();
        final List<EffortCertificationDetail> detailLines = certificationReportForm.getDetailLines();
        final List<EffortCertificationDetail> detailLinesToDelete = findDetailLinesInGroup(detailLines, groupId);
        detailLines.removeAll(detailLinesToDelete);

        return mapping.findForward(KFSConstants.MAPPING_BASIC);
    }

    /**
     * revert a detail line in summarized detail lines and make a corresponding update on the underlying detail lines
     */
    public ActionForward revertSummarizedDetailLine(
            final ActionMapping mapping, final ActionForm form, final HttpServletRequest request,
            final HttpServletResponse response) throws Exception {
        final CertificationReportForm certificationReportForm = (CertificationReportForm) form;

        final List<EffortCertificationDetail> summarizedDetailLines = certificationReportForm.getSummarizedDetailLines();
        final int lineToRevertIndex = getSelectedLine(request);
        final EffortCertificationDetail lineToRevert = summarizedDetailLines.get(lineToRevertIndex);

        final Map<String, DetailLineGroup> detailLineGroupMap =
                DetailLineGroup.groupDetailLines(certificationReportForm.getDetailLines());
        final DetailLineGroup detailLineGroup = detailLineGroupMap.get(DetailLineGroup.getKeysAsString(lineToRevert));
        final List<EffortCertificationDetail> detailLinesInGroup = detailLineGroup.getDetailLines();

        final List<EffortCertificationDetail> detailLines = certificationReportForm.getDetailLines();
        for (final EffortCertificationDetail detailLine : detailLinesInGroup) {
            revertDetailLine(detailLines, detailLine);
        }

        refreshDetailLineGroupMap(certificationReportForm);

        return mapping.findForward(KFSConstants.MAPPING_BASIC);
    }

    /**
     * find the detail lines belonging to the given group from the given detail lines
     *
     * @param detailLines the given detail lines
     * @param groupId     the given group id
     * @return the detail lines belonging to the given group
     */
    protected List<EffortCertificationDetail> findDetailLinesInGroup(
            final List<EffortCertificationDetail> detailLines,
            final String groupId) {
        final List<EffortCertificationDetail> detailLinesInGroup = new ArrayList<>();

        for (final EffortCertificationDetail line : detailLines) {
            if (StringUtils.equals(groupId, line.getGroupId())) {
                detailLinesInGroup.add(line);
            }
        }

        return detailLinesInGroup;
    }

    /**
     * determine whether the summarized detail lines need to be rendered
     *
     * @param certificationReportForm the action form
     * @return true if the summarized detail lines need to be rendered; otherwise, false
     */
    protected boolean isSummarizeDetailLinesRendered(final CertificationReportForm certificationReportForm) {
        populateAuthorizationFields(certificationReportForm);
        return certificationReportForm.getEditingMode().containsKey(EffortCertificationEditMode.SUMMARY_TAB_ENTRY);
    }

    /**
     * recalculate all detail lines with the information in summarized detail lines
     *
     * @param certificationReportForm the given action form
     */
    protected void recalculateAllDetailLines(final CertificationReportForm certificationReportForm) {
        final Map<String, DetailLineGroup> detailLineGroupMap =
                DetailLineGroup.groupDetailLines(certificationReportForm.getDetailLines());

        final EffortCertificationDocument effortDocument = (EffortCertificationDocument) certificationReportForm.getDocument();
        final KualiDecimal totalPayrollAmount = effortDocument.getTotalOriginalPayrollAmount();

        final List<EffortCertificationDetail> summarizedDetailLines = certificationReportForm.getSummarizedDetailLines();
        for (final EffortCertificationDetail detailLine : summarizedDetailLines) {
            // recalculate the selected summary line
            detailLine.recalculatePayrollAmount(totalPayrollAmount);

            // rebuild the detail line groups from the detail lines of the current document
            final DetailLineGroup detailLineGroup = detailLineGroupMap.get(DetailLineGroup.getKeysAsString(detailLine));
            updateDetailLineGroup(detailLineGroup, detailLine, totalPayrollAmount);
        }
    }

    /**
     * recalculate all detail lines with the information in summarized detail lines
     *
     * @param certificationReportForm the given action form
     */
    protected void updateDetailLinesFromSummaryLines(final CertificationReportForm certificationReportForm) {
        final EffortCertificationDocument effortDocument = (EffortCertificationDocument) certificationReportForm.getDocument();
        final List<EffortCertificationDetail> detailLines = certificationReportForm.getDetailLines();
        final List<EffortCertificationDetail> summarizedDetailLines = certificationReportForm.getSummarizedDetailLines();

        boolean isValid = true;
        int index = 0;
        for (final EffortCertificationDetail detailLine : summarizedDetailLines) {
            EffortCertificationDocumentRuleUtil.applyDefaultValues(detailLine);

            final String errorPathPrefix = KFSPropertyConstants.DOCUMENT + "." +
                                           EffortPropertyConstants.SUMMARIZED_DETAIL_LINES + "[" + index + "]";
            isValid &= invokeRules(new CheckDetailLineAmountEvent("", errorPathPrefix, effortDocument,
                    detailLine));

            ++index;
        }

        if (!isValid) {
            return;
        }

        for (final EffortCertificationDetail detailLine : summarizedDetailLines) {
            if (!detailLine.isNewLineIndicator()) {
                continue;
            }

            if (detailLine.isAccountExpiredOverride()) {
                detailLine.refreshReferenceObject(KFSPropertyConstants.ACCOUNT);
                updateDetailLineOverrideCode(detailLine);
            }

            final List<EffortCertificationDetail> detailLinesToUpdate = findDetailLinesInGroup(detailLines,
                    detailLine.getGroupId());
            for (final EffortCertificationDetail line : detailLinesToUpdate) {
                ObjectUtil.buildObject(line, detailLine, EffortConstants.DETAIL_LINES_GROUPING_FIELDS);

                line.setAccountExpiredOverride(detailLine.isAccountExpiredOverride());
                line.setAccountExpiredOverrideNeeded(detailLine.isAccountExpiredOverrideNeeded());
                line.setOverrideCode(detailLine.getOverrideCode());
            }
        }

        final Map<String, DetailLineGroup> detailLineGroupMap = DetailLineGroup.groupDetailLines(detailLines);
        final KualiDecimal totalPayrollAmount = effortDocument.getTotalOriginalPayrollAmount();
        for (final EffortCertificationDetail detailLine : summarizedDetailLines) {
            // rebuild the detail line groups from the detail lines of the current document
            detailLine.setGroupId(DetailLineGroup.getKeysAsString(detailLine));
            final DetailLineGroup detailLineGroup = detailLineGroupMap.get(DetailLineGroup.getKeysAsString(detailLine));
            updateDetailLineGroup(detailLineGroup, detailLine, totalPayrollAmount);
        }

        processDetailLineOverrides(detailLines);
        processDetailLineOverrides(summarizedDetailLines);
    }

    /**
     * update detail line group with the the information in the given detail line
     *
     * @param detailLineGroup    the given detail line group
     * @param detailLine         the given detail line
     * @param totalPayrollAmount the total payroll amount of the document associating with the detail line group
     */
    protected void updateDetailLineGroup(
            final DetailLineGroup detailLineGroup, final EffortCertificationDetail detailLine,
            final KualiDecimal totalPayrollAmount) {
        final EffortCertificationDetail summaryLine = detailLineGroup.getSummaryDetailLine();
        summaryLine.setEffortCertificationUpdatedOverallPercent(detailLine.getEffortCertificationUpdatedOverallPercent());
        summaryLine.setEffortCertificationPayrollAmount(detailLine.getEffortCertificationPayrollAmount());

        detailLineGroup.updateDetailLineEffortPercent();
        detailLineGroup.updateDetailLinePayrollAmount();
    }

    /**
     * Toggles the sort order between ascending and descending. If the current order is ascending, then the sort order
     * will be set to descending, and vice versa.
     */
    protected void toggleSortOrder(final CertificationReportForm certificationReportForm) {
        if (SortOrder.ASC.name().equals(certificationReportForm.getSortOrder())) {
            certificationReportForm.setSortOrder(SortOrder.DESC.name());
        } else {
            certificationReportForm.setSortOrder(SortOrder.ASC.name());
        }
    }

    /**
     * sort the detail lines based on the values of the sort order and sort column
     */
    protected void sortDetailLine(
            final CertificationReportForm certificationReportForm,
            final List<EffortCertificationDetail> detailLines, final String... sortColumn) {
        final String sortOrder = certificationReportForm.getSortOrder();
        DynamicCollectionComparator.sort(detailLines, SortOrder.valueOf(sortOrder), sortColumn);
    }

    /**
     * rebuild the detail line group map from the detail lines of the current document
     */
    protected Map<String, DetailLineGroup> refreshDetailLineGroupMap(final CertificationReportForm certificationReportForm) {
        LOG.debug("refreshDetailLineGroupMap() started");

        final List<EffortCertificationDetail> summarizedDetailLines = certificationReportForm.getSummarizedDetailLines();
        if (summarizedDetailLines == null) {
            final EffortCertificationDocument effortCertificationDocument =
                    (EffortCertificationDocument) certificationReportForm.getDocument();
            effortCertificationDocument.setSummarizedDetailLines(new ArrayList<>());
        }
        summarizedDetailLines.clear();

        final Map<String, DetailLineGroup> detailLineGroupMap =
                DetailLineGroup.groupDetailLines(certificationReportForm.getDetailLines());

        for (final String key : detailLineGroupMap.keySet()) {
            final EffortCertificationDetail summaryLine = detailLineGroupMap.get(key).getSummaryDetailLine();
            summarizedDetailLines.add(summaryLine);
        }

        return detailLineGroupMap;
    }

    /**
     * find the given detail line from the given collection of detail lines and revert it
     */
    protected void revertDetailLine(
            final List<EffortCertificationDetail> detailLines,
            final EffortCertificationDetail lineToRevert) {
        final int lineToRevertIndex = detailLines.lastIndexOf(lineToRevert);

        revertDetailLine(detailLines, lineToRevertIndex);
    }

    /**
     * revert the detail line in the specified position
     */
    protected void revertDetailLine(final List<EffortCertificationDetail> detailLines, final int lineToRevertIndex) {
        final EffortCertificationDetail lineToRevert = detailLines.get(lineToRevertIndex);

        final BusinessObjectService businessObjectService = SpringContext.getBean(BusinessObjectService.class);
        final EffortCertificationDetail revertedLine =
                (EffortCertificationDetail) businessObjectService.retrieve(lineToRevert);
        resetPersistedFields(revertedLine);

        detailLines.remove(lineToRevertIndex);
        detailLines.add(lineToRevertIndex, revertedLine);
    }

    /**
     * reset the persisted fields of the given detail line
     */
    protected void resetPersistedFields(final EffortCertificationDetail detailLine) {
        final int persistedEffortPercent = detailLine.getEffortCertificationUpdatedOverallPercent();
        detailLine.setPersistedEffortPercent(persistedEffortPercent);

        final BigDecimal persistedPayrollAmount = detailLine.getEffortCertificationPayrollAmount().bigDecimalValue();
        detailLine.setPersistedPayrollAmount(new KualiDecimal(persistedPayrollAmount));
    }

    @Override
    public ActionForward approve(
            final ActionMapping mapping, final ActionForm form, final HttpServletRequest request,
            final HttpServletResponse response) throws Exception {
        final KualiDocumentFormBase kualiDocumentFormBase = (KualiDocumentFormBase) form;
        final EffortCertificationDocument effortDocument = (EffortCertificationDocument) kualiDocumentFormBase.getDocument();

        final EffortCertificationDocumentService effortCertificationDocumentService =
                SpringContext.getBean(EffortCertificationDocumentService.class);
        effortCertificationDocumentService.addRouteLooping(effortDocument);

        return super.approve(mapping, kualiDocumentFormBase, request, response);
    }

    @Override
    public ActionForward insertBONote(
            final ActionMapping mapping, final ActionForm form, final HttpServletRequest request,
            final HttpServletResponse response) throws Exception {
        final KualiDocumentFormBase kualiDocumentFormBase = (KualiDocumentFormBase) form;
        kualiDocumentFormBase.getNewNote().setNewCollectionRecord(true);

        return super.insertBONote(mapping, form, request, response);
    }

    /**
     * Do one additional check on Use Transactional for Summary Tab
     */
    @Override
    public void populateAuthorizationFields(final KualiDocumentFormBase kualiDocumentFormBase) {
        //call super for authorization fields
        super.populateAuthorizationFields(kualiDocumentFormBase);

        final CertificationReportForm certificationReportForm = (CertificationReportForm) kualiDocumentFormBase;

        //get document authorizer
        final EffortCertificationDocumentAuthorizer certReportDocAuthorizer =
                (EffortCertificationDocumentAuthorizer) SpringContext.getBean(DocumentHelperService.class)
                        .getDocumentAuthorizer(EffortConstants.EffortDocumentTypes.EFFORT_CERTIFICATION_DOCUMENT);

        //grab document
        final EffortCertificationDocument effortDocument = (EffortCertificationDocument) kualiDocumentFormBase.getDocument();

        //get principal id
        final String currentPrincipalId = GlobalVariables.getUserSession().getPrincipalId();

        //set additional details for summary tab
        final Map<String, String> additionalPermissionDetails = new HashMap<>();
        additionalPermissionDetails.put(KimConstants.AttributeConstants.EDIT_MODE,
                EffortCertificationEditMode.SUMMARY_TAB_ENTRY);

        //get list of chart/account numbers
        final List<EffortCertificationDetail> summarizedDetailLines = createSummarizedDetailLines(certificationReportForm);

        //set additional role qualifiers
        final Map<String, String> additionalRoleQualifiers = new HashMap<>();
        final String proposalNumber = getProposalNumberForProjectDirector(currentPrincipalId, summarizedDetailLines);

        //set proposal number if found
        if (StringUtils.isNotBlank(proposalNumber)) {
            additionalRoleQualifiers.put(KFSPropertyConstants.PROPOSAL, proposalNumber);

            //re-check summary permission
            if (certReportDocAuthorizer.doPermissionExistsByTemplate(
                effortDocument,
                KFSConstants.CoreModuleNamespaces.KFS,
                KimConstants.PermissionTemplateNames.USE_TRANSACTIONAL_DOCUMENT,
                additionalPermissionDetails)
                && !certReportDocAuthorizer.isAuthorizedByTemplate(
                effortDocument,
                KFSConstants.CoreModuleNamespaces.KFS,
                KimConstants.PermissionTemplateNames.USE_TRANSACTIONAL_DOCUMENT,
                currentPrincipalId,
                additionalPermissionDetails,
                additionalRoleQualifiers)) {

                kualiDocumentFormBase.getEditingMode().put(EffortCertificationEditMode.SUMMARY_TAB_ENTRY,
                        KRADConstants.KUALI_DEFAULT_TRUE_VALUE);
            }
        }
    }

    /**
     * Retrieves a proposal number for a project director from a list of awards pulled by chart/account number.
     *
     * @param projectDirectorId
     * @return
     */
    protected String getProposalNumberForProjectDirector(
            final String projectDirectorId,
            final List<EffortCertificationDetail> summarizedDetailLines) {
        String proposalNumber = null;

        for (final EffortCertificationDetail line : summarizedDetailLines) {
            proposalNumber = SpringContext.getBean(ContractsAndGrantsModuleService.class)
                    .getProposalNumberForAccountAndProjectDirector(line.getChartOfAccountsCode(),
                            line.getAccountNumber(), projectDirectorId);

            //if found a proposal number, break and return
            if (StringUtils.isNotEmpty(proposalNumber)) {
                break;
            }
        }

        return proposalNumber;
    }

    protected List<EffortCertificationDetail> createSummarizedDetailLines(
            final CertificationReportForm certificationReportForm) {
        List<EffortCertificationDetail> summarizedDetailLines = certificationReportForm.getSummarizedDetailLines();

        if (ObjectUtils.isNull(summarizedDetailLines) || summarizedDetailLines.isEmpty()) {
            if (ObjectUtils.isNotNull(certificationReportForm.getDetailLines())
                    && !certificationReportForm.getDetailLines().isEmpty()) {
                summarizedDetailLines = certificationReportForm.getDetailLines();
            }
        }

        return summarizedDetailLines;
    }

    /**
     * Transforms the summarized effort detail lines into a chart/account number map
     *
     * @param summaryDetail
     * @return
     */
    protected Map<String, String> convertSummarizedDetailToChartAccountMap(
            final List<EffortCertificationDetail> summaryDetail) {
        final Map<String, String> chartAccountMap = new HashMap<>();

        for (final EffortCertificationDetail line : summaryDetail) {
            chartAccountMap.put(line.getChartOfAccountsCode(), line.getAccountNumber());
        }

        return chartAccountMap;
    }
}
