/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2021 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ec.businessobject;

import org.kuali.kfs.coa.businessobject.AccountingPeriod;
import org.kuali.kfs.integration.ec.EffortCertificationReport;
import org.kuali.kfs.krad.bo.PersistableBusinessObjectBase;
import org.kuali.kfs.module.ec.EffortPropertyConstants;
import org.kuali.kfs.module.ec.util.AccountingPeriodMonth;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.businessobject.FiscalYearBasedBusinessObject;
import org.kuali.kfs.sys.businessobject.SystemOptions;
import org.kuali.kfs.core.api.mo.common.active.MutableInactivatable;

import java.sql.Date;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

/**
 * Business Object for the Effort Certification Report Definition Table.
 */
public class EffortCertificationReportDefinition extends PersistableBusinessObjectBase implements
        EffortCertificationReport, MutableInactivatable, FiscalYearBasedBusinessObject {

    private Integer universityFiscalYear;
    private String effortCertificationReportNumber;
    private String effortCertificationReportPeriodTitle;
    private String effortCertificationReportPeriodStatusCode;
    private Integer expenseTransferFiscalYear;
    private String expenseTransferFiscalPeriodCode;
    private String effortCertificationReportTypeCode;
    private Date effortCertificationReportReturnDate;
    private Integer effortCertificationReportBeginFiscalYear;
    private String effortCertificationReportBeginPeriodCode;
    private Integer effortCertificationReportEndFiscalYear;
    private String effortCertificationReportEndPeriodCode;
    private boolean active;

    private SystemOptions options;
    private SystemOptions reportBeginFiscalYear;
    private SystemOptions reportEndFiscalYear;
    private AccountingPeriod reportBeginPeriod;
    private AccountingPeriod reportEndPeriod;
    private SystemOptions expenseTransferYear;
    private AccountingPeriod expenseTransferFiscalPeriod;
    private EffortCertificationPeriodStatusCode effortCertificationPeriodStatusCode;
    private EffortCertificationReportType effortCertificationReportType;
    private Collection<EffortCertificationReportPosition> effortCertificationReportPositions;

    private Map<Integer, Set<String>> reportPeriods;

    public EffortCertificationReportDefinition() {
        super();
        effortCertificationReportPositions = new ArrayList<>();
    }

    public Integer getUniversityFiscalYear() {
        return universityFiscalYear;
    }

    public void setUniversityFiscalYear(Integer universityFiscalYear) {
        this.universityFiscalYear = universityFiscalYear;
    }

    public String getEffortCertificationReportNumber() {
        return effortCertificationReportNumber;
    }

    public void setEffortCertificationReportNumber(String effortCertificationReportNumber) {
        this.effortCertificationReportNumber = effortCertificationReportNumber;
    }

    public String getEffortCertificationReportPeriodTitle() {
        return effortCertificationReportPeriodTitle;
    }

    public void setEffortCertificationReportPeriodTitle(String effortCertificationReportPeriodTitle) {
        this.effortCertificationReportPeriodTitle = effortCertificationReportPeriodTitle;
    }

    public String getEffortCertificationReportPeriodStatusCode() {
        return effortCertificationReportPeriodStatusCode;
    }

    public void setEffortCertificationReportPeriodStatusCode(String effortCertificationReportPeriodStatusCode) {
        this.effortCertificationReportPeriodStatusCode = effortCertificationReportPeriodStatusCode;
    }

    public Integer getExpenseTransferFiscalYear() {
        return expenseTransferFiscalYear;
    }

    public void setExpenseTransferFiscalYear(Integer expenseTransferFiscalYear) {
        this.expenseTransferFiscalYear = expenseTransferFiscalYear;
    }

    public String getExpenseTransferFiscalPeriodCode() {
        return expenseTransferFiscalPeriodCode;
    }

    public void setExpenseTransferFiscalPeriodCode(String expenseTransferFiscalPeriodCode) {
        this.expenseTransferFiscalPeriodCode = expenseTransferFiscalPeriodCode;
    }

    public String getEffortCertificationReportTypeCode() {
        return effortCertificationReportTypeCode;
    }

    public void setEffortCertificationReportTypeCode(String effortCertificationReportTypeCode) {
        this.effortCertificationReportTypeCode = effortCertificationReportTypeCode;
    }

    public Date getEffortCertificationReportReturnDate() {
        return effortCertificationReportReturnDate;
    }

    public void setEffortCertificationReportReturnDate(Date effortCertificationReportReturnDate) {
        this.effortCertificationReportReturnDate = effortCertificationReportReturnDate;
    }

    public Integer getEffortCertificationReportBeginFiscalYear() {
        return effortCertificationReportBeginFiscalYear;
    }

    public void setEffortCertificationReportBeginFiscalYear(Integer effortCertificationReportBeginFiscalYear) {
        this.effortCertificationReportBeginFiscalYear = effortCertificationReportBeginFiscalYear;
    }

    public String getEffortCertificationReportBeginPeriodCode() {
        return effortCertificationReportBeginPeriodCode;
    }

    public void setEffortCertificationReportBeginPeriodCode(String effortCertificationReportBeginPeriodCode) {
        this.effortCertificationReportBeginPeriodCode = effortCertificationReportBeginPeriodCode;
    }

    public Integer getEffortCertificationReportEndFiscalYear() {
        return effortCertificationReportEndFiscalYear;
    }

    public void setEffortCertificationReportEndFiscalYear(Integer effortCertificationReportEndFiscalYear) {
        this.effortCertificationReportEndFiscalYear = effortCertificationReportEndFiscalYear;
    }

    public String getEffortCertificationReportEndPeriodCode() {
        return effortCertificationReportEndPeriodCode;
    }

    public void setEffortCertificationReportEndPeriodCode(String effortCertificationReportEndPeriodCode) {
        this.effortCertificationReportEndPeriodCode = effortCertificationReportEndPeriodCode;
    }

    public AccountingPeriod getExpenseTransferFiscalPeriod() {
        return expenseTransferFiscalPeriod;
    }

    @Deprecated
    public void setExpenseTransferFiscalPeriod(AccountingPeriod expenseTransferFiscalPeriod) {
        this.expenseTransferFiscalPeriod = expenseTransferFiscalPeriod;
    }

    public SystemOptions getExpenseTransferYear() {
        return expenseTransferYear;
    }

    @Deprecated
    public void setExpenseTransferYear(SystemOptions expenseTransferYear) {
        this.expenseTransferYear = expenseTransferYear;
    }

    public SystemOptions getReportBeginFiscalYear() {
        return reportBeginFiscalYear;
    }

    @Deprecated
    public void setReportBeginFiscalYear(SystemOptions reportBeginFiscalYear) {
        this.reportBeginFiscalYear = reportBeginFiscalYear;
    }

    public SystemOptions getReportEndFiscalYear() {
        return reportEndFiscalYear;
    }

    @Deprecated
    public void setReportEndFiscalYear(SystemOptions reportEndFiscalYear) {
        this.reportEndFiscalYear = reportEndFiscalYear;
    }

    public AccountingPeriod getReportBeginPeriod() {
        return reportBeginPeriod;
    }

    @Deprecated
    public void setReportBeginPeriod(AccountingPeriod reportBeginPeriod) {
        this.reportBeginPeriod = reportBeginPeriod;
    }

    public AccountingPeriod getReportEndPeriod() {
        return reportEndPeriod;
    }

    @Deprecated
    public void setReportEndPeriod(AccountingPeriod reportEndPeriod) {
        this.reportEndPeriod = reportEndPeriod;
    }

    public EffortCertificationPeriodStatusCode getEffortCertificationPeriodStatusCode() {
        return effortCertificationPeriodStatusCode;
    }

    @Deprecated
    public void setEffortCertificationPeriodStatusCode(
            EffortCertificationPeriodStatusCode effortCertificationPeriodStatusCode) {
        this.effortCertificationPeriodStatusCode = effortCertificationPeriodStatusCode;
    }

    public EffortCertificationReportType getEffortCertificationReportType() {
        return effortCertificationReportType;
    }

    @Deprecated
    public void setEffortCertificationReportType(EffortCertificationReportType effortCertificationReportType) {
        this.effortCertificationReportType = effortCertificationReportType;
    }

    public Collection<EffortCertificationReportPosition> getEffortCertificationReportPositions() {
        return effortCertificationReportPositions;
    }

    public void setEffortCertificationReportPositions(
            Collection<EffortCertificationReportPosition> effortCertificationReportPositions) {
        this.effortCertificationReportPositions = effortCertificationReportPositions;
    }

    public boolean isActive() {
        return active;
    }

    public void setActive(boolean active) {
        this.active = active;
    }

    public SystemOptions getOptions() {
        return options;
    }

    @Deprecated
    public void setOptions(SystemOptions options) {
        this.options = options;
    }

    /**
     * build a primary key field map for the current report definition
     *
     * @return a primary key field map for the current report definition
     */
    public Map<String, String> buildKeyMapForCurrentReportDefinition() {
        return buildKeyMap(this.getUniversityFiscalYear(), this.getEffortCertificationReportNumber());
    }

    /**
     * build a primary key field map for a report definition from the given values
     *
     * @param universityFiscalYear the given fiscal year
     * @param reportNumber         the given report number
     * @return a primary key field map for a report definition
     */
    public static Map<String, String> buildKeyMap(Integer universityFiscalYear, String reportNumber) {
        Map<String, String> primaryKeyMap = new HashMap<>();

        String stringFiscalYear = universityFiscalYear == null ? "" : universityFiscalYear.toString();
        primaryKeyMap.put(KFSPropertyConstants.UNIVERSITY_FISCAL_YEAR, stringFiscalYear);
        primaryKeyMap.put(EffortPropertyConstants.EFFORT_CERTIFICATION_REPORT_NUMBER, reportNumber);

        return primaryKeyMap;
    }

    /**
     * build all report periods map covered by the specified report definition
     */
    public void buildReportPeriods() {
        Integer beginYear = this.getEffortCertificationReportBeginFiscalYear();
        String beginPeriodCode = this.getEffortCertificationReportBeginPeriodCode();
        Integer endYear = this.getEffortCertificationReportEndFiscalYear();
        String endPeriodCode = this.getEffortCertificationReportEndPeriodCode();

        this.setReportPeriods(AccountingPeriodMonth.findAccountingPeriodsBetween(beginYear, beginPeriodCode, endYear,
                endPeriodCode));
    }

    public Map<Integer, Set<String>> getReportPeriods() {
        if (reportPeriods == null) {
            this.buildReportPeriods();
        }
        return reportPeriods;
    }

    public void setReportPeriods(Map<Integer, Set<String>> reportPeriods) {
        this.reportPeriods = reportPeriods;
    }
}
