/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2021 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ec.document.service.impl;

import org.kuali.kfs.module.ec.businessobject.EffortCertificationReportDefinition;
import org.kuali.kfs.module.ec.dataaccess.EffortCertificationReportDefinitionDao;
import org.kuali.kfs.module.ec.document.service.EffortCertificationAutomaticReportPeriodUpdateService;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;

@Transactional
public class EffortCertificationAutomaticReportPeriodUpdateServiceImpl implements
        EffortCertificationAutomaticReportPeriodUpdateService {

    private EffortCertificationReportDefinitionDao reportDefinitionDao;

    public List<EffortCertificationReportDefinition> getAllReportDefinitions() {
        return this.reportDefinitionDao.getAll();
    }

    public boolean isAnOverlappingReportDefinition(EffortCertificationReportDefinition reportDefinition) {
        List<EffortCertificationReportDefinition> potentialOverlappingRecords =
                reportDefinitionDao.getAllOtherActiveByType(reportDefinition);

        for (EffortCertificationReportDefinition potentialOverlappingRecord : potentialOverlappingRecords) {
            if (isOverlapping(potentialOverlappingRecord, reportDefinition)) {
                return true;
            }
        }

        return false;
    }

    public EffortCertificationReportDefinitionDao getEffortCertificationReportDefinitionDao() {
        return reportDefinitionDao;
    }

    public void setEffortCertificationReportDefinitionDao(
            EffortCertificationReportDefinitionDao effortCertificationReportDefinitionDao) {
        this.reportDefinitionDao = effortCertificationReportDefinitionDao;
    }

    /**
     * Compares existingRecord and newRecord to see if they are overlapping (years and periods).
     *
     * @param existingReportDefinition
     * @param newReportDefinition
     * @return boolean representing whether or not the two report definitions overlap.
     */
    protected boolean isOverlapping(EffortCertificationReportDefinition existingReportDefinition,
            EffortCertificationReportDefinition newReportDefinition) {
        Integer existingStartYear = existingReportDefinition.getEffortCertificationReportBeginFiscalYear();
        String existingStartPeriod = existingReportDefinition.getEffortCertificationReportBeginPeriodCode();

        Integer existingEndYear = existingReportDefinition.getEffortCertificationReportEndFiscalYear();
        String existingEndPeriod = existingReportDefinition.getEffortCertificationReportEndPeriodCode();

        if (existingStartYear == null || existingStartPeriod == null || existingEndYear == null
                || existingEndPeriod == null) {
            return false;
        }

        Integer newStartYear = newReportDefinition.getEffortCertificationReportBeginFiscalYear();
        String newStartPeriod = newReportDefinition.getEffortCertificationReportBeginPeriodCode();

        Integer newEndYear = newReportDefinition.getEffortCertificationReportEndFiscalYear();
        String newEndPeriod = newReportDefinition.getEffortCertificationReportEndPeriodCode();

        boolean isNewStartPeriodWithin = isPeriodWithinRange(existingStartYear, existingStartPeriod, existingEndYear,
                existingEndPeriod, newStartYear, newStartPeriod);
        if (isNewStartPeriodWithin) {
            return true;
        }

        boolean isNewEndPeriodWithin = isPeriodWithinRange(existingStartYear, existingStartPeriod, existingEndYear,
                existingEndPeriod, newEndYear, newEndPeriod);
        if (isNewEndPeriodWithin) {
            return true;
        }

        boolean isExistingStartPeriodWithin = isPeriodWithinRange(newStartYear, newStartPeriod, newEndYear,
                newEndPeriod, existingStartYear, existingStartPeriod);
        if (isExistingStartPeriodWithin) {
            return true;
        }

        return isPeriodWithinRange(newStartYear, newStartPeriod, newEndYear,
                newEndPeriod, existingEndYear, existingEndPeriod);

    }

    protected boolean isPeriodWithinRange(Integer startYear, String startPeriod, Integer endYear, String endPeriod,
            Integer year, String period) {
        return comparePeriod(startYear, startPeriod, year, period) <= 0
                && comparePeriod(endYear, endPeriod, year, period) >= 0;
    }

    protected int comparePeriod(Integer year, String periodCode, Integer anotherYear, String anotherPeriodCode) {
        String period = year + periodCode;
        String anotherPeriod = anotherYear + anotherPeriodCode;

        return period.compareTo(anotherPeriod);
    }
}
