/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2021 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ec.util;

import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.module.ec.EffortParameterConstants;
import org.kuali.kfs.module.ec.batch.EffortCertificationCreateStep;
import org.kuali.kfs.module.ec.batch.EffortCertificationExtractStep;
import org.kuali.kfs.module.ec.document.EffortCertificationDocument;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.sys.service.impl.KfsParameterConstants;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 * A convenient utility that can delegate the calling client to retrieve system parameters of effort certification
 * module.
 */
public final class EffortCertificationParameterFinder {

    private static ParameterService parameterService = SpringContext.getBean(ParameterService.class);

    /**
     * Private Constructor since this is a util class that should never be instantiated.
     */
    private EffortCertificationParameterFinder() {
    }

    /**
     * @return the run indicator setup in system parameters
     */
    public static boolean getRunIndicator() {
        return parameterService.getParameterValueAsBoolean(EffortCertificationExtractStep.class,
                EffortParameterConstants.RUN_IND);
    }

    /**
     * @return the federal agency type codes setup in system parameters
     */
    public static Collection<String> getFederalAgencyTypeCodes() {
        return parameterService.getParameterValuesAsString(KfsParameterConstants.FINANCIAL_SYSTEM_ALL.class,
                KfsParameterConstants.FEDERAL_AGENCY_TYPE);
    }

    public static boolean getFederalOnlyBalanceIndicator() {
        return parameterService.getParameterValueAsBoolean(EffortCertificationExtractStep.class,
                EffortParameterConstants.FEDERAL_ONLY_BALANCE_IND);
    }

    public static boolean getFederalOnlyRouteIndicator() {
        return parameterService.getParameterValueAsBoolean(EffortCertificationDocument.class,
                EffortParameterConstants.FEDERAL_ONLY_ROUTE_IND);
    }

    public static List<String> getFederalOnlyBalanceIndicatorAsString() {
        List<String> indicatorList = new ArrayList<>();
        indicatorList.add(Boolean.toString(getFederalOnlyBalanceIndicator()));
        return indicatorList;
    }

    /**
     * @return the account type codes setup in system parameters
     */
    public static Collection<String> getAccountTypeCodes() {
        return parameterService.getParameterValuesAsString(EffortCertificationExtractStep.class,
                EffortParameterConstants.ACCOUNT_TYPE_CODE_BALANCE_SELECT);
    }

    /**
     * @return the report fiscal year setup in system parameters
     */
    public static Integer getExtractReportFiscalYear() {
        return Integer.valueOf(parameterService.getParameterValueAsString(EffortCertificationExtractStep.class,
                EffortParameterConstants.RUN_FISCAL_YEAR));
    }

    /**
     * @return the report number setup in system parameters
     */
    public static String getExtractReportNumber() {
        return parameterService.getParameterValueAsString(EffortCertificationExtractStep.class,
                EffortParameterConstants.RUN_REPORT_NUMBER);
    }

    /**
     * @return the report fiscal year setup in system parameters
     */
    public static Integer getCreateReportFiscalYear() {
        return Integer.valueOf(parameterService.getParameterValueAsString(EffortCertificationCreateStep.class,
                EffortParameterConstants.CREATE_FISCAL_YEAR));
    }

    /**
     * @return the report number setup in system parameters
     */
    public static String getCreateReportNumber() {
        return parameterService.getParameterValueAsString(EffortCertificationCreateStep.class,
                EffortParameterConstants.CREATE_REPORT_NUMBER);
    }
}
