/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2021 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.external.kc.service.impl;

import org.kuali.kfs.kns.datadictionary.BusinessObjectEntry;
import org.kuali.kfs.krad.util.KRADConstants;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.external.kc.service.ExternalizableBusinessObjectService;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.sys.service.impl.KfsModuleServiceImpl;
import org.kuali.kfs.core.api.config.property.ConfigurationService;
import org.kuali.kfs.krad.bo.ExternalizableBusinessObject;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class KcKfsModuleServiceImpl extends KfsModuleServiceImpl {

    protected ConfigurationService configurationService;

    @Override
    public <T extends ExternalizableBusinessObject> T getExternalizableBusinessObject(Class<T> businessObjectClass,
            Map<String, Object> fieldValues) {
        Class<? extends ExternalizableBusinessObject> implementationClass =
                getExternalizableBusinessObjectImplementation(businessObjectClass);
        return (T) getExternalizableBusinessObjectService(implementationClass).findByPrimaryKey(fieldValues);
    }

    @Override
    public <T extends ExternalizableBusinessObject> List<T> getExternalizableBusinessObjectsList(
            Class<T> businessObjectClass, Map<String, Object> fieldValues) {
        Class<? extends ExternalizableBusinessObject> implementationClass =
                getExternalizableBusinessObjectImplementation(businessObjectClass);
        return (List<T>) getExternalizableBusinessObjectService(implementationClass).findMatching(fieldValues);
    }

    @Override
    public boolean isExternal(Class boClass) {
        return isExternalizable(boClass);
    }

    /**
     * @param clazz
     * @return the business object service via the class to service mapping provided in the module configuration.
     */
    private ExternalizableBusinessObjectService getExternalizableBusinessObjectService(Class clazz) {
        String serviceName;
        ExternalizableBusinessObjectService eboService = null;

        Map<Class, String> externalizableBusinessObjectServices =
                ((KcFinancialSystemModuleConfiguration) getModuleConfiguration())
                        .getExternalizableBusinessObjectServiceImplementations();

        if (ObjectUtils.isNotNull(externalizableBusinessObjectServices) && ObjectUtils.isNotNull(clazz)) {
            serviceName = externalizableBusinessObjectServices.get(clazz);
            eboService = (ExternalizableBusinessObjectService) SpringContext.getService(serviceName);
        }

        return eboService;
    }

    /**
     * @return primary key fields from the DataDictionary entries for the object.
     */
    @Override
    public List listPrimaryKeyFieldNames(Class businessObjectInterfaceClass) {
        Class clazz = getExternalizableBusinessObjectImplementation(businessObjectInterfaceClass);
        final BusinessObjectEntry boEntry = getBusinessObjectDictionaryService()
                .getBusinessObjectEntry(clazz.getName());
        if (boEntry == null) {
            return null;
        }
        return boEntry.getPrimaryKeys();
    }

    /**
     * Changing the base url to KC url
     */
    @Override
    protected String getInquiryUrl(Class inquiryBusinessObjectClass) {
        String baseUrl = configurationService.getPropertyValueAsString(KFSConstants.KC_APPLICATION_URL_KEY);
        if (!baseUrl.endsWith("/")) {
            baseUrl = baseUrl + "/";
        }
        return baseUrl + KRADConstants.INQUIRY_ACTION;
    }

    /**
     * Mapping the kfs classes and parameters over to KC equivalents
     * @return
     */
    @Override
    protected Map<String, String> getUrlParameters(String businessObjectClassAttribute,
            Map<String, String[]> parameters) {
        Map<String, String> urlParameters = new HashMap<>();
        String paramNameToConvert;
        Map<String, String> kfsToKcInquiryUrlParameterMapping =
                ((KcFinancialSystemModuleConfiguration) getModuleConfiguration())
                        .getKfsToKcInquiryUrlParameterMapping();
        Map<String, String> kfsToKcInquiryUrlClassMapping =
                ((KcFinancialSystemModuleConfiguration) getModuleConfiguration()).getKfsToKcInquiryUrlClassMapping();

        for (String paramName : parameters.keySet()) {
            String parameterName = paramName;
            String[] parameterValues = parameters.get(paramName);

            if (parameterValues.length > 0) {
                //attempt to convert parameter name if necessary
                paramNameToConvert = businessObjectClassAttribute + "." + paramName;
                if (kfsToKcInquiryUrlParameterMapping.containsKey(paramNameToConvert)) {
                    parameterName = kfsToKcInquiryUrlParameterMapping.get(paramNameToConvert);
                }
                urlParameters.put(parameterName, parameterValues[0]);
            }
        }

        urlParameters.put(KRADConstants.BUSINESS_OBJECT_CLASS_ATTRIBUTE,
                kfsToKcInquiryUrlClassMapping.get(businessObjectClassAttribute));
        urlParameters.put(KRADConstants.DISPATCH_REQUEST_PARAMETER,
            KRADConstants.CONTINUE_WITH_INQUIRY_METHOD_TO_CALL);
        return urlParameters;
    }

    public ConfigurationService getConfigurationService() {
        return configurationService;
    }

    public void setConfigurationService(ConfigurationService configurationService) {
        this.configurationService = configurationService;
    }
}
