/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2018 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.coreservice.framework.parameter;

import org.kuali.kfs.coreservice.api.parameter.Parameter;

import java.util.Collection;

/**
 * This service is used by krad to interact with {@link Parameter Parameters}.
 * <p>
 * <p>
 * Generally krad client applications will want to use this service since it contains many convenient methods.
 * <p>
 * <p>
 * This service can be viewed a convenient wrapper around the
 * {@link org.kuali.rice.coreservice.api.parameter.ParameterRepositoryService}. Please see
 * {@link org.kuali.rice.coreservice.api.parameter.ParameterRepositoryService} for details on the behavior of this
 * service.
 */
public interface ParameterService {

    /**
     * This will create a {@link Parameter} exactly like the parameter passed in.
     *
     * @see org.kuali.rice.coreservice.api.parameter.ParameterRepositoryService#createParameter(org.kuali.rice.coreservice.api.parameter.Parameter)
     */
    Parameter createParameter(Parameter parameter);

    /**
     * This will update a {@link Parameter}.
     *
     * @see org.kuali.rice.coreservice.api.parameter.ParameterRepositoryService#updateParameter(org.kuali.rice.coreservice.api.parameter.Parameter)
     */
    Parameter updateParameter(Parameter parameter);

    /**
     * This method checks if a parameter exists.  It will never return null.
     * <p>
     * <p>
     * The parameter key is constructed from the following:
     * <ul>
     * <li>namespace code: from a {@link ParameterConstants.NAMESPACE} annotation on the componentClass</li>
     * <li>component code: from a {@link ParameterConstants.COMPONENT} annotation on the componentClass</li>
     * <li>parameter name: from the passed in parameter name</li>
     * <li>application id: from the client configuration of the service implementation</li>
     * </ul>
     * <p>
     * <p>
     * If the parameter does not exist under the application code, then this method will check if the parameter
     * exists under the default rice application id and will return that parameter.
     *
     * @param componentClass the class with the namespace & component annotations
     * @param parameterName  the parameter name
     * @return true or false
     * @throws IllegalArgumentException if any arguments are null
     * @throws IllegalStateException    if the application id is not configured correctly
     */
    Boolean parameterExists(Class<?> componentClass, String parameterName);

    /**
     * This method checks if a parameter exists.  It will never return null.
     * <p>
     * <p>
     * The parameter key is constructed from the following:
     * <ul>
     * <li>namespace code: from the passed in namespace code</li>
     * <li>component code: from the passed in component code</li>
     * <li>parameter name: from the passed in parameter name</li>
     * <li>application id: from the client configuration of the service implementation</li>
     * </ul>
     * <p>
     * <p>
     * If the parameter does not exist under the application code, then this method will check if the parameter
     * exists under the default rice application id and will return that parameter.
     *
     * @param namespaceCode the namespace code
     * @param componentCode the component code
     * @param parameterName the parameter name
     * @return true or false
     * @throws IllegalArgumentException if any arguments are null
     * @throws IllegalStateException    if the application id is not configured correctly
     */
    Boolean parameterExists(String namespaceCode, String componentCode, String parameterName);

    /**
     * Retrieves a parameter's string value.
     * <p>
     * <p>
     * The parameter key is constructed from the following:
     * <ul>
     * <li>namespace code: from a {@link ParameterConstants.NAMESPACE} annotation on the componentClass</li>
     * <li>component code: from a {@link ParameterConstants.COMPONENT} annotation on the componentClass</li>
     * <li>parameter name: from the passed in parameter name</li>
     * <li>application id: from the client configuration of the service implementation</li>
     * </ul>
     *
     * @param componentClass the class with the namespace & component annotations
     * @param parameterName  the parameter name
     * @return string value or null
     * @throws IllegalArgumentException if any arguments are null
     * @throws IllegalStateException    if the application id is not configured correctly
     * @see org.kuali.rice.coreservice.api.parameter.ParameterRepositoryService#getParameterValueAsString(org.kuali.rice.coreservice.api.parameter.ParameterKey)
     */
    String getParameterValueAsString(Class<?> componentClass, String parameterName);

    /**
     * Retrieves a parameter's string value.  If the parameter is not found the default value will be returned.
     * <p>
     * <p>
     * The parameter key is constructed from the following:
     * <ul>
     * <li>namespace code: from a {@link ParameterConstants.NAMESPACE} annotation on the componentClass</li>
     * <li>component code: from a {@link ParameterConstants.COMPONENT} annotation on the componentClass</li>
     * <li>parameter name: from the passed in parameter name</li>
     * <li>application id: from the client configuration of the service implementation</li>
     * </ul>
     *
     * @param componentClass the class with the namespace & component annotations
     * @param parameterName  the parameter name
     * @param defaultValue   the value to return is the parameter does not exist.  Can be any string value including null
     * @return string value or null
     * @throws IllegalArgumentException if any arguments are null
     * @throws IllegalStateException    if the application id is not configured correctly
     * @see org.kuali.rice.coreservice.api.parameter.ParameterRepositoryService#getParameterValueAsString(org.kuali.rice.coreservice.api.parameter.ParameterKey)
     */
    String getParameterValueAsString(Class<?> componentClass, String parameterName, String defaultValue);


    /**
     * Retrieves a parameter's string value.
     * <p>
     * <p>
     * The parameter key is constructed from the following:
     * <ul>
     * <li>namespace code: from the passed in namespace code</li>
     * <li>component code: from the passed in component code</li>
     * <li>parameter name: from the passed in parameter name</li>
     * <li>application id: from the client configuration of the service implementation</li>
     * </ul>
     *
     * @param namespaceCode the namespace code
     * @param componentCode the component code
     * @param parameterName the parameter name
     * @return string value or null
     * @throws IllegalArgumentException if any arguments are null
     * @throws IllegalStateException    if the application id is not configured correctly
     * @see org.kuali.rice.coreservice.api.parameter.ParameterRepositoryService#getParameterValueAsString(org.kuali.rice.coreservice.api.parameter.ParameterKey)
     */
    String getParameterValueAsString(String namespaceCode, String componentCode, String parameterName);

    /**
     * Retrieves a parameter's string value.  If the parameter is not found the default value will be returned.
     * <p>
     * <p>
     * The parameter key is constructed from the following:
     * <ul>
     * <li>namespace code: from the passed in namespace code</li>
     * <li>component code: from the passed in component code</li>
     * <li>parameter name: from the passed in parameter name</li>
     * <li>application id: from the client configuration of the service implementation</li>
     * </ul>
     *
     * @param namespaceCode the namespace code
     * @param componentCode the component code
     * @param parameterName the parameter name
     * @param defaultValue  the value to return is the parameter does not exist.  Can be any string value including null
     * @return string value or null
     * @throws IllegalArgumentException if any arguments are null
     * @throws IllegalStateException    if the application id is not configured correctly
     * @see org.kuali.rice.coreservice.api.parameter.ParameterRepositoryService#getParameterValueAsString(org.kuali.rice.coreservice.api.parameter.ParameterKey)
     */
    String getParameterValueAsString(String namespaceCode, String componentCode, String parameterName, String defaultValue);

    /**
     * Retrieves a parameter's boolean value.
     * <p>
     * <p>
     * The parameter key is constructed from the following:
     * <ul>
     * <li>namespace code: from a {@link ParameterConstants.NAMESPACE} annotation on the componentClass</li>
     * <li>component code: from a {@link ParameterConstants.COMPONENT} annotation on the componentClass</li>
     * <li>parameter name: from the passed in parameter name</li>
     * <li>application id: from the client configuration of the service implementation</li>
     * </ul>
     *
     * @param componentClass the class with the namespace & component annotations
     * @param parameterName  the parameter name
     * @return true, false, null
     * @throws IllegalArgumentException if any arguments are null
     * @throws IllegalStateException    if the application id is not configured correctly
     * @see org.kuali.rice.coreservice.api.parameter.ParameterRepositoryService#getParameterValueAsBoolean(org.kuali.rice.coreservice.api.parameter.ParameterKey)
     */
    Boolean getParameterValueAsBoolean(Class<?> componentClass, String parameterName);

    /**
     * Retrieves a parameter's boolean value.  If the parameter is not found the default value will be returned.
     * <p>
     * <p>
     * The parameter key is constructed from the following:
     * <ul>
     * <li>namespace code: from a {@link ParameterConstants.NAMESPACE} annotation on the componentClass</li>
     * <li>component code: from a {@link ParameterConstants.COMPONENT} annotation on the componentClass</li>
     * <li>parameter name: from the passed in parameter name</li>
     * <li>application id: from the client configuration of the service implementation</li>
     * </ul>
     *
     * @param componentClass the class with the namespace & component annotations
     * @param parameterName  the parameter name
     * @param defaultValue   the value to return is the parameter does not exist. Can be any Boolean value including null
     * @return true, false, or the defaultValue
     * @throws IllegalArgumentException if any arguments are null
     * @throws IllegalStateException    if the application id is not configured correctly
     * @see org.kuali.rice.coreservice.api.parameter.ParameterRepositoryService#getParameterValueAsBoolean(org.kuali.rice.coreservice.api.parameter.ParameterKey)
     */
    Boolean getParameterValueAsBoolean(Class<?> componentClass, String parameterName, Boolean defaultValue);

    /**
     * Retrieves a parameter's boolean value.
     * <p>
     * <p>
     * The parameter key is constructed from the following:
     * <ul>
     * <li>namespace code: from the passed in namespace code</li>
     * <li>component code: from the passed in component code</li>
     * <li>parameter name: from the passed in parameter name</li>
     * <li>application id: from the client configuration of the service implementation</li>
     * </ul>
     *
     * @param namespaceCode the namespace code
     * @param componentCode the component code
     * @param parameterName the parameter name
     * @return true, false, null
     * @throws IllegalArgumentException if any arguments are null
     * @throws IllegalStateException    if the application id is not configured correctly
     * @see org.kuali.rice.coreservice.api.parameter.ParameterRepositoryService#getParameterValueAsBoolean(org.kuali.rice.coreservice.api.parameter.ParameterKey)
     */
    Boolean getParameterValueAsBoolean(String namespaceCode, String componentCode, String parameterName);

    /**
     * Retrieves a parameter's boolean value.  If the parameter is not found the default value will be returned.
     * <p>
     * <p>
     * The parameter key is constructed from the following:
     * <ul>
     * <li>namespace code: from the passed in namespace code</li>
     * <li>component code: from the passed in component code</li>
     * <li>parameter name: from the passed in parameter name</li>
     * <li>application id: from the client configuration of the service implementation</li>
     * </ul>
     *
     * @param namespaceCode the namespace code
     * @param componentCode the component code
     * @param parameterName the parameter name
     * @param defaultValue  the value to return is the parameter does not exist. Can be any Boolean value including null
     * @return true, false, or the defaultValue
     * @throws IllegalArgumentException if any arguments are null
     * @throws IllegalStateException    if the application id is not configured correctly
     * @see org.kuali.rice.coreservice.api.parameter.ParameterRepositoryService#getParameterValueAsBoolean(org.kuali.rice.coreservice.api.parameter.ParameterKey)
     */
    Boolean getParameterValueAsBoolean(String namespaceCode, String componentCode, String parameterName, Boolean defaultValue);

    /**
     * Retrieves a parameter.
     * <p>
     * <p>
     * The parameter key is constructed from the following:
     * <ul>
     * <li>namespace code: from a {@link ParameterConstants.NAMESPACE} annotation on the componentClass</li>
     * <li>component code: from a {@link ParameterConstants.COMPONENT} annotation on the componentClass</li>
     * <li>parameter name: from the passed in parameter name</li>
     * <li>application id: from the client configuration of the service implementation</li>
     * </ul>
     *
     * @param componentClass the class with the namespace & component annotations
     * @param parameterName  the parameter name
     * @return true or false
     * @throws IllegalArgumentException if any arguments are null
     * @throws IllegalStateException    if the application id is not configured correctly
     * @see org.kuali.rice.coreservice.api.parameter.ParameterRepositoryService#getParameter(org.kuali.rice.coreservice.api.parameter.ParameterKey)
     */
    Parameter getParameter(Class<?> componentClass, String parameterName);

    /**
     * Retrieves a parameter.
     * <p>
     * <p>
     * The parameter key is constructed from the following:
     * <ul>
     * <li>namespace code: from the passed in namespace code</li>
     * <li>component code: from the passed in component code</li>
     * <li>parameter name: from the passed in parameter name</li>
     * <li>application id: from the client configuration of the service implementation</li>
     * </ul>
     *
     * @param namespaceCode the namespace code
     * @param componentCode the component code
     * @param parameterName the parameter name
     * @return true or false
     * @throws IllegalArgumentException if any arguments are null
     * @throws IllegalStateException    if the application id is not configured correctly
     * @see org.kuali.rice.coreservice.api.parameter.ParameterRepositoryService#getParameter(org.kuali.rice.coreservice.api.parameter.ParameterKey)
     */
    Parameter getParameter(String namespaceCode, String componentCode, String parameterName);

    /**
     * Retrieves a parameter's string values where a parameter contains 0 or more values.
     * <p>
     * <p>
     * The parameter key is constructed from the following:
     * <ul>
     * <li>namespace code: from a {@link ParameterConstants.NAMESPACE} annotation on the componentClass</li>
     * <li>component code: from a {@link ParameterConstants.COMPONENT} annotation on the componentClass</li>
     * <li>parameter name: from the passed in parameter name</li>
     * <li>application id: from the client configuration of the service implementation</li>
     * </ul>
     *
     * @param componentClass the class with the namespace & component annotations
     * @param parameterName  the parameter name
     * @return string values or empty Collection
     * @throws IllegalArgumentException if any arguments are null
     * @throws IllegalStateException    if the application id is not configured correctly
     * @see org.kuali.rice.coreservice.api.parameter.ParameterRepositoryService#getParameterValuesAsString(org.kuali.rice.coreservice.api.parameter.ParameterKey)
     */
    Collection<String> getParameterValuesAsString(Class<?> componentClass, String parameterName);

    /**
     * Retrieves a parameter's string values where a parameter contains 0 or more values.
     * <p>
     * <p>
     * The parameter key is constructed from the following:
     * <ul>
     * <li>namespace code: from a {@link ParameterConstants.NAMESPACE} annotation on the componentClass</li>
     * <li>component code: from a {@link ParameterConstants.COMPONENT} annotation on the componentClass</li>
     * <li>parameter name: from the passed in parameter name</li>
     * <li>application id: from the client configuration of the service implementation</li>
     * </ul>
     *
     * @param namespaceCode the namespace code
     * @param componentCode the component code
     * @param parameterName the parameter name
     * @return string values or empty Collection
     * @throws IllegalArgumentException if any arguments are null
     * @throws IllegalStateException    if the application id is not configured correctly
     * @see org.kuali.rice.coreservice.api.parameter.ParameterRepositoryService#getParameterValuesAsString(org.kuali.rice.coreservice.api.parameter.ParameterKey)
     */
    Collection<String> getParameterValuesAsString(String namespaceCode, String componentCode, String parameterName);

    /**
     * Retrieves a subParameter's string value.
     * <p>
     * <p>
     * The parameter key is constructed from the following:
     * <ul>
     * <li>namespace code: from a {@link ParameterConstants.NAMESPACE} annotation on the componentClass</li>
     * <li>component code: from a {@link ParameterConstants.COMPONENT} annotation on the componentClass</li>
     * <li>parameter name: from the passed in parameter name</li>
     * <li>application id: from the client configuration of the service implementation</li>
     * </ul>
     *
     * @param componentClass   the class with the namespace & component annotations
     * @param parameterName    the parameter name
     * @param subParameterName the subParameter name
     * @return string value or null
     * @throws IllegalArgumentException if any arguments are null
     * @throws IllegalStateException    if the application id is not configured correctly
     * @see org.kuali.rice.coreservice.api.parameter.ParameterRepositoryService#getSubParameterValueAsString(org.kuali.rice.coreservice.api.parameter.ParameterKey, String)
     */
    String getSubParameterValueAsString(Class<?> componentClass, String parameterName, String subParameterName);

    /**
     * Retrieves a subParameter's string value.
     * <p>
     * <p>
     * The parameter key is constructed from the following:
     * <ul>
     * <li>namespace code: from the passed in namespace code</li>
     * <li>component code: from the passed in component code</li>
     * <li>parameter name: from the passed in parameter name</li>
     * <li>application id: from the client configuration of the service implementation</li>
     * </ul>
     *
     * @param namespaceCode    the namespace code
     * @param componentCode    the component code
     * @param parameterName    the parameter name
     * @param subParameterName the subParameter name
     * @return string value or null
     * @throws IllegalArgumentException if any arguments are null
     * @throws IllegalStateException    if the application id is not configured correctly
     * @see org.kuali.rice.coreservice.api.parameter.ParameterRepositoryService#getSubParameterValueAsString(org.kuali.rice.coreservice.api.parameter.ParameterKey, String)
     */
    String getSubParameterValueAsString(String namespaceCode, String componentCode, String parameterName, String subParameterName);

    /**
     * Retrieves a subParameter's string values where a subParameter contains 0 or more values.
     * <p>
     * <p>
     * The parameter key is constructed from the following:
     * <ul>
     * <li>namespace code: from a {@link ParameterConstants.NAMESPACE} annotation on the componentClass</li>
     * <li>component code: from a {@link ParameterConstants.COMPONENT} annotation on the componentClass</li>
     * <li>parameter name: from the passed in parameter name</li>
     * <li>application id: from the client configuration of the service implementation</li>
     * </ul>
     *
     * @param componentClass   the class with the namespace & component annotations
     * @param parameterName    the parameter name
     * @param subParameterName the subParameter name
     * @return string values or empty Collection
     * @throws IllegalArgumentException if any arguments are null
     * @throws IllegalStateException    if the application id is not configured correctly
     * @see org.kuali.rice.coreservice.api.parameter.ParameterRepositoryService#getSubParameterValuesAsString(org.kuali.rice.coreservice.api.parameter.ParameterKey, String)
     */
    Collection<String> getSubParameterValuesAsString(Class<?> componentClass, String parameterName, String subParameterName);

    /**
     * Retrieves a subParameter's string values where a subParameter contains 0 or more values.
     * <p>
     * <p>
     * The parameter key is constructed from the following:
     * <ul>
     * <li>namespace code: from a {@link ParameterConstants.NAMESPACE} annotation on the componentClass</li>
     * <li>component code: from a {@link ParameterConstants.COMPONENT} annotation on the componentClass</li>
     * <li>parameter name: from the passed in parameter name</li>
     * <li>application id: from the client configuration of the service implementation</li>
     * </ul>
     *
     * @param namespaceCode    the namespace code
     * @param componentCode    the component code
     * @param parameterName    the parameter name
     * @param subParameterName the subParameter name
     * @return string values or empty Collection
     * @throws IllegalArgumentException if any arguments are null
     * @throws IllegalStateException    if the application id is not configured correctly
     * @see org.kuali.rice.coreservice.api.parameter.ParameterRepositoryService#getSubParameterValuesAsString(org.kuali.rice.coreservice.api.parameter.ParameterKey, String)
     */
    Collection<String> getSubParameterValuesAsString(String namespaceCode, String componentCode, String parameterName, String subParameterName);
}
