/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2018 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.kns.datadictionary;

import org.kuali.kfs.krad.exception.AuthorizationException;

import java.io.File;

/**
 * Entity to manage business object operations, such as "download", "delete", "edit", etc. Authorization for the
 * operations is the responsibility of the implementation and not of the caller.
 */
public interface BusinessObjectAdminService {

    /**
     * @return {@code true} if this instance supports downloading business objects via this interface; {@code false}
     *         otherwise.
     */
    boolean supportsDownload();

    /**
     * @param id identifier for the instance of the business object that should be retrieved.
     * @return File containing the business object identified by the provided {@code id}.
     * @throws AuthorizationException if the user does not have authorization to delete the business object instance.
     * @throws EntityNotFoundException if a problem is encountered while attempting to access the information needed.
     */
    File download(String id) throws AuthorizationException, EntityNotFoundException;

    /**
     * @return {@code true} if this instance supports deleting business objects via this interface; {@code false}
     *         otherwise.
     */
    boolean supportsDelete();

    /**
     * @param id identifier for the instance of the business object that should be deleted.
     * @return {@code true} if the delete operation was successful; {@code false} otherwise.
     * @throws AuthorizationException if the user does not have authorization to delete the business object instance.
     * @throws EntityNotFoundException if a problem is encountered while attempting to access the information needed.
     */
    boolean delete(String id) throws AuthorizationException, EntityNotFoundException;
}
