/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2018 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.krad.uif.component;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;

/**
 * Component security is used to flag permissions that exist in KIM for various component state (like edit and view).
 * <p>
 * <p>
 * In addition, properties such as additional role and permission details can be configured to use when checking the
 * KIM permissions.
 * <p>
 * <p>
 * Security subclasses exist adding on flags appropriate for that component.
 */
public class ComponentSecurity extends ConfigurableBase implements Serializable {

    private static final long serialVersionUID = 726347449984853891L;

    private boolean editAuthz;
    private boolean viewAuthz;

    private String namespaceAttribute;
    private String componentAttribute;
    private String idAttribute;

    private Map<String, String> additionalPermissionDetails;
    private Map<String, String> additionalRoleQualifiers;

    public ComponentSecurity() {
        editAuthz = false;
        viewAuthz = false;

        additionalPermissionDetails = new HashMap<>();
        additionalRoleQualifiers = new HashMap<>();
    }

    /**
     * @return boolean true if the component has edit authorization and KIM should be consulted, false if not.
     */
    public boolean isEditAuthz() {
        return editAuthz;
    }

    /**
     * @param editAuthz the edit authorization flag to set.
     */
    public void setEditAuthz(boolean editAuthz) {
        this.editAuthz = editAuthz;
    }

    /**
     * @return boolean true if the component has view authorization and KIM should be consulted, false if not.
     */
    public boolean isViewAuthz() {
        return viewAuthz;
    }

    /**
     * @param viewAuthz the view authorization flag to set.
     */
    public void setViewAuthz(boolean viewAuthz) {
        this.viewAuthz = viewAuthz;
    }

    /**
     * When the namespace code is a detail for a permission check, this property can be configured to override the
     * namespace derived by the system
     *
     * @return String namespace code that should be sent as permission detail when doing a permission check on this
     *         field.
     */
    public String getNamespaceAttribute() {
        return namespaceAttribute;
    }

    /**
     * @param namespaceAttribute the namespace code to use for details that should be set.
     */
    public void setNamespaceAttribute(String namespaceAttribute) {
        this.namespaceAttribute = namespaceAttribute;
    }

    /**
     * When the component code is a detail for a permission check, this property can be configured to override the
     * component code derived by the system
     *
     * @return component code that should be sent as permission detail when doing a permission check on this field.
     */
    public String getComponentAttribute() {
        return componentAttribute;
    }

    /**
     * @param componentAttribute the code to use for details to set.
     */
    public void setComponentAttribute(String componentAttribute) {
        this.componentAttribute = componentAttribute;
    }

    /**
     * By default they system will send the component id as a permission detail, this property can be configured to
     * send a different id for the permission check.
     *
     * @return id that should be sent as permission detail when doing a permission check on this field.
     */
    public String getIdAttribute() {
        return idAttribute;
    }

    /**
     * @param idAttribute the id to use for details to set.
     */
    public void setIdAttribute(String idAttribute) {
        this.idAttribute = idAttribute;
    }

    /**
     * Any details given here that will override details with the same key that were derived by the system.
     *
     * @return Map<String, String> of key value pairs that should be added as permission details when doing KIM
     *         permission checks for this component
     */
    public Map<String, String> getAdditionalPermissionDetails() {
        return additionalPermissionDetails;
    }

    /**
     * @param additionalPermissionDetails the map of additional permission details to set.
     */
    public void setAdditionalPermissionDetails(Map<String, String> additionalPermissionDetails) {
        this.additionalPermissionDetails = additionalPermissionDetails;
    }

    /**
     * Any qualifiers given here that will override qualifiers with the same key that were derived by the system.
     *
     * @return Map<String, String> of key value pairs that should be added as role qualifiers when doing KIM permission
     *         checks for this component.
     */
    public Map<String, String> getAdditionalRoleQualifiers() {
        return additionalRoleQualifiers;
    }

    /**
     * @param additionalRoleQualifiers the map of additional role qualifiers to set.
     */
    public void setAdditionalRoleQualifiers(Map<String, String> additionalRoleQualifiers) {
        this.additionalRoleQualifiers = additionalRoleQualifiers;
    }

    @Override
    protected void finalize() throws Throwable {
        try {
            idAttribute = null;
            componentAttribute = null;
            namespaceAttribute = null;
            additionalRoleQualifiers = null;
            additionalPermissionDetails = null;
        } finally {
            // don't call super.finalize() in attempt to avoid loop between maps.
        }
    }
}
