/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2018 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.krad.uif.container;

import org.kuali.kfs.krad.uif.component.Component;
import org.kuali.kfs.krad.uif.field.ErrorsField;
import org.kuali.kfs.krad.uif.field.HeaderField;
import org.kuali.kfs.krad.uif.field.MessageField;
import org.kuali.kfs.krad.uif.layout.LayoutManager;
import org.kuali.kfs.krad.uif.widget.Help;

import java.util.List;
import java.util.Set;

/**
 * Type of component that contains a collection of other components. All templates for <code> Container</code>
 * components must use a <code>LayoutManager</code> to render the contained components.
 * <p>
 * Each container has the following parts in addition to the contained components:
 * <ul>
 * <li><code>HeaderField</code></li>
 * <li>Summary <code>MessageField</code></li>
 * <li>Help component</li>
 * <li>Errors container</li>
 * <li>Footer <code>Group</code></li>
 * </ul>
 * Container implementations are free to add additional content as needed.
 *
 * @see Component
 */
public interface Container extends Component {

    /**
     * Contained components are rendered within the section template by calling the associated <code>LayoutManager</code>
     *
     * @return List component instances that are held by the container
     */
    List<? extends Component> getItems();

    /**
     * @param items list of components to set in container
     */
    void setItems(List<? extends Component> items);

    /**
     * If an empty or null list is returned, it is assumed the container supports all components. The returned set will
     * be used by dictionary validators and allows renders to make assumptions about the contained components
     *
     * @return Set component classes that may be placed into the container
     */
    Set<Class<? extends Component>> getSupportedComponents();

    /**
     * The template associated with the layout manager will be invoked passing in the List of components from the
     * container. This list is exported under the attribute name 'items'
     *
     * @return LayoutManager instance that should be used to layout the components in the container
     */
    LayoutManager getLayoutManager();

    /**
     * @param layoutManager layout manager to set.
     */
    void setLayoutManager(LayoutManager layoutManager);

    /**
     * Header fields are generally rendered at the beginning of the container to indicate a grouping, although this is
     * determined by the template associated with the container. The actual rendering configuration (style and so on) is
     * configured within the HeaderField instance
     * <p>
     * Header is only rendered if <code>Container#isRenderHeader</code> is true and getHeader() is not null
     *
     * @return HeaderField instance associated with the container or Null
     */
    HeaderField getHeader();

    /**
     * @param header header field to set.
     */
    void setHeader(HeaderField header);

    /**
     * The footer is usually rendered at the end of the container. Often this is a place to put actions (buttons) for
     * the container.
     * <p>
     * Footer is only rendered if <code>Container#isRenderFooter</code> is true and getFooter is not null
     *
     * @return Group footer instance associated with the container or Null
     */
    Group getFooter();

    /**
     * @param footer footer to set.
     */
    void setFooter(Group footer);

    /**
     * Text is encapsulated in a <code>MessageField</code> that contains rendering configuration.
     * <p>
     * Summary <code>MessageField</code> only rendered if this methods does not return null
     *
     * @return MessageField instance containg text for the container that provides a summary description or instructions
     *         or Null
     */
    MessageField getInstructionalMessageField();

    /**
     * @param summaryMessageField summary message field to set.
     */
    void setInstructionalMessageField(MessageField summaryMessageField);

    /**
     * Containers can collect the errors for the contained component and display either all the messages or counts. This
     * <code>Field</code> is used to render those messages. Styling and other configuration is done through the
     * <code>ErrorsField</code>
     *
     * @return ErrorsField holding the container errors
     */
    ErrorsField getErrorsField();

    /**
     * @param errorsField errors field to set.
     */
    void setErrorsField(ErrorsField errorsField);

    /**
     * External help information can be configured for the container. The <code>Help</code> object can the configuration
     * for rendering a link to that help information.
     *
     * @return Help configuration object for container
     */
    Help getHelp();

    /**
     * @param help help content to set.
     */
    void setHelp(Help help);

    /**
     * @return true if the container is used to display a group of fields that is visually a single field - this has an
     *         effect on where errors will show up for these fields.
     */
    boolean isFieldContainer();

    /**
     * @param fieldContainer the fieldContainer to set
     */
    void setFieldContainer(boolean fieldContainer);

}
