/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2018 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.service.impl;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.gl.GeneralLedgerConstants;
import org.kuali.kfs.gl.businessobject.OriginEntryStatistics;
import org.kuali.kfs.gl.service.OriginEntryGroupService;
import org.kuali.kfs.gl.service.impl.OriginEntryGroupServiceImpl;
import org.kuali.kfs.gl.service.impl.OriginEntryServiceImpl;
import org.kuali.kfs.module.ld.businessobject.LaborOriginEntry;
import org.kuali.kfs.module.ld.service.LaborOriginEntryService;
import org.kuali.kfs.module.ld.util.LaborOriginEntryFileIterator;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.Message;
import org.kuali.rice.core.api.datetime.DateTimeService;
import org.kuali.rice.core.api.util.type.KualiDecimal;
import org.springframework.transaction.annotation.Transactional;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * Service implementation of LaborOriginEntryService.
 */
@Transactional
public class LaborOriginEntryServiceImpl extends OriginEntryGroupServiceImpl implements LaborOriginEntryService {
    private static final Logger LOG = LogManager.getLogger(OriginEntryServiceImpl.class);

    private OriginEntryGroupService originEntryGroupService;
    private DateTimeService dateTimeService;

    private String batchFileDirectoryName;

    public OriginEntryStatistics getStatistics(String fileName) {
        LOG.debug("getStatistics() started");
        OriginEntryStatistics oes = new OriginEntryStatistics();
        KualiDecimal totalCredit = KualiDecimal.ZERO;
        KualiDecimal totalDebit = KualiDecimal.ZERO;
        Integer rowCount = 0;
        FileReader INPUT_FILE = null;
        BufferedReader INPUT_FILE_br;
        try {
            INPUT_FILE = new FileReader(fileName);
        } catch (FileNotFoundException e) {
            throw new RuntimeException(e);
        }
        Collection<LaborOriginEntry> entryCollection = new ArrayList();
        INPUT_FILE_br = new BufferedReader(INPUT_FILE);

        try {
            String currentLine = INPUT_FILE_br.readLine();
            while (currentLine != null) {
                KualiDecimal amount = KualiDecimal.ZERO;
                if (!currentLine.substring(109, 126).trim().equals(GeneralLedgerConstants.EMPTY_CODE)) {
                    try {
                        amount = new KualiDecimal(currentLine.substring(109, 126).trim());

                        // TODO:- Check with FIS (Row count should be all rows?)
                        rowCount++;
                    } catch (NumberFormatException e) {
                    }
                } else {
                    amount = KualiDecimal.ZERO;
                }
                String debitOrCreditCode = currentLine.substring(126, 127);

                if (debitOrCreditCode.equals(KFSConstants.GL_CREDIT_CODE)) {
                    totalCredit.add(amount);
                } else if (debitOrCreditCode.equals(KFSConstants.GL_DEBIT_CODE)) {
                    totalDebit.add(amount);
                }
                currentLine = INPUT_FILE_br.readLine();
            }
            INPUT_FILE_br.close();
        } catch (IOException e) {
            // FIXME: do whatever should be done here
            throw new RuntimeException(e);
        }

        oes.setCreditTotalAmount(totalCredit);
        oes.setDebitTotalAmount(totalDebit);
        oes.setRowCount(rowCount);

        return oes;
    }

    public Map getEntriesByGroupIdWithPath(String fileNameWithPath, List<LaborOriginEntry> originEntryList) {
        FileReader INPUT_GLE_FILE = null;
        BufferedReader INPUT_GLE_FILE_br;
        try {
            INPUT_GLE_FILE = new FileReader(fileNameWithPath);
        } catch (FileNotFoundException e) {
            throw new RuntimeException(e);
        }
        INPUT_GLE_FILE_br = new BufferedReader(INPUT_GLE_FILE);

        boolean loadError = false;
        Map returnMessageMap = getEntriesByBufferedReader(INPUT_GLE_FILE_br, originEntryList);

        try {
            INPUT_GLE_FILE_br.close();
            INPUT_GLE_FILE.close();
        } catch (IOException e) {
            throw new RuntimeException(e);
        }

        return returnMessageMap;
    }

    public Map getEntriesByBufferedReader(BufferedReader inputBufferedReader, List<LaborOriginEntry> originEntryList) {
        String line;
        int lineNumber = 0;
        Map returnMessageMap = new HashMap();
        try {
            List<Message> tmperrors = new ArrayList();
            while ((line = inputBufferedReader.readLine()) != null) {
                lineNumber++;
                LaborOriginEntry laborOriginEntry = new LaborOriginEntry();
                tmperrors = laborOriginEntry.setFromTextFileForBatch(line, lineNumber);
                laborOriginEntry.setEntryId(lineNumber);
                if (tmperrors.size() > 0) {
                    returnMessageMap.put(new Integer(lineNumber), tmperrors);
                } else {
                    originEntryList.add(laborOriginEntry);
                }
            }
        } catch (IOException e) {
            throw new RuntimeException(e);
        }

        return returnMessageMap;
    }

    public Integer getGroupCount(String fileNameWithPath) {
        File file = new File(fileNameWithPath);
        Iterator<LaborOriginEntry> fileIterator = new LaborOriginEntryFileIterator(file);
        int count = 0;

        while (fileIterator.hasNext()) {
            count++;
            fileIterator.next();
        }
        return count;
    }

    public void setDateTimeService(DateTimeService dateTimeService) {
        this.dateTimeService = dateTimeService;
    }

    public void setOriginEntryGroupService(OriginEntryGroupService originEntryGroupService) {
        this.originEntryGroupService = originEntryGroupService;
    }

    public void setBatchFileDirectoryName(String batchFileDirectoryName) {
        this.batchFileDirectoryName = batchFileDirectoryName;
    }
}
