/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.batch.service.impl;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.coa.businessobject.BalanceType;
import org.kuali.kfs.gl.batch.service.PostTransaction;
import org.kuali.kfs.gl.businessobject.Transaction;
import org.kuali.kfs.module.ld.LaborConstants;
import org.kuali.kfs.module.ld.batch.service.LaborAccountingCycleCachingService;
import org.kuali.kfs.module.ld.businessobject.LaborGeneralLedgerEntry;
import org.kuali.kfs.module.ld.service.LaborGeneralLedgerEntryService;
import org.kuali.kfs.module.ld.util.DebitCreditUtil;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.ObjectUtil;
import org.kuali.kfs.sys.service.ReportWriterService;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.springframework.transaction.annotation.Transactional;

import java.sql.Date;

/**
 * This class is used to post a transaction into labor GL entry table
 */
@Transactional
public class LaborGLLedgerEntryPoster implements PostTransaction {

    private LaborGeneralLedgerEntryService laborGeneralLedgerEntryService;
    private LaborAccountingCycleCachingService laborAccountingCycleCachingService;

    @Override
    public String post(
            final Transaction transaction, final int mode, final java.util.Date postDate,
            final ReportWriterService posterReportWriterService) {
        final String operationType = KFSConstants.OperationType.INSERT;
        final LaborGeneralLedgerEntry laborGeneralLedgerEntry = new LaborGeneralLedgerEntry();
        ObjectUtil.buildObject(laborGeneralLedgerEntry, transaction);

        laborGeneralLedgerEntry.setTransactionDate(new Date(postDate.getTime()));

        final BalanceType balanceType = getBalanceType(transaction.getFinancialBalanceTypeCode());
        if (balanceType.isFinancialOffsetGenerationIndicator()) {
            laborGeneralLedgerEntry.setTransactionDebitCreditCode(getDebitCreditCode(transaction));
        } else {
            laborGeneralLedgerEntry.setTransactionDebitCreditCode(" ");
        }
        laborGeneralLedgerEntry.setTransactionLedgerEntryAmount(getTransactionAmount(transaction));

        final String encumbranceUpdateCode = getEncumbranceUpdateCode(transaction);
        if (StringUtils.isNotEmpty(encumbranceUpdateCode)) {
            laborGeneralLedgerEntry.setTransactionEncumbranceUpdateCode(encumbranceUpdateCode);
        }

        final Integer sequenceNumber = laborGeneralLedgerEntryService.getMaxSequenceNumber(laborGeneralLedgerEntry) + 1;
        laborGeneralLedgerEntry.setTransactionLedgerEntrySequenceNumber(sequenceNumber);
        try {
            laborGeneralLedgerEntryService.save(laborGeneralLedgerEntry);
        } catch (final Exception e) {
            throw new RuntimeException(e);
        }
        return operationType;
    }

    protected BalanceType getBalanceType(final String balanceTypeCode) {
        return laborAccountingCycleCachingService.getBalanceType(balanceTypeCode);
    }

    public void setLaborAccountingCycleCachingService(
            final LaborAccountingCycleCachingService laborAccountingCycleCachingService) {
        this.laborAccountingCycleCachingService = laborAccountingCycleCachingService;
    }

    protected String getDebitCreditCode(final Transaction transaction) {
        final KualiDecimal transactionAmount = transaction.getTransactionLedgerEntryAmount();
        return DebitCreditUtil.getDebitCreditCode(transactionAmount, false);
    }

    protected KualiDecimal getTransactionAmount(final Transaction transaction) {
        final KualiDecimal transactionAmount = transaction.getTransactionLedgerEntryAmount();
        return transactionAmount.abs();
    }

    protected String getEncumbranceUpdateCode(final Transaction transaction) {
        final String encumbranceUpdateCode = transaction.getTransactionEncumbranceUpdateCode();
        if (KFSConstants.ENCUMB_UPDT_DOCUMENT_CD.equals(encumbranceUpdateCode)
                || KFSConstants.ENCUMB_UPDT_REFERENCE_DOCUMENT_CD.equals(encumbranceUpdateCode)) {
            return encumbranceUpdateCode;
        }
        return null;
    }

    @Override
    public String getDestinationName() {
        return LaborConstants.DestinationNames.LABOR_GL_ENTRY;
    }

    public void setLaborGeneralLedgerEntryService(final LaborGeneralLedgerEntryService laborGeneralLedgerEntryService) {
        this.laborGeneralLedgerEntryService = laborGeneralLedgerEntryService;
    }
}
