/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.batch.service.impl;

import org.kuali.kfs.core.api.datetime.DateTimeService;
import org.kuali.kfs.gl.GeneralLedgerConstants;
import org.kuali.kfs.gl.businessobject.OriginEntryFull;
import org.kuali.kfs.gl.report.LedgerSummaryReport;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.module.ld.LaborConstants;
import org.kuali.kfs.module.ld.batch.service.LaborNightlyOutService;
import org.kuali.kfs.module.ld.businessobject.LaborGeneralLedgerEntry;
import org.kuali.kfs.module.ld.businessobject.LaborLedgerPendingEntry;
import org.kuali.kfs.module.ld.businessobject.LaborOriginEntry;
import org.kuali.kfs.module.ld.dataaccess.LaborClearGeneralLedgerEntryDao;
import org.kuali.kfs.module.ld.service.LaborLedgerPendingEntryService;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.ObjectUtil;
import org.kuali.kfs.sys.batch.service.WrappingBatchService;
import org.kuali.kfs.sys.service.ReportWriterService;
import org.springframework.transaction.annotation.Transactional;

import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.sql.Date;
import java.util.Collection;
import java.util.Iterator;

/**
 * The class implements loading and cleanup methods for nightly batch jobs
 */
@Transactional
public class LaborNightlyOutServiceImpl implements LaborNightlyOutService {

    protected LaborLedgerPendingEntryService laborLedgerPendingEntryService;
    protected BusinessObjectService businessObjectService;
    protected DateTimeService dateTimeService;
    protected String batchFileDirectoryName;
    protected String batchGlFileDirectoryName;
    protected LaborClearGeneralLedgerEntryDao laborClearGeneralLedgerEntryDao;
    protected ReportWriterService laborPendingEntryLedgerReportWriterService;
    protected ReportWriterService laborGLEntryReportWriterService;

    @Override
    public void deleteCopiedPendingLedgerEntries() {
        laborLedgerPendingEntryService.deleteByFinancialDocumentApprovedCode(
                KFSConstants.PENDING_ENTRY_APPROVED_STATUS_CODE.PROCESSED);
    }

    @Override
    public void copyApprovedPendingLedgerEntries() {
        final Date runDate = dateTimeService.getCurrentSqlDate();

        final String outputFile = batchFileDirectoryName + File.separator + LaborConstants.BatchFileSystem.NIGHTLY_OUT_FILE +
                                  GeneralLedgerConstants.BatchFileSystem.EXTENSION;
        final PrintStream outputFilePs;

        try {
            outputFilePs = new PrintStream(outputFile);
        } catch (final IOException e) {
            // FIXME: do whatever is supposed to be done here
            throw new RuntimeException(e);
        }

        //TODO:- might need to change this part to use file not collection
        final Iterator<LaborLedgerPendingEntry> pendingEntries =
                laborLedgerPendingEntryService.findApprovedPendingLedgerEntries();

        final LedgerSummaryReport nightlyOutLedgerSummaryReport = new LedgerSummaryReport();
        while (pendingEntries != null && pendingEntries.hasNext()) {
            final LaborLedgerPendingEntry pendingEntry = pendingEntries.next();
            final LaborOriginEntry entry = new LaborOriginEntry(pendingEntry);

            try {
                outputFilePs.printf("%s\n", entry.getLine());
                nightlyOutLedgerSummaryReport.summarizeEntry(entry);
            } catch (final Exception e) {
                throw new RuntimeException(e);
            }

            // update the pending entry to indicate it has been copied
            pendingEntry.setFinancialDocumentApprovedCode(KFSConstants.PENDING_ENTRY_APPROVED_STATUS_CODE.PROCESSED);
            pendingEntry.setTransactionDate(runDate);
            businessObjectService.save(pendingEntry);
        }

        outputFilePs.close();

        //create done file
        final String doneFileName = outputFile.replace(GeneralLedgerConstants.BatchFileSystem.EXTENSION,
                GeneralLedgerConstants.BatchFileSystem.DONE_FILE_EXTENSION);
        final File doneFile = new File(doneFileName);
        if (!doneFile.exists()) {
            try {
                doneFile.createNewFile();
            } catch (final IOException e) {
                throw new RuntimeException();
            }
        }

        try {
            ((WrappingBatchService) laborPendingEntryLedgerReportWriterService).initialize();
            nightlyOutLedgerSummaryReport.writeReport(laborPendingEntryLedgerReportWriterService);
        } finally {
            ((WrappingBatchService) laborPendingEntryLedgerReportWriterService).destroy();
        }
    }

    @Override
    public void deleteCopiedLaborGenerealLedgerEntries() {
        laborClearGeneralLedgerEntryDao.deleteCopiedLaborGenerealLedgerEntries();
    }

    @Override
    public void copyLaborGenerealLedgerEntries() {
        final String outputFile = batchGlFileDirectoryName + File.separator +
                                  LaborConstants.BatchFileSystem.LABOR_GL_ENTRY_FILE + GeneralLedgerConstants.BatchFileSystem.EXTENSION;
        final PrintStream outputFilePs;

        try {
            outputFilePs = new PrintStream(outputFile);
        } catch (final IOException e) {
            // FIXME: do whatever is supposed to be done here
            throw new RuntimeException(e);
        }

        // copy the labor general ledger entry to origin entry table
        final Collection<LaborGeneralLedgerEntry> generalLedgerEntries =
                businessObjectService.findAll(LaborGeneralLedgerEntry.class);

        final LedgerSummaryReport laborGLSummaryReport = new LedgerSummaryReport();
        for (final LaborGeneralLedgerEntry entry : generalLedgerEntries) {
            final OriginEntryFull originEntry = new OriginEntryFull();
            ObjectUtil.buildObject(originEntry, entry);

            //write to file
            try {
                outputFilePs.printf("%s\n", originEntry.getLine());
                laborGLSummaryReport.summarizeEntry(originEntry);
            } catch (final Exception e) {
                throw new RuntimeException(e);
            }
        }

        outputFilePs.close();

        //create done file
        final String doneFileName = outputFile.replace(GeneralLedgerConstants.BatchFileSystem.EXTENSION,
                GeneralLedgerConstants.BatchFileSystem.DONE_FILE_EXTENSION);
        final File doneFile = new File(doneFileName);
        if (!doneFile.exists()) {
            try {
                doneFile.createNewFile();
            } catch (final IOException e) {
                throw new RuntimeException();
            }
        }

        try {
            ((WrappingBatchService) laborGLEntryReportWriterService).initialize();
            laborGLSummaryReport.writeReport(laborGLEntryReportWriterService);
        } finally {
            ((WrappingBatchService) laborGLEntryReportWriterService).destroy();
        }
    }

    public void setBusinessObjectService(final BusinessObjectService businessObjectService) {
        this.businessObjectService = businessObjectService;
    }

    public void setDateTimeService(final DateTimeService dateTimeService) {
        this.dateTimeService = dateTimeService;
    }

    public void setLaborLedgerPendingEntryService(final LaborLedgerPendingEntryService laborLedgerPendingEntryService) {
        this.laborLedgerPendingEntryService = laborLedgerPendingEntryService;
    }

    public void setBatchFileDirectoryName(final String batchFileDirectoryName) {
        this.batchFileDirectoryName = batchFileDirectoryName;
    }

    public void setBatchGlFileDirectoryName(final String batchGlFileDirectoryName) {
        this.batchGlFileDirectoryName = batchGlFileDirectoryName;
    }

    public void setLaborPendingEntryLedgerReportWriterService(
            final ReportWriterService laborPendingEntryLedgerReportWriterService) {
        this.laborPendingEntryLedgerReportWriterService = laborPendingEntryLedgerReportWriterService;
    }

    public void setLaborGLEntryReportWriterService(final ReportWriterService laborGLEntryReportWriterService) {
        this.laborGLEntryReportWriterService = laborGLEntryReportWriterService;
    }

    public void setLaborClearGeneralLedgerEntryDao(final LaborClearGeneralLedgerEntryDao laborClearGeneralLedgerEntryDao) {
        this.laborClearGeneralLedgerEntryDao = laborClearGeneralLedgerEntryDao;
    }
}
